/*
 * Copyright (C) 2011, Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef DefaultAudioDestinationNode_h
#define DefaultAudioDestinationNode_h

#include <memory>
#include "modules/webaudio/AudioDestinationNode.h"
#include "platform/audio/AudioDestination.h"
#include "public/platform/WebAudioLatencyHint.h"

namespace blink {

class BaseAudioContext;
class ExceptionState;
class WebAudioLatencyHint;

class DefaultAudioDestinationHandler final : public AudioDestinationHandler {
 public:
  static PassRefPtr<DefaultAudioDestinationHandler> Create(
      AudioNode&,
      const WebAudioLatencyHint&);
  ~DefaultAudioDestinationHandler() override;

  // AudioHandler
  void Dispose() override;
  void Initialize() override;
  void Uninitialize() override;
  void SetChannelCount(unsigned long, ExceptionState&) override;

  // AudioDestinationHandler
  void StartRendering() override;
  void StopRendering() override;
  unsigned long MaxChannelCount() const override;
  // Returns the rendering callback buffer size.
  size_t CallbackBufferSize() const override;
  double SampleRate() const override;
  int FramesPerBuffer() const override;

  // AudioNode
  double TailTime() const override { return 0; }
  double LatencyTime() const override { return 0; }

 private:
  explicit DefaultAudioDestinationHandler(AudioNode&,
                                          const WebAudioLatencyHint&);
  void CreateDestination();

  std::unique_ptr<AudioDestination> destination_;
  String input_device_id_;
  unsigned number_of_input_channels_;
  const WebAudioLatencyHint latency_hint_;
};

class DefaultAudioDestinationNode final : public AudioDestinationNode {
 public:
  static DefaultAudioDestinationNode* Create(BaseAudioContext*,
                                             const WebAudioLatencyHint&);

  size_t CallbackBufferSize() const { return Handler().CallbackBufferSize(); };

 private:
  explicit DefaultAudioDestinationNode(BaseAudioContext&,
                                       const WebAudioLatencyHint&);
};

}  // namespace blink

#endif  // DefaultAudioDestinationNode_h
