/*
 * Copyright (C) 1999 Lars Knoll (knoll@kde.org)
 * Copyright (C) 2004, 2005, 2006, 2007, 2009 Apple Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef LayoutReplaced_h
#define LayoutReplaced_h

#include "core/CoreExport.h"
#include "core/layout/LayoutBox.h"

namespace blink {

// LayoutReplaced is the base class for a replaced element as defined by CSS:
//
// "An element whose content is outside the scope of the CSS formatting model,
// such as an image, embedded document, or applet."
// http://www.w3.org/TR/CSS2/conform.html#defs
//
// Blink consider that replaced elements have an intrinsic sizes (e.g. the
// natural size of an image or a video). The intrinsic size is stored by
// m_intrinsicSize.
//
// The computation sometimes ask for the intrinsic ratio, defined as follow:
//
//                      intrinsicWidth
//   intrinsicRatio = -------------------
//                      intrinsicHeight
//
// The intrinsic ratio is used to keep the same proportion as the intrinsic
// size (thus avoiding visual distortions if width / height doesn't match
// the intrinsic value).
class CORE_EXPORT LayoutReplaced : public LayoutBox {
 public:
  LayoutReplaced(Element*);
  LayoutReplaced(Element*, const LayoutSize& intrinsic_size);
  ~LayoutReplaced() override;

  LayoutUnit ComputeReplacedLogicalWidth(
      ShouldComputePreferred = kComputeActual) const override;
  LayoutUnit ComputeReplacedLogicalHeight(
      LayoutUnit estimated_used_width = LayoutUnit()) const override;

  bool HasReplacedLogicalHeight() const;
  // This function returns the local rect of the replaced content.
  virtual LayoutRect ReplacedContentRect() const;

  bool NeedsPreferredWidthsRecalculation() const override;

  // These values are specified to be 300 and 150 pixels in the CSS 2.1 spec.
  // http://www.w3.org/TR/CSS2/visudet.html#inline-replaced-width
  static const int kDefaultWidth;
  static const int kDefaultHeight;
  bool CanHaveChildren() const override { return false; }
  virtual void PaintReplaced(const PaintInfo&, const LayoutPoint&) const {}
  LayoutRect LocalSelectionRect() const final;

  bool HasObjectFit() const {
    return Style()->GetObjectFit() != ComputedStyle::InitialObjectFit();
  }

  void Paint(const PaintInfo&, const LayoutPoint&) const override;

  // Replaced objects often have contents to paint.
  bool PaintedOutputOfObjectHasNoEffectRegardlessOfSize() const override {
    return false;
  }

  struct IntrinsicSizingInfo {
    STACK_ALLOCATED();
    IntrinsicSizingInfo() : has_width(true), has_height(true) {}

    FloatSize size;
    FloatSize aspect_ratio;
    bool has_width;
    bool has_height;

    void Transpose();
  };

 protected:
  void WillBeDestroyed() override;

  void UpdateLayout() override;

  LayoutSize IntrinsicSize() const final { return intrinsic_size_; }
  virtual void ComputeIntrinsicSizingInfo(IntrinsicSizingInfo&) const;

  void ComputePositionedLogicalWidth(
      LogicalExtentComputedValues&) const override;
  void ComputePositionedLogicalHeight(
      LogicalExtentComputedValues&) const override;

  void ComputeIntrinsicLogicalWidths(LayoutUnit& min_logical_width,
                                     LayoutUnit& max_logical_width) const final;

  // This function calculates the placement of the replaced contents. It takes
  // intrinsic size of the replaced contents, stretch to fit CSS content box
  // according to object-fit.
  LayoutRect ComputeObjectFit(
      const LayoutSize* overridden_intrinsic_size = nullptr) const;

  virtual LayoutUnit IntrinsicContentLogicalHeight() const {
    return IntrinsicLogicalHeight();
  }

  virtual LayoutUnit MinimumReplacedHeight() const { return LayoutUnit(); }

  void SetSelectionState(SelectionState) final;

  void StyleDidChange(StyleDifference, const ComputedStyle* old_style) override;

  void SetIntrinsicSize(const LayoutSize& intrinsic_size) {
    intrinsic_size_ = intrinsic_size;
  }

  // This callback is invoked whenever the intrinsic size changed.
  //
  // The intrinsic size can change due to the network (from the default
  // intrinsic size [see above] to the actual intrinsic size) or to some
  // CSS properties like 'zoom' or 'image-orientation'.
  virtual void IntrinsicSizeChanged();

  virtual LayoutReplaced* EmbeddedReplacedContent() const { return nullptr; }

  PositionWithAffinity PositionForPoint(const LayoutPoint&) override;

  bool IsOfType(LayoutObjectType type) const override {
    return type == kLayoutObjectLayoutReplaced || LayoutBox::IsOfType(type);
  }

 private:
  void ComputePreferredLogicalWidths() final;

  bool CanBeSelectionLeaf() const override { return true; }

  void ComputeIntrinsicSizingInfoForReplacedContent(LayoutReplaced*,
                                                    IntrinsicSizingInfo&) const;
  FloatSize ConstrainIntrinsicSizeToMinMax(const IntrinsicSizingInfo&) const;

  LayoutUnit ComputeConstrainedLogicalWidth(ShouldComputePreferred) const;

  mutable LayoutSize intrinsic_size_;
};

}  // namespace blink

#endif
