static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright (C) 2014 Simon Budig <simon@gimp.org>                            \n"
" *                                                                            \n"
" * implemented following this paper:                                          \n"
" *   A. Meijster, J.B.T.M. Roerdink, and W.H. Hesselink                       \n"
" *   \"A General Algorithm for Computing Distance Transforms in Linear Time\",\n"
" *   Mathematical Morphology and its Applications to Image and                \n"
" *   Signal Processing, Kluwer Acad. Publ., 2000, pp. 331-340.                \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_dt_metric)                                                   \n"
"  enum_value (GEGL_DT_METRIC_EUCLIDEAN,  \"euclidean\",  N_(\"Euclidean\"))   \n"
"  enum_value (GEGL_DT_METRIC_MANHATTAN,  \"manhattan\",  N_(\"Manhattan\"))   \n"
"  enum_value (GEGL_DT_METRIC_CHESSBOARD, \"chessboard\", N_(\"Chessboard\"))  \n"
"enum_end (GeglDTMetric)                                                       \n"
"                                                                              \n"
"property_enum (metric, _(\"Metric\"),                                         \n"
"    GeglDTMetric, gegl_dt_metric, GEGL_DT_METRIC_EUCLIDEAN)                   \n"
"    description (_(\"Metric to use for the distance calculation\"))           \n"
"                                                                              \n"
"property_double (threshold_lo, _(\"Threshold low\"), 0.0001)                  \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"property_double (threshold_hi, _(\"Threshold high\"), 1.0)                    \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"property_int    (averaging, _(\"Grayscale Averaging\"), 0)                    \n"
"    description (_(\"Number of computations for grayscale averaging\"))       \n"
"    value_range (0, 1000)                                                     \n"
"    ui_range    (0, 256)                                                      \n"
"    ui_gamma    (1.5)                                                         \n"
"                                                                              \n"
"property_boolean (normalize, _(\"Normalize\"), TRUE)                          \n"
"  description(_(\"Normalize output to range 0.0 to 1.0.\"))                   \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME     distance_transform                                   \n"
"#define GEGL_OP_C_SOURCE distance-transform.c                                 \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"#define EPSILON 0.000000000001                                                \n"
"                                                                              \n"
"gfloat edt_f   (gfloat x, gfloat i, gfloat g_i);                              \n"
"gint   edt_sep (gint i, gint u, gfloat g_i, gfloat g_u);                      \n"
"gfloat mdt_f   (gfloat x, gfloat i, gfloat g_i);                              \n"
"gint   mdt_sep (gint i, gint u, gfloat g_i, gfloat g_u);                      \n"
"gfloat cdt_f   (gfloat x, gfloat i, gfloat g_i);                              \n"
"gint   cdt_sep (gint i, gint u, gfloat g_i, gfloat g_u);                      \n"
"                                                                              \n"
"/**                                                                           \n"
" * Prepare function of gegl filter.                                           \n"
" * @param operation given Gegl operation                                      \n"
" */                                                                           \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  const Babl *format = babl_format (\"Y float\");                             \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\", format);                   \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"/**                                                                           \n"
" * Returns the cached region. This is an area filter, which acts on the whole image.\n"
" * @param operation given Gegl operation                                      \n"
" * @param roi the rectangle of interest                                       \n"
" * @return result the new rectangle                                           \n"
" */                                                                           \n"
"static GeglRectangle                                                          \n"
"get_cached_region (GeglOperation       *operation,                            \n"
"                   const GeglRectangle *roi)                                  \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Meijster helper functions for euclidean distance transform */              \n"
"gfloat                                                                        \n"
"edt_f (gfloat x, gfloat i, gfloat g_i)                                        \n"
"{                                                                             \n"
"  return sqrt ((x - i) * (x - i) + g_i * g_i);                                \n"
"}                                                                             \n"
"                                                                              \n"
"gint                                                                          \n"
"edt_sep (gint i, gint u, gfloat g_i, gfloat g_u)                              \n"
"{                                                                             \n"
"  return (u * u - i * i + ((gint) (g_u * g_u - g_i * g_i))) / (2 * (u - i));  \n"
"}                                                                             \n"
"                                                                              \n"
"/* Meijster helper functions for manhattan distance transform */              \n"
"gfloat                                                                        \n"
"mdt_f (gfloat x, gfloat i, gfloat g_i)                                        \n"
"{                                                                             \n"
"  return fabs (x - i) + g_i;                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"gint                                                                          \n"
"mdt_sep (gint i, gint u, gfloat g_i, gfloat g_u)                              \n"
"{                                                                             \n"
"  if (g_u >= g_i + u - i + EPSILON)                                           \n"
"    return INT32_MAX / 4;                                                     \n"
"  if (g_i > g_u + u - i + EPSILON)                                            \n"
"    return INT32_MIN / 4;                                                     \n"
"                                                                              \n"
"  return (((gint) (g_u - g_i)) + u + i) / 2;                                  \n"
"}                                                                             \n"
"                                                                              \n"
"/* Meijster helper functions for chessboard distance transform */             \n"
"gfloat                                                                        \n"
"cdt_f (gfloat x, gfloat i, gfloat g_i)                                        \n"
"{                                                                             \n"
"  return MAX (fabs (x - i), g_i);                                             \n"
"}                                                                             \n"
"                                                                              \n"
"gint                                                                          \n"
"cdt_sep (gint i, gint u, gfloat g_i, gfloat g_u)                              \n"
"{                                                                             \n"
"  if (g_i <= g_u)                                                             \n"
"    return MAX (i + (gint) g_u, (i+u)/2);                                     \n"
"  else                                                                        \n"
"    return MIN (u - (gint) g_i, (i+u)/2);                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"binary_dt_2nd_pass (GeglOperation *operation,                                 \n"
"                    gint           width,                                     \n"
"                    gint           height,                                    \n"
"                    gfloat         thres_lo,                                  \n"
"                    GeglDTMetric   metric,                                    \n"
"                    gfloat        *src,                                       \n"
"                    gfloat        *dest)                                      \n"
"{                                                                             \n"
"  gint u, y;                                                                  \n"
"  gint q, w, *t, *s;                                                          \n"
"  gfloat *g, *row_copy;                                                       \n"
"                                                                              \n"
"  gfloat (*dt_f)   (gfloat, gfloat, gfloat);                                  \n"
"  gint   (*dt_sep) (gint, gint, gfloat, gfloat);                              \n"
"                                                                              \n"
"  switch (metric)                                                             \n"
"    {                                                                         \n"
"      case GEGL_DT_METRIC_CHESSBOARD:                                         \n"
"        dt_f   = cdt_f;                                                       \n"
"        dt_sep = cdt_sep;                                                     \n"
"        break;                                                                \n"
"      case GEGL_DT_METRIC_MANHATTAN:                                          \n"
"        dt_f   = mdt_f;                                                       \n"
"        dt_sep = mdt_sep;                                                     \n"
"        break;                                                                \n"
"      default: /* GEGL_DT_METRIC_EUCLIDEAN */                                 \n"
"        dt_f   = edt_f;                                                       \n"
"        dt_sep = edt_sep;                                                     \n"
"        break;                                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* sorry for the variable naming, they are taken from the paper */          \n"
"                                                                              \n"
"  s = gegl_calloc (sizeof (gint), width);                                     \n"
"  t = gegl_calloc (sizeof (gint), width);                                     \n"
"  row_copy = gegl_calloc (sizeof (gfloat), width);                            \n"
"                                                                              \n"
"  /* this could be parallelized */                                            \n"
"  for (y = 0; y < height; y++)                                                \n"
"    {                                                                         \n"
"      q = 0;                                                                  \n"
"      s[0] = 0;                                                               \n"
"      t[0] = 0;                                                               \n"
"      g = dest + y * width;                                                   \n"
"                                                                              \n"
"      dest[0 + y * width] = MIN (dest[0 + y * width], 1.0);                   \n"
"      dest[width - 1 + y * width] = MIN (dest[width - 1 + y * width], 1.0);   \n"
"                                                                              \n"
"      for (u = 1; u < width; u++)                                             \n"
"        {                                                                     \n"
"          while (q >= 0 &&                                                    \n"
"                 dt_f (t[q], s[q], g[s[q]]) >= dt_f (t[q], u, g[u]) + EPSILON)\n"
"            {                                                                 \n"
"              q --;                                                           \n"
"            }                                                                 \n"
"                                                                              \n"
"          if (q < 0)                                                          \n"
"            {                                                                 \n"
"              q = 0;                                                          \n"
"              s[0] = u;                                                       \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              /* function Sep from paper */                                   \n"
"              w = dt_sep (s[q], u, g[s[q]], g[u]);                            \n"
"              w += 1;                                                         \n"
"                                                                              \n"
"              if (w < width)                                                  \n"
"                {                                                             \n"
"                  q ++;                                                       \n"
"                  s[q] = u;                                                   \n"
"                  t[q] = w;                                                   \n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      memcpy (row_copy, g, width * sizeof (gfloat));                          \n"
"                                                                              \n"
"      for (u = width - 1; u >= 0; u--)                                        \n"
"        {                                                                     \n"
"          if (u == s[q])                                                      \n"
"            g[u] = row_copy[u];                                               \n"
"          else                                                                \n"
"            g[u] = dt_f (u, s[q], row_copy[s[q]]);                            \n"
"                                                                              \n"
"          if (q > 0 && u == t[q])                                             \n"
"            {                                                                 \n"
"              q--;                                                            \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      gegl_operation_progress (operation,                                     \n"
"                               (gdouble) y / (gdouble) height / 2.0 + 0.5, \"\");\n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (t);                                                              \n"
"  gegl_free (s);                                                              \n"
"  gegl_free (row_copy);                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"binary_dt_1st_pass (GeglOperation *operation,                                 \n"
"                    gint           width,                                     \n"
"                    gint           height,                                    \n"
"                    gfloat         thres_lo,                                  \n"
"                    gfloat        *src,                                       \n"
"                    gfloat        *dest)                                      \n"
"{                                                                             \n"
"  int x, y;                                                                   \n"
"                                                                              \n"
"  /* this loop could be parallelized */                                       \n"
"  for (x = 0; x < width; x++)                                                 \n"
"    {                                                                         \n"
"      /* consider out-of-range as 0, i.e. the outside is \"empty\" */         \n"
"      dest[x + 0 * width] = src[x + 0 * width] > thres_lo ? 1.0 : 0.0;        \n"
"                                                                              \n"
"      for (y = 1; y < height; y++)                                            \n"
"        {                                                                     \n"
"          if (src[x + y * width] > thres_lo)                                  \n"
"            dest[x + y * width] = 1.0 + dest[x + (y - 1) * width];            \n"
"          else                                                                \n"
"            dest[x + y * width] = 0.0;                                        \n"
"        }                                                                     \n"
"                                                                              \n"
"      dest[x + (height - 1) * width] = MIN (dest[x + (height - 1) * width], 1.0);\n"
"      for (y = height - 2; y >= 0; y--)                                       \n"
"        {                                                                     \n"
"          if (dest[x + (y + 1) * width] + 1.0 < dest[x + y * width])          \n"
"            dest [x + y * width] = dest[x + (y + 1) * width] + 1.0;           \n"
"        }                                                                     \n"
"                                                                              \n"
"      gegl_operation_progress (operation,                                     \n"
"                               (gdouble) x / (gdouble) width / 2.0, \"\");    \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**                                                                           \n"
" * Process the gegl filter                                                    \n"
" * @param operation the given Gegl operation                                  \n"
" * @param input the input buffer.                                             \n"
" * @param output the output buffer.                                           \n"
" * @param result the region of interest.                                      \n"
" * @param level the level of detail                                           \n"
" * @return True, if the filter was successfull applied.                       \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties         *o = GEGL_PROPERTIES (operation);                    \n"
"  const Babl  *input_format = babl_format (\"Y float\");                      \n"
"  const int bytes_per_pixel = babl_format_get_bytes_per_pixel (input_format); \n"
"                                                                              \n"
"  GeglDTMetric metric;                                                        \n"
"  gint     width, height, averaging, i;                                       \n"
"  gfloat   threshold_lo, threshold_hi, maxval, *src_buf, *dst_buf;            \n"
"  gboolean normalize;                                                         \n"
"                                                                              \n"
"  width  = result->width;                                                     \n"
"  height = result->height;                                                    \n"
"                                                                              \n"
"  threshold_lo = o->threshold_lo;                                             \n"
"  threshold_hi = o->threshold_hi;                                             \n"
"  normalize    = o->normalize;                                                \n"
"  metric       = o->metric;                                                   \n"
"  averaging    = o->averaging;                                                \n"
"                                                                              \n"
"  src_buf = gegl_malloc (width * height * bytes_per_pixel);                   \n"
"  dst_buf = gegl_calloc (width * height, bytes_per_pixel);                    \n"
"                                                                              \n"
"  gegl_operation_progress (operation, 0.0, \"\");                             \n"
"                                                                              \n"
"  gegl_buffer_get (input, result, 1.0, input_format, src_buf,                 \n"
"                   GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                     \n"
"                                                                              \n"
"  if (!averaging)                                                             \n"
"    {                                                                         \n"
"      binary_dt_1st_pass (operation, width, height, threshold_lo,             \n"
"                          src_buf, dst_buf);                                  \n"
"      binary_dt_2nd_pass (operation, width, height, threshold_lo, metric,     \n"
"                          src_buf, dst_buf);                                  \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      gfloat *tmp_buf;                                                        \n"
"      gint i, j;                                                              \n"
"                                                                              \n"
"      tmp_buf = gegl_malloc (width * height * bytes_per_pixel);               \n"
"                                                                              \n"
"      for (i = 0; i < averaging; i++)                                         \n"
"        {                                                                     \n"
"          gfloat thres;                                                       \n"
"                                                                              \n"
"          thres = (i+1) * (threshold_hi - threshold_lo) / (averaging + 1);    \n"
"          thres += threshold_lo;                                              \n"
"                                                                              \n"
"          binary_dt_1st_pass (operation, width, height, thres,                \n"
"                              src_buf, tmp_buf);                              \n"
"          binary_dt_2nd_pass (operation, width, height, thres, metric,        \n"
"                              src_buf, tmp_buf);                              \n"
"                                                                              \n"
"          for (j = 0; j < width * height; j++)                                \n"
"            dst_buf[j] += tmp_buf[j];                                         \n"
"        }                                                                     \n"
"                                                                              \n"
"      gegl_free (tmp_buf);                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (normalize)                                                              \n"
"    {                                                                         \n"
"      maxval = EPSILON;                                                       \n"
"                                                                              \n"
"      for (i = 0; i < width * height; i++)                                    \n"
"        maxval = MAX (dst_buf[i], maxval);                                    \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      maxval = averaging;                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (averaging > 0 || normalize)                                             \n"
"    {                                                                         \n"
"      for (i = 0; i < width * height; i++)                                    \n"
"        dst_buf[i] = dst_buf[i] * threshold_hi / maxval;                      \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_buffer_set (output, result, 0, input_format, dst_buf,                  \n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"                                                                              \n"
"  gegl_operation_progress (operation, 1.0, \"\");                             \n"
"                                                                              \n"
"  gegl_free (dst_buf);                                                        \n"
"  gegl_free (src_buf);                                                        \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"  gchar                    *composition =                                     \n"
"    \"<?xml version='1.0' encoding='UTF-8'?>\"                                \n"
"    \"<gegl>\"                                                                \n"
"    \"<node operation='gegl:distance-transform'>\"                            \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='metric'>euclidean</param>\"                            \n"
"    \"    <param name='threshold_lo'>0.0001</param>\"                         \n"
"    \"    <param name='threshold_hi'>1.0</param>\"                            \n"
"    \"    <param name='averaging'>0</param>\"                                 \n"
"    \"    <param name='normalize'>true</param>\"                              \n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"<node operation='gegl:load'>\"                                          \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='path'>standard-input.png</param>\"                     \n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"</gegl>\";                                                              \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  operation_class->threaded          = FALSE;                                 \n"
"  operation_class->prepare           = prepare;                               \n"
"  operation_class->get_cached_region = get_cached_region;                     \n"
"  filter_class->process              = process;                               \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:distance-transform\",                             \n"
"    \"title\",       _(\"Distance Transform\"),                               \n"
"    \"categories\",  \"map\",                                                 \n"
"    \"description\", _(\"Calculate a distance transform\"),                   \n"
"    \"reference-composition\", composition,                                   \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
