//
// "$Id: driver.cpp,v 1.4 2005/06/17 20:54:20 mike Exp $"
//
//   Common functions for the CUPS printer driver for Windows.
//
//   Copyright 2001-2005 by Easy Software Products.
//
//   These coded instructions, statements, and computer programs are the
//   property of Easy Software Products and are protected by Federal
//   copyright law.  Distribution and use rights are outlined in the file
//   "LICENSE.txt" which should have been included with this file.  If this
//   file is missing or damaged please contact Easy Software Products
//   at:
//
//       Attn: CUPS Licensing Information
//       Easy Software Products
//       44141 Airport View Drive, Suite 204
//       Hollywood, Maryland 20636 USA
//
//       Voice: (301) 373-9600
//       EMail: cups-info@cups.org
//         WWW: http://www.cups.org/
//
// Contents:
//
//   cupsDebug()              - Output a debugging message.
//   cupsDevMode()            - Manipulate CUPS device mode data.
//   DllCanUnloadNow()        - Determine if we can unload the DLL now.
//   DllGetClassObject()      - Get a class factory.
//   DllMain()                - Initialize the driver module.
//   cupsCF::cupsCF()         - Create a CUPS class factory.
//   cupsCF::~cupsCF()        - Destroy a CUPS class factory.
//   cupsCF::AddRef()         - Increment the reference count.
//   cupsCF::CreateInstance() - Create an instance of the class.
//   cupsCF::LockServer()     - Lock or unlock the server...
//   cupsCF::QueryInterface() - Get the class associated with this factory.
//   cupsCF::Release()        - Decrement the reference count and destroy as needed.
//

#ifndef CUPS_CLASS_PREFIX
#  error CUPS_CLASS_PREFIX must be defined as L"cupsPS" or L"cupsUI"!
#endif // !CUPS_CLASS_PREFIX

#ifndef CUPS_CLASS_NAME
#  error CUPS_CLASS_NAME must be defined as cupsPS or cupsUI!
#endif // !CUPS_CLASS_NAME

#ifndef CUPS_CLASS_ID
#  error CUPS_CLASS_ID must be defined as CLSID_OEMRENDER or CLSID_OEMUI!
#endif // !CUPS_CLASS_ID


//
// Global data...
//

static long	num_components = 0;
static long	num_locks = 0;


//
// 'cupsDebug()' - Output a debugging message.
//

void
cupsDebug(int     adjust,		// I - Amount to adjust indentation
          LPCWSTR format,		// I - Printf-style format string
          ...)				// I - Additional arguments as needed
{
  int		i;			// Looping var
  va_list	ap;			// Argument pointer
  WCHAR		buffer[2048];		// Output buffer
  static int	indent = 0;		// Indentation...


  wcscpy(buffer, CUPS_CLASS_PREFIX);
  for (i = 0; i <= indent; i ++)
    buffer[i + 8] = ' ';

  va_start(ap, format);
  vswprintf(buffer + indent + 9, format, ap);
  va_end(ap);

  OutputDebugStringW(buffer);

  indent += adjust;
  if (indent < 0)
    indent = 0;
}


//
// 'cupsDevMode()' - Manipulate CUPS device mode data.
//

HRESULT					// O - S_OK on success, E_FAIL on error
cupsDevMode(DWORD       mode,		// I - DevMode operation to perform
            POEMDMPARAM param)		// O - Parameter data
{
  cups_dev_t	*src,			// Source device mode data
		*dst;			// Destination device mode data


  cupsDebug(4, L"cupsDevMode(mode=%d, param=%p)\n", mode, param);

  cupsDebug(0, L"param:\n");
  cupsDebug(0, L"    cbSize = %d\n", param->cbSize);
  cupsDebug(0, L"    pdriverobj = %#x\n", param->pdriverobj);
  cupsDebug(0, L"    hPrinter = %#x\n", param->hPrinter);
  cupsDebug(0, L"    hModule = %#x\n", param->hModule);
  cupsDebug(0, L"    pPublicDMIn = %#x\n", param->pPublicDMIn);
  cupsDebug(0, L"    pPublicDMOut = %#x\n", param->pPublicDMOut);
  cupsDebug(0, L"    pOEMDMIn = %#x\n", param->pOEMDMIn);
  cupsDebug(0, L"    pOEMDMOut = %#x\n", param->pOEMDMOut);
  cupsDebug(0, L"    cbBufSize = %d\n", param->cbBufSize);

  // Range check input...
  if ((mode != OEMDM_SIZE && mode != OEMDM_DEFAULT && mode != OEMDM_CONVERT &&
       mode != OEMDM_MERGE) || !param)
  {
    SetLastError(ERROR_INVALID_PARAMETER);

    cupsDebug(-4, L"returning E_FAIL...\n");

    return (E_FAIL);
  }

  // Get the device mode data pointers...
  src = (cups_dev_t *)param->pOEMDMIn;
  dst = (cups_dev_t *)param->pOEMDMOut;

  switch (mode)
  {
    case OEMDM_CONVERT :
        // Convert the device data to the proper format.
        cupsDebug(0, L"OWMDM_CONVERT\n");

    case OEMDM_MERGE :
        // Convert and validate device data...
        cupsDebug(0, L"OWMDM_MERGE\n");

	if (src->dm_extra.dwSignature == CUPS_SIGNATURE &&
	    src->dm_extra.dwVersion == CUPS_WINVERSION)
	{
	  cupsDebug(0, L"copying CUPS devmode data...");

	  // Already the proper CUPS driver format, so just copy it...
	  memcpy(dst, src, sizeof(cups_dev_t));

          // and make sure that we have valid data...
	  dst->page_label[127]     = 0;
	  dst->job_billing[127]    = 0;
          break;
	}

	// Otherwise initialize to the default...

    case OEMDM_DEFAULT :
        // Initialize device data to the defaults.
        cupsDebug(0, L"OWMDM_DEFAULT\n");

        memset(dst, 0, sizeof(cups_dev_t));

	dst->dm_extra.dwSize = sizeof(cups_dev_t);
	dst->dm_extra.dwSignature = CUPS_SIGNATURE;
	dst->dm_extra.dwVersion   = CUPS_WINVERSION;
        break;

    case OEMDM_SIZE :
        // Get the size of the device mode data...
        cupsDebug(0, L"OWMDM_SIZE\n");

	param->cbBufSize = sizeof(cups_dev_t);
        break;
  }

  if (mode == OEMDM_SIZE)
  {
    cupsDebug(0, L"cbBufSize = %d\n", param->cbBufSize);
  }
  else
  {
    cupsDebug(0, L"dm_extra.dwSize      = %d\n", dst->dm_extra.dwSize);
    cupsDebug(0, L"dm_extra.dwSignature = %08x\n", dst->dm_extra.dwSignature);
    cupsDebug(0, L"dm_extra.dwVersion   = %08x\n", dst->dm_extra.dwVersion);
    cupsDebug(0, L"job_billing          = \"%s\"\n", dst->job_billing);
    cupsDebug(0, L"page_label           = \"%s\"\n", dst->page_label);
  }

  cupsDebug(-4, L"returning S_OK...\n");

  // Return with no error...
  return (S_OK);
}


//
// 'DllCanUnloadNow()' - Determine if we can unload the DLL now.
//

STDAPI					// O - S_OK if we can, S_FALSE if not
DllCanUnloadNow()
{
  cupsDebug(4, L"DllCanUnloadNow()\n");

  if (!num_components && !num_locks)
  {
    cupsDebug(-4, L"returning S_OK...\n");

    return (S_OK);
  }
  else
  {
    cupsDebug(-4, L"returning S_FALSE...\n");

    return (S_FALSE);
  }
}


//
// 'DllGetClassObject()' - Get a class factory.
//

STDAPI					// O - Status
DllGetClassObject(const CLSID &clsid,	// I - Class ID
                  const IID   &iid,	// I - Interface ID
                  void        **data)	// O - Data
{
  HRESULT	result;			// Return status
  cupsCF	*cf;			// Class factory


  cupsDebug(4, L"DllGetClassObject(clsid=%p, iid=%p, data=%p)\n", clsid, iid, data);

  // Range check class ID...
  if (clsid != CUPS_CLASS_ID)
  {
    cupsDebug(-4, L"returning CLASS_E_CLASSNOTAVAILABLE...\n");

    return (CLASS_E_CLASSNOTAVAILABLE);
  }

  // Make a new CUPS class factory...
  if ((cf = new cupsCF()) == NULL)
  {
    cupsDebug(-4, L"returning E_OUTOFMEMORY...\n");

    return (E_OUTOFMEMORY);
  }

  // Get the requested interface...
  result = cf->QueryInterface(iid, data);

  // Release the class factory...
  cf->Release();

  cupsDebug(-4, L"returning %d...\n", result);

  // Return the status of the request...
  return (result);
}


//
// 'DllMain()' - Initialize the driver module.
//

BOOL WINAPI				// O - TRUE if successful, FALSE otherwise
DllMain(HINSTANCE inst,			// I - Instance of driver
        WORD      reason,		// I - Callback reason
	LPVOID    reserved)		// I - Reserved for future use
{
  cupsDebug(0, L"DllMain(inst=%x, reason=%d, reserved=%p)\n", inst, reason,
            reserved);

  // Return TRUE indicating success.
  return (TRUE);
}


//
// 'cupsCF::cupsCF()' - Create a CUPS class factory.
//

cupsCF::cupsCF()
{
  cupsDebug(0, L"cupsCF::cupsCF()\n");

  ref = 1;
}


//
// 'cupsCF::~cupsCF()' - Destroy a CUPS class factory.
//

cupsCF::~cupsCF()
{
  cupsDebug(0, L"cupsCF::~cupsCF()\n");
}


//
// 'cupsCF::AddRef()' - Increment the reference count.
//

ULONG __stdcall				// O - New reference count
cupsCF::AddRef()
{
  ULONG	temp;				// New reference count


  cupsDebug(4, L"cupsCF::AddRef()\n");

  temp = InterlockedIncrement(&ref);

  cupsDebug(-4, L"returning %ld...\n", temp);

  return (temp);
}


//
// 'cupsCF::CreateInstance()' - Create an instance of the class.
//

HRESULT __stdcall			// O - Status
cupsCF::CreateInstance(IUnknown  *outer,// I - Outer factory
                       const IID &iid,	// I - Interface ID
                       void      **data)// O - Data
{
  HRESULT		result;		// Result of query
  CUPS_CLASS_NAME	*temp;		// New class instance


  cupsDebug(4, L"cupsCF::CreateInstance(outer=%p, iid=%p, data=%p)\n",
            outer, iid, data);

  // Range check input...
  if (outer)
  {
    cupsDebug(-4, L"returning CLASS_E_NOAGGREGATION...\n");

    return (CLASS_E_NOAGGREGATION);
  }

  // Create an instance of the class...
  if ((temp = new CUPS_CLASS_NAME()) == NULL)
  {
    cupsDebug(-4, L"returning E_OUTOFMEMORY...\n");

    return (E_OUTOFMEMORY);
  }

  // Get the requested interface...
  result = temp->QueryInterface(iid, data);

  // Release the class and return...
  temp->Release();

  cupsDebug(-4, L"returning %d...\n", result);

  return (result);
}

//
// 'cupsCF::LockServer()' - Lock or unlock the server...
//

HRESULT __stdcall			// O - Status
cupsCF::LockServer(BOOL lock)		// I - TRUE = lock, FALSE = unlock
{
  cupsDebug(4, L"cupsCF::LockServer(lock=%d)\n", lock);

  if (lock)
    InterlockedIncrement(&num_locks);
  else
    InterlockedDecrement(&num_locks);

  cupsDebug(-4, L"returning S_OK, num_locks=%ld...\n", num_locks);

  return (S_OK);
}


//
// 'cupsCF::QueryInterface()' - Get the class associated with this factory.
//

HRESULT __stdcall			// O - Status
cupsCF::QueryInterface(const IID &iid,	// I - Interface ID
                       void      **data)// O - Data
{
  cupsDebug(4, L"cupsCF::QueryInterface(iid=%p, data=%p)\n", iid, data);

  // Get a pointer to the class instance...
  if (iid == IID_IUnknown || iid == IID_IClassFactory)
  {
    *data = static_cast<cupsCF*>(this);
    AddRef();

    cupsDebug(-4, L"returning S_OK, *data=%p...\n", this);

    return (S_OK);
  }
  else
  {
    *data = NULL ;

    cupsDebug(-4, L"returning E_NOINTERFACE...\n");

    return (E_NOINTERFACE);
  }
}


//
// 'cupsCF::Release()' - Decrement the reference count and destroy as needed.
//

ULONG __stdcall				// O - New reference count
cupsCF::Release()
{
  cupsDebug(4, L"cupsCF::Release()\n");

  if (!InterlockedDecrement(&ref))
  {
    // Reference count is now 0, so delete it...
    delete this;
    cupsDebug(-4, L"returning 0...\n");
    return (0);
  }
  else
  {
    cupsDebug(-4, L"returning %ld...\n", ref);
    return (ref);
  }
}


//
// End of "$Id: driver.cpp,v 1.4 2005/06/17 20:54:20 mike Exp $".
//
