/*
     Helper functions for  Chronos Temporal Extensions
     Author:     Scott Bailey
     License:    BSD

    Copyright (c) 2009, Scott Bailey

    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are met:
        * Redistributions of source code must retain the above copyright
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright
          notice, this list of conditions and the following disclaimer in the
          documentation and/or other materials provided with the distribution.
        * Neither the name of the <organization> nor the
          names of its contributors may be used to endorse or promote products
          derived from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY Scott Bailey ''AS IS'' AND ANY
    EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
    DISCLAIMED. IN NO EVENT SHALL <copyright holder> BE LIABLE FOR ANY
    DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
    LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
    ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************
*                             CONFIG
*****************************************************************/

/*
*  These settings controls how much to increment or decrement when
*  adding numeric values to date/time values.
*  Common values:
*  '1us'   or   '1 microsecond' (default for temporal project period)
*  '1ms'        '1 millisecond'
*  '1s'         '1 second' (default for timestamp, period and time)
*  '1m'         '1 minute'
*  '1h'         '1 hour'
*  '1d'         '1 day'    (default for date)
*/

-- sets granularity for timestamp calculation
CREATE OR REPLACE FUNCTION timestamp_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1s';
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION timestamp_granularity() 
IS 'Configuration for units to use in time/timestamp math.';

-- sets granularity for date calculations
CREATE OR REPLACE FUNCTION date_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1 day';
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION date_granularity() 
IS 'Configuration for units to use in date math.';

-- sets granularity for period calculations
-- If you c period type, set this to '1us'
CREATE OR REPLACE FUNCTION period_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1s';
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION period_granularity()
IS 'Configuration for units to use in period math.';

/*****************************************************************
*                      timestamp functions
*****************************************************************/

-- 1 unit(seconds) prior to ts
CREATE OR REPLACE FUNCTION prior(ts1 timestampTz)
RETURNS timestampTz AS
$$
    SELECT $1 - timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION prior(timestampTz) 
IS 'Timestamp immediately prior to ts1 (1 granule)';

-- 1 unit(seconds) prior to ts
CREATE OR REPLACE FUNCTION prior(ts1 timestamp)
RETURNS timestamp AS
$$
    SELECT $1 - timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION prior(timestamp) 
IS 'Timestamp immediately prior to ts1 (1 granule)';

-- 1 unit(seconds) after ts
CREATE OR REPLACE FUNCTION next(ts1 timestampTz)
RETURNS timestampTz AS
$$
    SELECT $1 + timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION next(timestampTz)
IS 'Timestamp immediately after ts1 (1 granule)';

-- 1 unit(seconds) after ts
CREATE OR REPLACE FUNCTION next(timestamp)
RETURNS timestamp AS
$$
    SELECT $1 + timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION next(timestamp)
IS 'Timestamp immediately after ts1 (1 granule)';

-- add n units(seconds) to timestampTz
CREATE OR REPLACE FUNCTION timestamp_add(ts1 timestampTz, n numeric)
RETURNS timestampTz AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION timestamp_add(timestampTz, numeric)
IS 'Add n timestamp granules to ts1';

-- add n units(seconds) to timestamp
CREATE OR REPLACE FUNCTION timestamp_add(ts1 timestamp, n numeric)
RETURNS timestamp AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION timestamp_add(timestamp, numeric)
IS 'Add n timestamp granules to ts1';

-- subtract n units(seconds) from timestamp
CREATE OR REPLACE FUNCTION timestamp_subtract(ts1 timestampTz, n numeric)
RETURNS timestampTz AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION timestamp_subtract(timestampTz, numeric)
IS 'Subtract n timestamp granules from ts1';

-- subtract n units(seconds) to timestamp
CREATE OR REPLACE FUNCTION timestamp_subtract(ts1 timestamp, n numeric)
RETURNS timestamp AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION timestamp_subtract(timestamp, numeric)
IS 'Subtract n timestamp granules from ts1';

/*****************************************************************
*                       date functions
*****************************************************************/

-- add n units(days) to date 
-- default date add only allows interger addition
CREATE OR REPLACE FUNCTION date_add(d1 date, n numeric)
RETURNS timestampTz AS
$$
    SELECT $1::timestampTz + date_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION date_add(date, numeric)
IS 'Add n date granules to d1. The date type only allows interger
addition. This allows numeric addition and returns a timestampTz.';

-- subtract n units(days) from date
CREATE OR REPLACE FUNCTION date_subtract(d1 date, n numeric)
RETURNS timestampTz AS
$$
    SELECT $1::timestampTz - date_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION date_subtract(date, numeric)
IS 'Subtract n date granules from d1. The date type only allows interger
subtraction. This allows numeric subtraction and returns a timestampTz.';

-- Get the last day of the month
CREATE OR REPLACE FUNCTION last_day(d1 date)
RETURNS date AS
$$
    SELECT 
      CASE WHEN mo IN (1,3,5,7,8,10,12)
        THEN mon_start + 30 -- 31 days
      WHEN mo IN (4,6,9,11)
       THEN mon_start + 29 -- 30 days
      WHEN MOD(yr, 4) != 0 
        OR (MOD(yr, 100) = 0 AND MOD(yr, 400) != 0)
        THEN mon_start + 27 -- 28 days
      ELSE mon_start + 28 -- 29 days
    END
    FROM (
      SELECT EXTRACT(MONTH FROM $1)::int AS mo,
        EXTRACT(YEAR FROM $1)::int AS yr,
        DATE_TRUNC('MONTH', $1)::date AS mon_start
    ) sub
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION last_day(date)
IS 'Returns the last day in the month containing d1';

-- Get the week of month
-- a positive direction will index from the first of the month
-- a negative direction will index from the end of the month (and return negative values)
CREATE OR REPLACE FUNCTION week_of_month(d1 date, dir int)
RETURNS int AS
$$
  SELECT CASE WHEN $2 >= 0 THEN
    CEIL(EXTRACT(DAY FROM $1) / 7)::int
  ELSE 
    0 - CEIL(
      (EXTRACT(DAY FROM last_day($1)) - EXTRACT(DAY FROM $1) + 1) / 7
    )::int
  END
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION week_of_month(date, int)
IS 'Get the week of month from either the start or end of the month.';

CREATE OR REPLACE FUNCTION week_of_month(d1 date)
RETURNS int AS
$$
  SELECT week_of_month($1, 1);
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION week_of_month(date)
IS 'Get the week of month from the start of the month.';

CREATE OR REPLACE FUNCTION day_of_cycle(
  dt	       date,
  anchor_date  date,
  cycle_length int
) RETURNS int AS
$$
  SELECT CASE WHEN $1 > $2 THEN
    MOD($1 - $2, $3)
  WHEN $1 < $2 THEN 
    MOD($3 - MOD($1 - $2, $3), $3)
  ELSE 0 END;
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION day_of_cycle(date,date,int)
IS 'Find where a date is in a cycle given the anchor date and length of a cycle.';

/*****************************************************************
*                        time functions
*****************************************************************/

-- add n units to time
CREATE OR REPLACE FUNCTION time_add(t1 time, n numeric)
RETURNS time AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION time_add(time, numeric)
IS 'Add n timestamp granules to t1.';

-- add n units to time
CREATE OR REPLACE FUNCTION time_add(t1 timeTz, n numeric)
RETURNS timeTz AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION time_add(timeTz, numeric)
IS 'Add n timestamp granules to t1.';

-- subtract n units to time
CREATE OR REPLACE FUNCTION time_subtract(t1 time, n numeric)
RETURNS time AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION time_subtract(time, numeric)
IS 'Subtract n timestamp granules from t1';

-- subtract n units from time
CREATE OR REPLACE FUNCTION time_subtract(t1 timeTz, n numeric)
RETURNS timeTz AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION time_subtract(timeTz, numeric)
IS 'Subtract n timestamp granules from t1';

/*****************************************************************
*                        interval functions
*****************************************************************/

-- Number of seconds in interval
CREATE OR REPLACE FUNCTION seconds(i interval)
RETURNS numeric AS
$$
    SELECT EXTRACT(EPOCH FROM $1)::numeric;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION seconds(interval)
IS 'Convert interval to seconds. Opposite of to_interval(numeric).';

CREATE OR REPLACE FUNCTION to_interval(n numeric)
RETURNS interval AS
$$
  SELECT INTERVAL '1s' * $1;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION to_interval(numeric)
IS 'Convert seconds to interval. Opposite of duration(interval)';
/*****************************************************************
*                          operators
*****************************************************************/

CREATE OPERATOR +(
  PROCEDURE = timestamp_add,
  LEFTARG = timestampTz,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = timestamp_subtract,
  LEFTARG = timestampTz,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = timestamp_add,
  LEFTARG = timestamp,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = timestamp_subtract,
  LEFTARG = timestamp,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = date_add,
  LEFTARG = date,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = date_subtract,
  LEFTARG = date,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = time_add,
  LEFTARG = time,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = time_subtract,
  LEFTARG = time,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = time_add,
  LEFTARG = timetz,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = time_subtract,
  LEFTARG = timetz,
  RIGHTARG = numeric
);

