"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSvelteModuleLoader = void 0;
const typescript_1 = __importDefault(require("typescript"));
const fileCollection_1 = require("../../lib/documents/fileCollection");
const utils_1 = require("../../utils");
const svelte_sys_1 = require("./svelte-sys");
const utils_2 = require("./utils");
/**
 * Caches resolved modules.
 */
class ModuleResolutionCache {
    constructor() {
        this.cache = new fileCollection_1.FileMap();
        this.getCanonicalFileName = (0, utils_1.createGetCanonicalFileName)(typescript_1.default.sys.useCaseSensitiveFileNames);
    }
    /**
     * Tries to get a cached module.
     * Careful: `undefined` can mean either there's no match found, or that the result resolved to `undefined`.
     */
    get(moduleName, containingFile) {
        return this.cache.get(this.getKey(moduleName, containingFile));
    }
    /**
     * Checks if has cached module.
     */
    has(moduleName, containingFile) {
        return this.cache.has(this.getKey(moduleName, containingFile));
    }
    /**
     * Caches resolved module (or undefined).
     */
    set(moduleName, containingFile, resolvedModule) {
        this.cache.set(this.getKey(moduleName, containingFile), resolvedModule);
    }
    /**
     * Deletes module from cache. Call this if a file was deleted.
     * @param resolvedModuleName full path of the module
     */
    delete(resolvedModuleName) {
        resolvedModuleName = this.getCanonicalFileName(resolvedModuleName);
        this.cache.forEach((val, key) => {
            if (val && this.getCanonicalFileName(val.resolvedFileName) === resolvedModuleName) {
                this.cache.delete(key);
            }
        });
    }
    /**
     * Deletes everything from cache that resolved to `undefined`
     * and which might match the path.
     */
    deleteUnresolvedResolutionsFromCache(path) {
        const fileNameWithoutEnding = (0, utils_1.getLastPartOfPath)(this.getCanonicalFileName(path)).split('.').shift() || '';
        this.cache.forEach((val, key) => {
            const moduleName = key.split(':::').pop() || '';
            if (!val && moduleName.includes(fileNameWithoutEnding)) {
                this.cache.delete(key);
            }
        });
    }
    getKey(moduleName, containingFile) {
        return containingFile + ':::' + (0, utils_2.ensureRealSvelteFilePath)(moduleName);
    }
}
class ImpliedNodeFormatResolver {
    constructor() {
        this.alreadyResolved = new fileCollection_1.FileMap();
    }
    resolve(importPath, importIdxInFile, sourceFile, compilerOptions) {
        if ((0, utils_2.isSvelteFilePath)(importPath)) {
            // Svelte imports should use the old resolution algorithm, else they are not found
            return undefined;
        }
        let mode = undefined;
        if (sourceFile) {
            if (!sourceFile.impliedNodeFormat && (0, utils_2.isSvelteFilePath)(sourceFile.fileName)) {
                // impliedNodeFormat is not set for Svelte files, because the TS function which
                // calculates this works with a fixed set of file extensions,
                // which .svelte is obv not part of. Make it work by faking a TS file.
                if (!this.alreadyResolved.has(sourceFile.fileName)) {
                    sourceFile.impliedNodeFormat = typescript_1.default.getImpliedNodeFormatForFile((0, utils_2.toVirtualSvelteFilePath)(sourceFile.fileName), undefined, typescript_1.default.sys, compilerOptions);
                    this.alreadyResolved.set(sourceFile.fileName, sourceFile.impliedNodeFormat);
                }
                else {
                    sourceFile.impliedNodeFormat = this.alreadyResolved.get(sourceFile.fileName);
                }
            }
            mode = typescript_1.default.getModeForResolutionAtIndex(sourceFile, importIdxInFile);
        }
        return mode;
    }
}
/**
 * Creates a module loader specifically for `.svelte` files.
 *
 * The typescript language service tries to look up other files that are referenced in the currently open svelte file.
 * For `.ts`/`.js` files this works, for `.svelte` files it does not by default.
 * Reason: The typescript language service does not know about the `.svelte` file ending,
 * so it assumes it's a normal typescript file and searches for files like `../Component.svelte.ts`, which is wrong.
 * In order to fix this, we need to wrap typescript's module resolution and reroute all `.svelte.ts` file lookups to .svelte.
 *
 * @param getSnapshot A function which returns a (in case of svelte file fully preprocessed) typescript/javascript snapshot
 * @param compilerOptions The typescript compiler options
 */
function createSvelteModuleLoader(getSnapshot, compilerOptions, tsSystem, tsResolveModuleName) {
    const svelteSys = (0, svelte_sys_1.createSvelteSys)(getSnapshot, tsSystem);
    const moduleCache = new ModuleResolutionCache();
    const impliedNodeFormatResolver = new ImpliedNodeFormatResolver();
    return {
        fileExists: svelteSys.fileExists,
        readFile: svelteSys.readFile,
        readDirectory: svelteSys.readDirectory,
        deleteFromModuleCache: (path) => {
            svelteSys.deleteFromCache(path);
            moduleCache.delete(path);
        },
        deleteUnresolvedResolutionsFromCache: (path) => {
            svelteSys.deleteFromCache(path);
            moduleCache.deleteUnresolvedResolutionsFromCache(path);
        },
        resolveModuleNames
    };
    function resolveModuleNames(moduleNames, containingFile, _reusedNames, _redirectedReference, _options, containingSourceFile) {
        return moduleNames.map((moduleName, index) => {
            if (moduleCache.has(moduleName, containingFile)) {
                return moduleCache.get(moduleName, containingFile);
            }
            const resolvedModule = resolveModuleName(moduleName, containingFile, containingSourceFile, index);
            moduleCache.set(moduleName, containingFile, resolvedModule);
            return resolvedModule;
        });
    }
    function resolveModuleName(name, containingFile, containingSourceFile, index) {
        const mode = impliedNodeFormatResolver.resolve(name, index, containingSourceFile, compilerOptions);
        // Delegate to the TS resolver first.
        // If that does not bring up anything, try the Svelte Module loader
        // which is able to deal with .svelte files.
        const tsResolvedModule = tsResolveModuleName(name, containingFile, compilerOptions, typescript_1.default.sys, undefined, undefined, mode).resolvedModule;
        if (tsResolvedModule && !(0, utils_2.isVirtualSvelteFilePath)(tsResolvedModule.resolvedFileName)) {
            return tsResolvedModule;
        }
        const svelteResolvedModule = tsResolveModuleName(name, containingFile, compilerOptions, svelteSys, undefined, undefined, mode).resolvedModule;
        if (!svelteResolvedModule ||
            !(0, utils_2.isVirtualSvelteFilePath)(svelteResolvedModule.resolvedFileName)) {
            return svelteResolvedModule;
        }
        const resolvedFileName = (0, utils_2.ensureRealSvelteFilePath)(svelteResolvedModule.resolvedFileName);
        const snapshot = getSnapshot(resolvedFileName);
        const resolvedSvelteModule = {
            extension: (0, utils_2.getExtensionFromScriptKind)(snapshot && snapshot.scriptKind),
            resolvedFileName,
            isExternalLibraryImport: svelteResolvedModule.isExternalLibraryImport
        };
        return resolvedSvelteModule;
    }
}
exports.createSvelteModuleLoader = createSvelteModuleLoader;
//# sourceMappingURL=module-loader.js.map