﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormationRequest.h>
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFormation {
namespace Model {

/**
 * <p>The input for the <a>DeleteChangeSet</a> action.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudformation-2010-05-15/DeleteChangeSetInput">AWS
 * API Reference</a></p>
 */
class DeleteChangeSetRequest : public CloudFormationRequest {
 public:
  AWS_CLOUDFORMATION_API DeleteChangeSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteChangeSet"; }

  AWS_CLOUDFORMATION_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDFORMATION_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name or Amazon Resource Name (ARN) of the change set that you want to
   * delete.</p>
   */
  inline const Aws::String& GetChangeSetName() const { return m_changeSetName; }
  inline bool ChangeSetNameHasBeenSet() const { return m_changeSetNameHasBeenSet; }
  template <typename ChangeSetNameT = Aws::String>
  void SetChangeSetName(ChangeSetNameT&& value) {
    m_changeSetNameHasBeenSet = true;
    m_changeSetName = std::forward<ChangeSetNameT>(value);
  }
  template <typename ChangeSetNameT = Aws::String>
  DeleteChangeSetRequest& WithChangeSetName(ChangeSetNameT&& value) {
    SetChangeSetName(std::forward<ChangeSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If you specified the name of a change set to delete, specify the stack name
   * or Amazon Resource Name (ARN) that's associated with it.</p>
   */
  inline const Aws::String& GetStackName() const { return m_stackName; }
  inline bool StackNameHasBeenSet() const { return m_stackNameHasBeenSet; }
  template <typename StackNameT = Aws::String>
  void SetStackName(StackNameT&& value) {
    m_stackNameHasBeenSet = true;
    m_stackName = std::forward<StackNameT>(value);
  }
  template <typename StackNameT = Aws::String>
  DeleteChangeSetRequest& WithStackName(StackNameT&& value) {
    SetStackName(std::forward<StackNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_changeSetName;

  Aws::String m_stackName;
  bool m_changeSetNameHasBeenSet = false;
  bool m_stackNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
