/*
 * lws-minimal-secure-streams-proxy
 *
 * Written in 2010-2020 by Andy Green <andy@warmcat.com>
 *
 * This file is made available under the Creative Commons CC0 1.0
 * Universal Public Domain Dedication.
 *
 *
 * This is the proxy part for examples built to use it to connect to... it has
 * the policy and the core SS function, but it doesn't contain any of the user
 * code "business logic"... that's in the clients.
 *
 * The proxy side has the policy and performs the onward connection proxying
 * fulfilment.  The clients state the streamtype name they want and ask for the
 * client to do the connection part.
 *
 * Rideshare information is being parsed out at the proxy side; the SSS RX part
 * also brings with it rideshare names.
 *
 * Metadata is passed back over SSS from the client in the TX messages for the
 * proxy to use per the policy.
 */

#include <libwebsockets.h>
#include <string.h>
#include <signal.h>

#if defined(__APPLE__) || defined(__linux__)
#include <execinfo.h>
#include <assert.h>
#endif

#define TEST_CLIENT_ID "SN12345678"

static int interrupted, bad = 1, port = 0 /* unix domain socket */;
static const char *ibind = NULL; /* default to unix domain skt "proxy.ss.lws" */
static lws_state_notify_link_t nl;
static struct lws_context *context;

/*
 * We just define enough policy so it can fetch the latest one securely.
 *
 * If using SSPC, we use the proxy's policy, not this
 */


static const char * const default_ss_policy =
	"{"
	  "\"release\":"			"\"01234567\","
	  "\"product\":"			"\"myproduct\","
	  "\"schema-version\":"			"1,"
	  "\"retry\": ["	/* named backoff / retry strategies */
		"{\"default\": {"
			"\"backoff\": ["	 "1000,"
						 "2000,"
						 "3000,"
						 "5000,"
						"10000"
				"],"
			"\"conceal\":"		"5,"
			"\"jitterpc\":"		"20,"
			"\"svalidping\":"	"30,"
			"\"svalidhup\":"	"35"
		"}}"
	  "],"
	  "\"certs\": [" /* named individual certificates in BASE64 DER */
		/*
		 * Let's Encrypt certs for warmcat.com / libwebsockets.org
		 *
		 * We fetch the real policy from there using SS and switch to
		 * using that.
		 */
		"{\"isrg_root_x1\": \""
	"MIIFazCCA1OgAwIBAgIRAIIQz7DSQONZRGPgu2OCiwAwDQYJKoZIhvcNAQELBQAw"
	"TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh"
	"cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwHhcNMTUwNjA0MTEwNDM4"
	"WhcNMzUwNjA0MTEwNDM4WjBPMQswCQYDVQQGEwJVUzEpMCcGA1UEChMgSW50ZXJu"
	"ZXQgU2VjdXJpdHkgUmVzZWFyY2ggR3JvdXAxFTATBgNVBAMTDElTUkcgUm9vdCBY"
	"MTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAK3oJHP0FDfzm54rVygc"
	"h77ct984kIxuPOZXoHj3dcKi/vVqbvYATyjb3miGbESTtrFj/RQSa78f0uoxmyF+"
	"0TM8ukj13Xnfs7j/EvEhmkvBioZxaUpmZmyPfjxwv60pIgbz5MDmgK7iS4+3mX6U"
	"A5/TR5d8mUgjU+g4rk8Kb4Mu0UlXjIB0ttov0DiNewNwIRt18jA8+o+u3dpjq+sW"
	"T8KOEUt+zwvo/7V3LvSye0rgTBIlDHCNAymg4VMk7BPZ7hm/ELNKjD+Jo2FR3qyH"
	"B5T0Y3HsLuJvW5iB4YlcNHlsdu87kGJ55tukmi8mxdAQ4Q7e2RCOFvu396j3x+UC"
	"B5iPNgiV5+I3lg02dZ77DnKxHZu8A/lJBdiB3QW0KtZB6awBdpUKD9jf1b0SHzUv"
	"KBds0pjBqAlkd25HN7rOrFleaJ1/ctaJxQZBKT5ZPt0m9STJEadao0xAH0ahmbWn"
	"OlFuhjuefXKnEgV4We0+UXgVCwOPjdAvBbI+e0ocS3MFEvzG6uBQE3xDk3SzynTn"
	"jh8BCNAw1FtxNrQHusEwMFxIt4I7mKZ9YIqioymCzLq9gwQbooMDQaHWBfEbwrbw"
	"qHyGO0aoSCqI3Haadr8faqU9GY/rOPNk3sgrDQoo//fb4hVC1CLQJ13hef4Y53CI"
	"rU7m2Ys6xt0nUW7/vGT1M0NPAgMBAAGjQjBAMA4GA1UdDwEB/wQEAwIBBjAPBgNV"
	"HRMBAf8EBTADAQH/MB0GA1UdDgQWBBR5tFnme7bl5AFzgAiIyBpY9umbbjANBgkq"
	"hkiG9w0BAQsFAAOCAgEAVR9YqbyyqFDQDLHYGmkgJykIrGF1XIpu+ILlaS/V9lZL"
	"ubhzEFnTIZd+50xx+7LSYK05qAvqFyFWhfFQDlnrzuBZ6brJFe+GnY+EgPbk6ZGQ"
	"3BebYhtF8GaV0nxvwuo77x/Py9auJ/GpsMiu/X1+mvoiBOv/2X/qkSsisRcOj/KK"
	"NFtY2PwByVS5uCbMiogziUwthDyC3+6WVwW6LLv3xLfHTjuCvjHIInNzktHCgKQ5"
	"ORAzI4JMPJ+GslWYHb4phowim57iaztXOoJwTdwJx4nLCgdNbOhdjsnvzqvHu7Ur"
	"TkXWStAmzOVyyghqpZXjFaH3pO3JLF+l+/+sKAIuvtd7u+Nxe5AW0wdeRlN8NwdC"
	"jNPElpzVmbUq4JUagEiuTDkHzsxHpFKVK7q4+63SM1N95R1NbdWhscdCb+ZAJzVc"
	"oyi3B43njTOQ5yOf+1CceWxG1bQVs5ZufpsMljq4Ui0/1lvh+wjChP4kqKOJ2qxq"
	"4RgqsahDYVvTH9w7jXbyLeiNdd8XM2w9U/t7y0Ff/9yi0GE44Za4rF2LN9d11TPA"
	"mRGunUHBcnWEvgJBQl9nJEiU0Zsnvgc/ubhPgXRR4Xq37Z0j4r7g1SgEEzwxA57d"
	"emyPxgcYxn/eR44/KJ4EBs+lVDR3veyJm+kXQ99b21/+jh5Xos1AnX5iItreGCc="
		"\"}"
	  "],"
	  "\"trust_stores\": [" /* named cert chains */
		"{"
			"\"name\": \"le_via_isrg\","
			"\"stack\": ["
				"\"isrg_root_x1\""
			"]"
		"}"
	  "],"
	  "\"s\": [{"
		"\"captive_portal_detect\": {"
			"\"endpoint\": \"connectivitycheck.android.com\","
			"\"http_url\": \"generate_204\","
			"\"port\": 80,"
			"\"protocol\": \"h1\","
			"\"http_method\": \"GET\","
			"\"opportunistic\": true,"
			"\"http_expect\": 204,"
			"\"http_fail_redirect\": true"
		"},"
		"\"fetch_policy\": {"
			"\"endpoint\":"		"\"warmcat.com\","
			"\"port\":"		"443,"
			"\"protocol\":"		"\"h1\","
			"\"http_method\":"	"\"GET\","
			"\"http_url\":"		"\"policy/minimal-proxy-v4.2-v2.json\","
			"\"tls\":"		"true,"
			"\"opportunistic\":"	"true,"
			"\"retry\":"		"\"default\","
			"\"tls_trust_store\":"	"\"le_via_isrg\""
		"}}"
	"}"
;

static const char *canned_root_token_payload =
	"grant_type=refresh_token"
	"&refresh_token=Atzr|IwEBIJedGXjDqsU_vMxykqOMg"
	"SHfYe3CPcedueWEMWSDMaDnEmiW8RlR1Kns7Cb4B-TOSnqp7ifVsY4BMY2B8tpHfO39XP"
	"zfu9HapGjTR458IyHX44FE71pWJkGZ79uVBpljP4sazJuk8XS3Oe_yLnm_DIO6fU1nU3Y"
	"0flYmsOiOAQE_gRk_pdlmEtHnpMA-9rLw3mkY5L89Ty9kUygBsiFaYatouROhbsTn8-jW"
	"k1zZLUDpT6ICtBXSnrCIg0pUbZevPFhTwdXd6eX-u4rq0W-XaDvPWFO7au-iPb4Zk5eZE"
	"iX6sissYrtNmuEXc2uHu7MnQO1hHCaTdIO2CANVumf-PHSD8xseamyh04sLV5JgFzY45S"
	"KvKMajiUZuLkMokOx86rjC2Hdkx5DO7G-dbG1ufBDG-N79pFMSs7Ck5pc283IdLoJkCQc"
	"AGvTX8o8I29QqkcGou-9TKhOJmpX8As94T61ok0UqqEKPJ7RhfQHHYdCtsdwxgvfVr9qI"
	"xL_hDCcTho8opCVX-6QhJHl6SQFlTw13"
	"&client_id="
		"amzn1.application-oa2-client.4823334c434b4190a2b5a42c07938a2d";

#if defined(LWS_WITH_SECURE_STREAMS_AUTH_SIGV4)
static char *aws_keyid = NULL,
	    *aws_key = NULL;
#endif

static const uint8_t client_key[] = {
	0x30, 0x82, 0x04, 0xa4, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00,
	0xd9, 0x64, 0x2a, 0x43, 0x48, 0x0d, 0xe0, 0x31, 0xa0, 0x2d, 0xc7, 0x4c, 
	0x6d, 0xd0, 0x3a, 0x16, 0xe4, 0x90, 0x13, 0xfa, 0xb2, 0x04, 0x24, 0xf9,
	0xd5, 0x3c, 0xa9, 0x5c, 0x77, 0x1a, 0xb3, 0x92, 0x64, 0xc0, 0x9f, 0x76,
	0xfb, 0x4e, 0x83, 0x8c, 0x50, 0xa2, 0xa3, 0xd4, 0xff, 0xd3, 0x2f, 0x59,
	0xb8, 0xee, 0x7a, 0x8f, 0xcd, 0x41, 0x6c, 0x38, 0xf1, 0x79, 0xc9, 0xe7, 
	0x8b, 0xdd, 0xe6, 0x2e, 0x0c, 0xe4, 0x29, 0x5e, 0x3d, 0xf0, 0x7f, 0x36,
	0x87, 0xeb, 0xce, 0xc5, 0xc4, 0xe1, 0x5c, 0x68, 0xed, 0x2b, 0x4e, 0xb0,
	0xee, 0x96, 0xe8, 0xc1, 0xa9, 0xea, 0x85, 0x03, 0x7b, 0x2c, 0x6b, 0xf0,
	0x7d, 0x98, 0x4b, 0x7a, 0xd2, 0x9d, 0x26, 0x31, 0x74, 0xf6, 0x21, 0x10, 
	0x2f, 0x79, 0x7f, 0x79, 0x5d, 0x82, 0xb8, 0x10, 0x7d, 0x82, 0x52, 0xd0,
	0xc2, 0x91, 0xc9, 0x9c, 0xa6, 0x38, 0x00, 0x6b, 0x56, 0x73, 0x41, 0x01,
	0x6f, 0x67, 0x80, 0x3e, 0xfe, 0x18, 0x91, 0xa0, 0x9f, 0x12, 0x0b, 0x3e,
	0x72, 0xed, 0xe2, 0xbd, 0x5e, 0xa6, 0xd1, 0xd3, 0xbf, 0x1a, 0x76, 0x98, 
	0xd4, 0xab, 0xf7, 0x72, 0x52, 0x05, 0xc5, 0x1f, 0xb1, 0xcd, 0x61, 0xc5,
	0x02, 0x60, 0xc9, 0x6d, 0xc1, 0xcb, 0x80, 0x39, 0x77, 0x0c, 0x99, 0xd6,
	0xdd, 0x83, 0xd2, 0x9e, 0x71, 0xd7, 0x2d, 0x4a, 0xa4, 0x51, 0xd8, 0xb0,
	0x71, 0xaf, 0xf6, 0x92, 0x45, 0x89, 0x4a, 0x5f, 0x3d, 0x66, 0x92, 0xda, 
	0x7c, 0xc4, 0x14, 0x1b, 0x2c, 0x15, 0x61, 0xba, 0x95, 0xd3, 0xed, 0xe7,
	0x7d, 0x75, 0x33, 0x43, 0xec, 0x9a, 0x15, 0x27, 0x12, 0xd2, 0x0b, 0x46,
	0x9f, 0x0a, 0xb7, 0xb8, 0x6d, 0x8a, 0xe6, 0xb4, 0x57, 0x28, 0x12, 0x48,
	0x33, 0x44, 0x45, 0x72, 0xe6, 0x92, 0xac, 0x1e, 0xca, 0xed, 0xb8, 0x0f, 
	0xfb, 0x4f, 0x11, 0x6b, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x82, 0x01,
	0x01, 0x00, 0xa3, 0x9f, 0x17, 0x8c, 0x42, 0x7c, 0x67, 0x25, 0x12, 0xe9,
	0xc1, 0xda, 0xcd, 0xc0, 0x6f, 0x28, 0x71, 0xe9, 0xdb, 0x8f, 0xd2, 0x52,
	0xfa, 0x3d, 0xac, 0x78, 0x97, 0x04, 0x88, 0x84, 0xe9, 0x69, 0xa3, 0x39, 
	0xb4, 0x0c, 0x8a, 0xc4, 0x22, 0x91, 0x06, 0xd4, 0x13, 0x39, 0xab, 0x3f,
	0x99, 0xa1, 0x0f, 0x67, 0x37, 0x8a, 0x6a, 0xb3, 0xf6, 0x2f, 0xb9, 0xd8,
	0xaf, 0xd5, 0xfe, 0x59, 0x0a, 0xf9, 0xd8, 0x2c, 0x16, 0xd2, 0xdf, 0x0d,
	0xd7, 0xc2, 0x04, 0x75, 0xc0, 0xd0, 0xec, 0x09, 0xaf, 0x53, 0x2e, 0x47, 
	0x6d, 0x58, 0x10, 0x8c, 0x37, 0x2e, 0x24, 0xa4, 0xe6, 0x6e, 0xa3, 0x53,
	0xe0, 0xfe, 0x46, 0x99, 0xcd, 0x1d, 0xe9, 0xcd, 0xec, 0x44, 0x47, 0x43,
	0x45, 0xc9, 0x77, 0xe2, 0x1b, 0xc7, 0x0e, 0x51, 0xe8, 0x16, 0x49, 0x51,
	0x40, 0xe1, 0xed, 0xc0, 0x6b, 0x1d, 0xe3, 0xa1, 0xbc, 0xf5, 0xa8, 0xf9, 
	0x2c, 0xa6, 0x89, 0x6d, 0xfe, 0x60, 0x72, 0x1a, 0xa8, 0xc4, 0x5c, 0x7c,
	0x5d, 0x46, 0x46, 0xfc, 0xc2, 0x91, 0x6e, 0x76, 0x90, 0x6c, 0x1d, 0xa8,
	0x39, 0x94, 0xf9, 0x85, 0x46, 0x36, 0xa7, 0x5a, 0xa2, 0x84, 0x35, 0xf8,
	0x1d, 0xb4, 0xa4, 0x3b, 0x83, 0x9a, 0xd2, 0x25, 0xa2, 0xcd, 0xca, 0x93, 
	0xcf, 0x8a, 0xb6, 0x26, 0x25, 0x0e, 0x7c, 0xd9, 0x33, 0x5f, 0xf8, 0x8e,
	0x59, 0x51, 0x88, 0x7b, 0x51, 0x0b, 0xdb, 0x72, 0x4b, 0xfe, 0x74, 0xe5,
	0x77, 0x8e, 0xc9, 0xe3, 0x59, 0x28, 0xee, 0x20, 0x6f, 0xd2, 0x12, 0x5f,
	0x6a, 0x55, 0xc4, 0x47, 0xd5, 0x20, 0xb6, 0x03, 0xb6, 0x97, 0xd2, 0xc1, 
	0xa4, 0x1a, 0x4f, 0xec, 0x99, 0xe3, 0x98, 0xa7, 0x6c, 0xad, 0xd5, 0xb2,
	0x99, 0xa3, 0xf8, 0x1a, 0x38, 0x5b, 0xbc, 0x9c, 0x80, 0x79, 0x1f, 0x48,
	0x86, 0x09, 0xab, 0xfc, 0xd8, 0xd9, 0x02, 0x81, 0x81, 0x00, 0xff, 0xc1,
	0xc0, 0x3c, 0xac, 0x68, 0xab, 0x38, 0xce, 0xc9, 0xbb, 0xa6, 0xb9, 0x6e, 
	0x29, 0x09, 0xbe, 0xf5, 0x3a, 0xf6, 0xc5, 0x22, 0x88, 0xc7, 0x2a, 0xfc,
	0x36, 0x7a, 0x6b, 0xbc, 0x90, 0x10, 0xe5, 0x27, 0x94, 0xc2, 0xa7, 0xf8,
	0x78, 0x4c, 0x50, 0xe5, 0xde, 0x14, 0x58, 0xb0, 0x18, 0x95, 0x92, 0x84,
	0x6e, 0x0c, 0x6d, 0xe6, 0x3a, 0x7e, 0x34, 0x99, 0x27, 0x4b, 0x85, 0x18, 
	0xf3, 0x20, 0x96, 0xa8, 0x8a, 0x80, 0xb1, 0xd6, 0x53, 0x49, 0xfd, 0xc7,
	0x44, 0x51, 0x6c, 0x7c, 0xa9, 0x20, 0x6e, 0xea, 0x4e, 0x5f, 0xce, 0xa4,
	0x66, 0x94, 0xee, 0x75, 0xa9, 0xf7, 0xee, 0xe5, 0x19, 0x49, 0xf7, 0xb3,
	0x15, 0xc3, 0x2a, 0x94, 0x19, 0x44, 0x31, 0xd2, 0x4f, 0xdc, 0x97, 0x50,
	0x66, 0x93, 0xcc, 0x72, 0xc8, 0x73, 0x77, 0xfb, 0x78, 0xf2, 0xcb, 0xc1,
	0x8b, 0xce, 0x36, 0x68, 0x8b, 0x5d, 0x02, 0x81, 0x81, 0x00, 0xd9, 0x99,
	0x13, 0x88, 0xb5, 0x9f, 0xa5, 0x5f, 0xb2, 0x69, 0xdd, 0x83, 0x00, 0x56,
	0x94, 0xfb, 0x91, 0x08, 0x73, 0x4e, 0x2b, 0x3e, 0x22, 0x79, 0x7f, 0x67, 
	0x57, 0xd8, 0x84, 0x80, 0x9c, 0xb7, 0x92, 0xec, 0x73, 0x89, 0x5a, 0x20,
	0x9f, 0x90, 0x61, 0x4e, 0x74, 0x18, 0x53, 0x6d, 0x39, 0x21, 0xb6, 0xad,
	0x29, 0xea, 0x1d, 0xd1, 0xac, 0x62, 0x43, 0xdb, 0x13, 0xe2, 0x3c, 0x4f,
	0x4a, 0xa6, 0x8a, 0x9e, 0x3f, 0x3a, 0x37, 0x55, 0x9c, 0xea, 0x6d, 0x76,
	0xf5, 0x5f, 0x53, 0x0f, 0xb3, 0x08, 0xb1, 0x66, 0xa5, 0x21, 0x3b, 0x60,
	0xf9, 0x4f, 0xc0, 0xbc, 0x12, 0x3e, 0x71, 0xdd, 0x36, 0x63, 0x0c, 0x92,
	0x44, 0x9a, 0x19, 0x1f, 0x52, 0x45, 0x68, 0x57, 0xe3, 0xde, 0x87, 0xd3,
	0x7b, 0x12, 0xa2, 0x9a, 0xbe, 0x47, 0xfc, 0x80, 0x34, 0x05, 0x0e, 0xfc, 
	0xe5, 0x95, 0xec, 0x2a, 0x0b, 0x67, 0x02, 0x81, 0x81, 0x00, 0xd8, 0x22,
	0x8c, 0xaa, 0xb9, 0xa8, 0xda, 0xc9, 0xef, 0x37, 0x5c, 0x75, 0xd1, 0x4e,
	0xdf, 0x2f, 0x75, 0x49, 0x18, 0xc3, 0x62, 0x65, 0x84, 0xe1, 0x8b, 0xda,
	0x08, 0xcf, 0xfc, 0x2c, 0x26, 0x33, 0xa0, 0xf4, 0x8a, 0x2a, 0xd6, 0x78, 
	0x8c, 0x71, 0x32, 0x0c, 0x90, 0x58, 0xf1, 0x7a, 0x1f, 0xcc, 0x87, 0x3d,
	0x93, 0x22, 0xbd, 0xdd, 0x4a, 0xa1, 0x4d, 0x22, 0x00, 0xab, 0x39, 0x97,
	0x94, 0x1f, 0x7e, 0x32, 0x80, 0x99, 0xb1, 0xb0, 0x46, 0x42, 0xc9, 0x4c,
	0x94, 0xd7, 0x9a, 0xae, 0xde, 0xde, 0xa1, 0xdd, 0x30, 0xce, 0x96, 0x7b, 
	0x28, 0x95, 0xc3, 0xcf, 0x7c, 0x0c, 0x22, 0xbf, 0x49, 0x57, 0xd1, 0x14,
	0x6e, 0xd7, 0xb8, 0x2b, 0xe0, 0x0e, 0x59, 0x6c, 0x3a, 0x41, 0x59, 0x3e,
	0x80, 0xf1, 0x76, 0x5d, 0xaa, 0x38, 0xaa, 0xb4, 0xc1, 0xc9, 0xad, 0x97,
	0xaa, 0xe6, 0x61, 0xe4, 0x87, 0xa1, 0x02, 0x81, 0x80, 0x60, 0xbc, 0xfd, 
	0x42, 0x39, 0xfc, 0xcf, 0xb8, 0xcb, 0xc8, 0xe5, 0x01, 0xe8, 0x94, 0x95,
	0x87, 0x81, 0xe3, 0x0f, 0xe0, 0x20, 0xb9, 0x6e, 0xfa, 0x34, 0xf9, 0xec,
	0xc4, 0x05, 0xa4, 0x68, 0x07, 0xf4, 0xf9, 0xca, 0x23, 0xb6, 0x3d, 0xea,
	0xb2, 0x75, 0x90, 0x3b, 0x6c, 0xd5, 0x2a, 0x29, 0xcc, 0x23, 0x3a, 0xb8, 
	0xf7, 0xac, 0x6a, 0x1b, 0x09, 0x54, 0xc1, 0x81, 0x05, 0x83, 0x6a, 0x7c,
	0xf5, 0xad, 0x8a, 0xcc, 0xf1, 0x44, 0xc8, 0x7b, 0x0a, 0x7d, 0xbf, 0xf5,
	0x36, 0x7b, 0xe6, 0xee, 0xe1, 0x5e, 0x7d, 0x13, 0xcc, 0xc2, 0x69, 0x4a,
	0x6d, 0x7a, 0x72, 0x22, 0x6a, 0x40, 0x94, 0xe8, 0x29, 0x00, 0xdd, 0xc6, 
	0x41, 0xfc, 0x09, 0x40, 0xba, 0x3a, 0xb9, 0x58, 0x45, 0x06, 0x7d, 0xae,
	0xa9, 0x8a, 0x40, 0xe1, 0x77, 0x7f, 0x7a, 0x3f, 0xa6, 0x40, 0x13, 0x8e,
	0xbe, 0x28, 0x50, 0x48, 0xe1, 0x02, 0x81, 0x80, 0x29, 0x09, 0x92, 0xd7,
	0x60, 0xb9, 0x05, 0xa9, 0xfb, 0xc7, 0x42, 0x58, 0xa2, 0x50, 0x64, 0x88, 
	0xc8, 0xa5, 0xed, 0x2e, 0x9d, 0xb6, 0x78, 0xed, 0xd1, 0x17, 0x0d, 0xd7,
	0x21, 0x6d, 0x8b, 0xff, 0x04, 0xe7, 0xe4, 0xc4, 0x45, 0x40, 0x73, 0xf8,
	0x9d, 0xa1, 0x24, 0xb0, 0x11, 0x1c, 0x54, 0xce, 0xef, 0x70, 0xe9, 0xd5,
	0x26, 0xc1, 0xd6, 0xce, 0xa6, 0x4b, 0x21, 0xc6, 0x19, 0x00, 0x8e, 0x89, 
	0xe8, 0x8f, 0xb0, 0x11, 0x03, 0x00, 0xda, 0x5f, 0x9a, 0x53, 0x2a, 0x2b,
	0x23, 0x01, 0xa9, 0x25, 0xe4, 0xbb, 0xf8, 0xf4, 0x15, 0x92, 0x95, 0xb1,
	0xb1, 0x13, 0xff, 0xea, 0xb9, 0x2d, 0x86, 0xe3, 0xd8, 0xe0, 0x41, 0xe0,
	0x7b, 0x18, 0x8d, 0x7a, 0x79, 0x10, 0x13, 0x0f, 0xfc, 0xca, 0x08, 0x4d,
	0xa8, 0x9e, 0xcb, 0x66, 0x7c, 0x53, 0xea, 0xae, 0xee, 0xf1, 0x8b, 0x2f,
	0x94, 0x08, 0x72, 0x7c
};

static const uint8_t client_cert_der[] = {
	0x30, 0x82, 0x03, 0x9b, 0x30, 0x82, 0x02, 0x83, 0xa0, 0x03, 0x02, 0x01,
	0x02, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 
	0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81, 0x90, 0x31, 0x0b,
	0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31,
	0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0e, 0x55, 0x6e,
	0x69, 0x74, 0x65, 0x64, 0x20, 0x4b, 0x69, 0x6e, 0x67, 0x64, 0x6f, 0x6d, 
	0x31, 0x0e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x05, 0x44,
	0x65, 0x72, 0x62, 0x79, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
	0x0a, 0x0c, 0x09, 0x4d, 0x6f, 0x73, 0x71, 0x75, 0x69, 0x74, 0x74, 0x6f,
	0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x02, 0x43, 
	0x41, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d,
	0x6d, 0x6f, 0x73, 0x71, 0x75, 0x69, 0x74, 0x74, 0x6f, 0x2e, 0x6f, 0x72,
	0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x72, 0x6f, 0x67, 0x65, 0x72, 0x40, 
	0x61, 0x74, 0x63, 0x68, 0x6f, 0x6f, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x1e,
	0x17, 0x0d, 0x32, 0x31, 0x31, 0x31, 0x30, 0x33, 0x32, 0x32, 0x33, 0x39,
	0x35, 0x31, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x32, 0x30, 0x31, 0x32,
	0x32, 0x33, 0x39, 0x35, 0x31, 0x5a, 0x30, 0x75, 0x31, 0x0b, 0x30, 0x09, 
	0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30,
	0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x43, 0x61, 0x6c, 0x69,
	0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03,
	0x55, 0x04, 0x07, 0x0c, 0x09, 0x53, 0x75, 0x6e, 0x6e, 0x79, 0x76, 0x61, 
	0x6c, 0x65, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
	0x0d, 0x6c, 0x69, 0x62, 0x77, 0x65, 0x62, 0x73, 0x6f, 0x63, 0x6b, 0x65,
	0x74, 0x73, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c,
	0x0d, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x53, 0x74, 0x72, 0x65, 
	0x61, 0x6d, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c,
	0x04, 0x4d, 0x51, 0x54, 0x54, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
	0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00,
	0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 
	0x01, 0x00, 0xd9, 0x64, 0x2a, 0x43, 0x48, 0x0d, 0xe0, 0x31, 0xa0, 0x2d,
	0xc7, 0x4c, 0x6d, 0xd0, 0x3a, 0x16, 0xe4, 0x90, 0x13, 0xfa, 0xb2, 0x04,
	0x24, 0xf9, 0xd5, 0x3c, 0xa9, 0x5c, 0x77, 0x1a, 0xb3, 0x92, 0x64, 0xc0,
	0x9f, 0x76, 0xfb, 0x4e, 0x83, 0x8c, 0x50, 0xa2, 0xa3, 0xd4, 0xff, 0xd3, 
	0x2f, 0x59, 0xb8, 0xee, 0x7a, 0x8f, 0xcd, 0x41, 0x6c, 0x38, 0xf1, 0x79,
	0xc9, 0xe7, 0x8b, 0xdd, 0xe6, 0x2e, 0x0c, 0xe4, 0x29, 0x5e, 0x3d, 0xf0,
	0x7f, 0x36, 0x87, 0xeb, 0xce, 0xc5, 0xc4, 0xe1, 0x5c, 0x68, 0xed, 0x2b,
	0x4e, 0xb0, 0xee, 0x96, 0xe8, 0xc1, 0xa9, 0xea, 0x85, 0x03, 0x7b, 0x2c, 
	0x6b, 0xf0, 0x7d, 0x98, 0x4b, 0x7a, 0xd2, 0x9d, 0x26, 0x31, 0x74, 0xf6,
	0x21, 0x10, 0x2f, 0x79, 0x7f, 0x79, 0x5d, 0x82, 0xb8, 0x10, 0x7d, 0x82,
	0x52, 0xd0, 0xc2, 0x91, 0xc9, 0x9c, 0xa6, 0x38, 0x00, 0x6b, 0x56, 0x73,
	0x41, 0x01, 0x6f, 0x67, 0x80, 0x3e, 0xfe, 0x18, 0x91, 0xa0, 0x9f, 0x12, 
	0x0b, 0x3e, 0x72, 0xed, 0xe2, 0xbd, 0x5e, 0xa6, 0xd1, 0xd3, 0xbf, 0x1a,
	0x76, 0x98, 0xd4, 0xab, 0xf7, 0x72, 0x52, 0x05, 0xc5, 0x1f, 0xb1, 0xcd,
	0x61, 0xc5, 0x02, 0x60, 0xc9, 0x6d, 0xc1, 0xcb, 0x80, 0x39, 0x77, 0x0c,
	0x99, 0xd6, 0xdd, 0x83, 0xd2, 0x9e, 0x71, 0xd7, 0x2d, 0x4a, 0xa4, 0x51, 
	0xd8, 0xb0, 0x71, 0xaf, 0xf6, 0x92, 0x45, 0x89, 0x4a, 0x5f, 0x3d, 0x66,
	0x92, 0xda, 0x7c, 0xc4, 0x14, 0x1b, 0x2c, 0x15, 0x61, 0xba, 0x95, 0xd3,
	0xed, 0xe7, 0x7d, 0x75, 0x33, 0x43, 0xec, 0x9a, 0x15, 0x27, 0x12, 0xd2,
	0x0b, 0x46, 0x9f, 0x0a, 0xb7, 0xb8, 0x6d, 0x8a, 0xe6, 0xb4, 0x57, 0x28, 
	0x12, 0x48, 0x33, 0x44, 0x45, 0x72, 0xe6, 0x92, 0xac, 0x1e, 0xca, 0xed,
	0xb8, 0x0f, 0xfb, 0x4f, 0x11, 0x6b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3,
	0x1a, 0x30, 0x18, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02,
	0x30, 0x00, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 
	0x02, 0x05, 0xe0, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x61,
	0x85, 0x63, 0xdb, 0xf1, 0x2d, 0xa2, 0xa8, 0x6f, 0x38, 0xe6, 0x69, 0x65,
	0x5e, 0xbe, 0x10, 0xc8, 0x53, 0xea, 0xfb, 0xef, 0xa1, 0x4d, 0x37, 0xb5, 
	0x1c, 0x32, 0x08, 0xf9, 0x09, 0x7a, 0x92, 0x64, 0x25, 0x29, 0x1b, 0x8d,
	0x67, 0x35, 0x28, 0xd1, 0x2e, 0x21, 0x6d, 0xd4, 0x5a, 0xa9, 0x32, 0xfd,
	0xc1, 0x64, 0x30, 0x73, 0xd4, 0xd4, 0x4c, 0x60, 0x32, 0x34, 0xe5, 0x60,
	0x1a, 0x2e, 0x37, 0x6d, 0x5e, 0xb4, 0x4f, 0x0a, 0xbe, 0xc9, 0x3f, 0xab, 
	0xab, 0xd4, 0xd5, 0x90, 0xd0, 0x4f, 0x5f, 0x98, 0x4a, 0x11, 0x3d, 0xe3,
	0xdd, 0xa2, 0x48, 0xf3, 0xec, 0xd3, 0xd4, 0x82, 0x5f, 0x8e, 0x52, 0x11,
	0xe6, 0xa3, 0x86, 0x13, 0xf3, 0x21, 0x53, 0x65, 0x5a, 0x8c, 0x27, 0x18,
	0xbf, 0x6a, 0x0a, 0x66, 0x05, 0x98, 0x01, 0x11, 0x62, 0x7b, 0xe0, 0x6b,
	0xc3, 0x25, 0xcb, 0x4f, 0x14, 0x20, 0xf4, 0x2f, 0x59, 0x26, 0xc8, 0x5f,
	0x87, 0x19, 0x92, 0xca, 0xea, 0xc7, 0xf9, 0x0c, 0x70, 0xa5, 0x4c, 0x99,
	0x08, 0xd8, 0x20, 0xef, 0x60, 0x42, 0xdf, 0xec, 0x1d, 0xde, 0xdf, 0xd8,
	0x19, 0x60, 0x05, 0x97, 0x26, 0x70, 0x6f, 0x92, 0xb7, 0xd1, 0x91, 0xd9, 
	0x54, 0xf0, 0x68, 0x16, 0xf5, 0xb9, 0x94, 0x48, 0x3e, 0x6d, 0x5f, 0xd2,
	0x11, 0xf2, 0x33, 0x7f, 0x6b, 0x1f, 0x08, 0x77, 0xe1, 0x82, 0x18, 0x04,
	0x6c, 0x01, 0x39, 0xf9, 0x75, 0xcb, 0x9d, 0x87, 0x62, 0x11, 0x08, 0x15,
	0xe3, 0xd6, 0x05, 0x79, 0x51, 0xa1, 0x51, 0xb9, 0x9b, 0x61, 0x52, 0x82,
	0x9f, 0x8f, 0x1b, 0x39, 0xbb, 0xb5, 0xcd, 0xb2, 0x37, 0x74, 0xfd, 0x08,
	0x85, 0xe7, 0x8f, 0xbd, 0xe1, 0xd0, 0x18, 0x57, 0xd0, 0x3d, 0xbb, 0x37,
	0x80, 0xbf, 0x3e, 0x5f, 0x6f, 0x2a, 0x1b, 0x73, 0xc7, 0x41, 0xa7, 0x62,
	0x9a, 0x9b, 0x7f
};

static int
app_system_state_nf(lws_state_manager_t *mgr, lws_state_notify_link_t *link,
		    int current, int target)
{
	struct lws_context *context = lws_system_context_from_system_mgr(mgr);
	lws_system_blob_t *ab = lws_system_get_blob(context,
				LWS_SYSBLOB_TYPE_AUTH, 1 /* AUTH_IDX_ROOT */);
	size_t size;

	/*
	 * For the things we care about, let's notice if we are trying to get
	 * past them when we haven't solved them yet, and make the system
	 * state wait while we trigger the dependent action.
	 */
	switch (target) {
	case LWS_SYSTATE_REGISTERED:
		size = lws_system_blob_get_size(ab);
		if (size)
			break;

		/* let's register our canned root token so auth can use it */
		lws_system_blob_direct_set(ab,
				(const uint8_t *)canned_root_token_payload,
				strlen(canned_root_token_payload));
		break;
	case LWS_SYSTATE_OPERATIONAL:
		if (current == LWS_SYSTATE_OPERATIONAL) {
#if defined(LWS_WITH_SECURE_STREAMS_AUTH_SIGV4)

			if (lws_aws_filesystem_credentials_helper(
						  "~/.aws/credentials",
						  "aws_access_key_id",
						  "aws_secret_access_key",
						  &aws_keyid, &aws_key))
				return -1;

			lws_ss_sigv4_set_aws_key(context, 0, aws_keyid, aws_key);
#endif

			lws_system_blob_direct_set(
				lws_system_get_blob(context, LWS_SYSBLOB_TYPE_MQTT_CLIENT_ID, 0),
				(const uint8_t*)TEST_CLIENT_ID, strlen(TEST_CLIENT_ID));

			lws_system_blob_direct_set(
				lws_system_get_blob(context, LWS_SYSBLOB_TYPE_CLIENT_CERT_DER, 0),
				client_cert_der, sizeof(client_cert_der));

			lws_system_blob_direct_set(
				lws_system_get_blob(context, LWS_SYSBLOB_TYPE_CLIENT_KEY_DER, 0),
				client_key, sizeof(client_key));

			/*
			 * At this point we have DHCP, ntp, system auth token
			 * and we can reasonably create the proxy
			 */
			if (lws_ss_proxy_create(context, ibind, port)) {
				lwsl_err("%s: failed to create ss proxy\n",
						__func__);
				return -1;
			}
		}
		break;
	case LWS_SYSTATE_POLICY_INVALID:
		/*
		 * This is a NOP since we used direct set... but in a real
		 * system this could easily change to be done on the heap, then
		 * this would be important
		 */
		lws_system_blob_destroy(lws_system_get_blob(context,
					LWS_SYSBLOB_TYPE_AUTH,
					1 /* AUTH_IDX_ROOT */));
		break;
	}

	return 0;
}

static lws_state_notify_link_t * const app_notifier_list[] = {
	&nl, NULL
};

#if defined(LWS_WITH_SYS_METRICS)

static int
my_metric_report(lws_metric_pub_t *mp)
{
	lws_metric_bucket_t *sub = mp->u.hist.head;
	char buf[192];

	do {
		if (lws_metrics_format(mp, &sub, buf, sizeof(buf)))
			lwsl_user("%s: %s\n", __func__, buf);
	} while ((mp->flags & LWSMTFL_REPORT_HIST) && sub);

	/* 0 = leave metric to accumulate, 1 = reset the metric */

	return 1;
}

static const lws_system_ops_t system_ops = {
	.metric_report = my_metric_report,
};

#endif

static void
sigint_handler(int sig)
{
	lwsl_notice("%s\n", __func__);
	interrupted = 1;
	lws_cancel_service(context);
}

int main(int argc, const char **argv)
{
	struct lws_context_creation_info info;
	const char *p;
	int n = 0;

	memset(&info, 0, sizeof info);
	lws_cmdline_option_handle_builtin(argc, argv, &info);

	signal(SIGINT, sigint_handler);

	/* connect to ssproxy via UDS by default, else via tcp with this port */
	if ((p = lws_cmdline_option(argc, argv, "-p")))
		port = atoi(p);

	/* UDS "proxy.ss.lws" in abstract namespace, else this socket path;
	 * when -p given this can specify the network interface to bind to */
	if ((p = lws_cmdline_option(argc, argv, "-i")))
		ibind = p;

	lwsl_user("LWS secure streams Proxy [-d<verb>]\n");

	info.fd_limit_per_thread = 1 + 26 + 1;
	info.port = CONTEXT_PORT_NO_LISTEN;
	info.pss_policies_json = default_ss_policy;
	info.options = LWS_SERVER_OPTION_EXPLICIT_VHOSTS |
		       LWS_SERVER_OPTION_H2_JUST_FIX_WINDOW_UPDATE_OVERFLOW |
		       LWS_SERVER_OPTION_DO_SSL_GLOBAL_INIT;

	/* integrate us with lws system state management when context created */
	nl.name = "app";
	nl.notify_cb = app_system_state_nf;
	info.register_notifier_list = app_notifier_list;

	info.pt_serv_buf_size = (unsigned int)((6144 * 2) + 2048);
	info.max_http_header_data = (unsigned short)(6144 + 2048);

#if defined(LWS_WITH_SYS_METRICS)
	info.system_ops = &system_ops;
	info.metrics_prefix = "ssproxy";
#endif

	context = lws_create_context(&info);
	if (!context) {
		lwsl_err("lws init failed\n");
		return 1;
	}

	/* the event loop */

	do {
		n = lws_service(context, 0);
	} while (n >= 0 && !interrupted);

	bad = 0;

#if defined(LWS_WITH_SECURE_STREAMS_AUTH_SIGV4)
	if (aws_keyid)
		free(aws_keyid);
	if (aws_key)
		free(aws_key);
#endif

	lws_context_destroy(context);
	lwsl_user("Completed: %s\n", bad ? "failed" : "OK");

	return bad;
}
