"""
The hardcoded information on this module was loaded from the Unicode file Scripts-16.0.0.txt
https://www.unicode.org/Public/UNIDATA/Scripts.txt
Python's standard "unicodedata" library doesn't offer this table.
"""

from enum import IntEnum
from functools import lru_cache


class UnicodeScript(IntEnum):
    COMMON = 0
    LATIN = 1
    GREEK = 2
    CYRILLIC = 3
    ARMENIAN = 4
    HEBREW = 5
    ARABIC = 6
    SYRIAC = 7
    THAANA = 8
    DEVANAGARI = 9
    BENGALI = 10
    GURMUKHI = 11
    GUJARATI = 12
    ORIYA = 13
    TAMIL = 14
    TELUGU = 15
    KANNADA = 16
    MALAYALAM = 17
    SINHALA = 18
    THAI = 19
    LAO = 20
    TIBETAN = 21
    MYANMAR = 22
    GEORGIAN = 23
    HANGUL = 24
    ETHIOPIC = 25
    CHEROKEE = 26
    CANADIAN_ABORIGINAL = 27
    OGHAM = 28
    RUNIC = 29
    KHMER = 30
    MONGOLIAN = 31
    HIRAGANA = 32
    KATAKANA = 33
    BOPOMOFO = 34
    HAN = 35
    YI = 36
    OLD_ITALIC = 37
    GOTHIC = 38
    DESERET = 39
    INHERITED = 40
    TAGALOG = 41
    HANUNOO = 42
    BUHID = 43
    TAGBANWA = 44
    LIMBU = 45
    TAI_LE = 46
    LINEAR_B = 47
    UGARITIC = 48
    SHAVIAN = 49
    OSMANYA = 50
    CYPRIOT = 51
    BRAILLE = 52
    BUGINESE = 53
    COPTIC = 54
    NEW_TAI_LUE = 55
    GLAGOLITIC = 56
    TIFINAGH = 57
    SYLOTI_NAGRI = 58
    OLD_PERSIAN = 59
    KHAROSHTHI = 60
    BALINESE = 61
    CUNEIFORM = 62
    PHOENICIAN = 63
    PHAGS_PA = 64
    NKO = 65
    SUNDANESE = 66
    LEPCHA = 67
    OL_CHIKI = 68
    VAI = 69
    SAURASHTRA = 70
    KAYAH_LI = 71
    REJANG = 72
    LYCIAN = 73
    CARIAN = 74
    LYDIAN = 75
    CHAM = 76
    TAI_THAM = 77
    TAI_VIET = 78
    AVESTAN = 79
    EGYPTIAN_HIEROGLYPHS = 80
    SAMARITAN = 81
    LISU = 82
    BAMUM = 83
    JAVANESE = 84
    MEETEI_MAYEK = 85
    IMPERIAL_ARAMAIC = 86
    OLD_SOUTH_ARABIAN = 87
    INSCRIPTIONAL_PARTHIAN = 88
    INSCRIPTIONAL_PAHLAVI = 89
    OLD_TURKIC = 90
    KAITHI = 91
    BATAK = 92
    BRAHMI = 93
    MANDAIC = 94
    CHAKMA = 95
    MEROITIC_CURSIVE = 96
    MEROITIC_HIEROGLYPHS = 97
    MIAO = 98
    SHARADA = 99
    SORA_SOMPENG = 100
    TAKRI = 101
    CAUCASIAN_ALBANIAN = 102
    BASSA_VAH = 103
    DUPLOYAN = 104
    ELBASAN = 105
    GRANTHA = 106
    PAHAWH_HMONG = 107
    KHOJKI = 108
    LINEAR_A = 109
    MAHAJANI = 110
    MANICHAEAN = 111
    MENDE_KIKAKUI = 112
    MODI = 113
    MRO = 114
    OLD_NORTH_ARABIAN = 115
    NABATAEAN = 116
    PALMYRENE = 117
    PAU_CIN_HAU = 118
    OLD_PERMIC = 119
    PSALTER_PAHLAVI = 120
    SIDDHAM = 121
    KHUDAWADI = 122
    TIRHUTA = 123
    WARANG_CITI = 124
    AHOM = 125
    ANATOLIAN_HIEROGLYPHS = 126
    HATRAN = 127
    MULTANI = 128
    OLD_HUNGARIAN = 129
    SIGNWRITING = 130
    ADLAM = 131
    BHAIKSUKI = 132
    MARCHEN = 133
    NEWA = 134
    OSAGE = 135
    TANGUT = 136
    MASARAM_GONDI = 137
    NUSHU = 138
    SOYOMBO = 139
    ZANABAZAR_SQUARE = 140
    DOGRA = 141
    GUNJALA_GONDI = 142
    MAKASAR = 143
    MEDEFAIDRIN = 144
    HANIFI_ROHINGYA = 145
    SOGDIAN = 146
    OLD_SOGDIAN = 147
    ELYMAIC = 148
    NANDINAGARI = 149
    NYIAKENG_PUACHUE_HMONG = 150
    WANCHO = 151
    CHORASMIAN = 152
    DIVES_AKURU = 153
    KHITAN_SMALL_SCRIPT = 154
    YEZIDI = 155
    CYPRO_MINOAN = 156
    OLD_UYGHUR = 157
    TANGSA = 158
    TOTO = 159
    VITHKUQI = 160
    KAWI = 161
    NAG_MUNDARI = 162
    GARAY = 163
    GURUNG_KHEMA = 164
    KIRAT_RAI = 165
    OL_ONAL = 166
    SUNUWAR = 167
    TODHRI = 168
    TULU_TIGALARI = 169
    UNKNOWN = 999


UNICODE_RANGE_TO_SCRIPT: tuple[tuple[int, int, int], ...] = (
    (0, 64, 0),
    (65, 90, 1),
    (91, 96, 0),
    (97, 122, 1),
    (123, 169, 0),
    (170, 170, 1),
    (171, 185, 0),
    (186, 186, 1),
    (187, 191, 0),
    (192, 214, 1),
    (215, 215, 0),
    (216, 246, 1),
    (247, 247, 0),
    (248, 696, 1),
    (697, 735, 0),
    (736, 740, 1),
    (741, 745, 0),
    (746, 747, 34),
    (748, 767, 0),
    (768, 879, 40),
    (880, 883, 2),
    (884, 884, 0),
    (885, 887, 2),
    (890, 893, 2),
    (894, 894, 0),
    (895, 895, 2),
    (900, 900, 2),
    (901, 901, 0),
    (902, 902, 2),
    (903, 903, 0),
    (904, 906, 2),
    (908, 908, 2),
    (910, 929, 2),
    (931, 993, 2),
    (994, 1007, 54),
    (1008, 1023, 2),
    (1024, 1156, 3),
    (1157, 1158, 40),
    (1159, 1327, 3),
    (1329, 1366, 4),
    (1369, 1418, 4),
    (1421, 1423, 4),
    (1425, 1479, 5),
    (1488, 1514, 5),
    (1519, 1524, 5),
    (1536, 1540, 6),
    (1541, 1541, 0),
    (1542, 1547, 6),
    (1548, 1548, 0),
    (1549, 1562, 6),
    (1563, 1563, 0),
    (1564, 1566, 6),
    (1567, 1567, 0),
    (1568, 1599, 6),
    (1600, 1600, 0),
    (1601, 1610, 6),
    (1611, 1621, 40),
    (1622, 1647, 6),
    (1648, 1648, 40),
    (1649, 1756, 6),
    (1757, 1757, 0),
    (1758, 1791, 6),
    (1792, 1805, 7),
    (1807, 1866, 7),
    (1869, 1871, 7),
    (1872, 1919, 6),
    (1920, 1969, 8),
    (1984, 2042, 65),
    (2045, 2047, 65),
    (2048, 2093, 81),
    (2096, 2110, 81),
    (2112, 2139, 94),
    (2142, 2142, 94),
    (2144, 2154, 7),
    (2160, 2190, 6),
    (2192, 2193, 6),
    (2199, 2273, 6),
    (2274, 2274, 0),
    (2275, 2303, 6),
    (2304, 2384, 9),
    (2385, 2388, 40),
    (2389, 2403, 9),
    (2404, 2405, 0),
    (2406, 2431, 9),
    (2432, 2435, 10),
    (2437, 2444, 10),
    (2447, 2448, 10),
    (2451, 2472, 10),
    (2474, 2480, 10),
    (2482, 2482, 10),
    (2486, 2489, 10),
    (2492, 2500, 10),
    (2503, 2504, 10),
    (2507, 2510, 10),
    (2519, 2519, 10),
    (2524, 2525, 10),
    (2527, 2531, 10),
    (2534, 2558, 10),
    (2561, 2563, 11),
    (2565, 2570, 11),
    (2575, 2576, 11),
    (2579, 2600, 11),
    (2602, 2608, 11),
    (2610, 2611, 11),
    (2613, 2614, 11),
    (2616, 2617, 11),
    (2620, 2620, 11),
    (2622, 2626, 11),
    (2631, 2632, 11),
    (2635, 2637, 11),
    (2641, 2641, 11),
    (2649, 2652, 11),
    (2654, 2654, 11),
    (2662, 2678, 11),
    (2689, 2691, 12),
    (2693, 2701, 12),
    (2703, 2705, 12),
    (2707, 2728, 12),
    (2730, 2736, 12),
    (2738, 2739, 12),
    (2741, 2745, 12),
    (2748, 2757, 12),
    (2759, 2761, 12),
    (2763, 2765, 12),
    (2768, 2768, 12),
    (2784, 2787, 12),
    (2790, 2801, 12),
    (2809, 2815, 12),
    (2817, 2819, 13),
    (2821, 2828, 13),
    (2831, 2832, 13),
    (2835, 2856, 13),
    (2858, 2864, 13),
    (2866, 2867, 13),
    (2869, 2873, 13),
    (2876, 2884, 13),
    (2887, 2888, 13),
    (2891, 2893, 13),
    (2901, 2903, 13),
    (2908, 2909, 13),
    (2911, 2915, 13),
    (2918, 2935, 13),
    (2946, 2947, 14),
    (2949, 2954, 14),
    (2958, 2960, 14),
    (2962, 2965, 14),
    (2969, 2970, 14),
    (2972, 2972, 14),
    (2974, 2975, 14),
    (2979, 2980, 14),
    (2984, 2986, 14),
    (2990, 3001, 14),
    (3006, 3010, 14),
    (3014, 3016, 14),
    (3018, 3021, 14),
    (3024, 3024, 14),
    (3031, 3031, 14),
    (3046, 3066, 14),
    (3072, 3084, 15),
    (3086, 3088, 15),
    (3090, 3112, 15),
    (3114, 3129, 15),
    (3132, 3140, 15),
    (3142, 3144, 15),
    (3146, 3149, 15),
    (3157, 3158, 15),
    (3160, 3162, 15),
    (3165, 3165, 15),
    (3168, 3171, 15),
    (3174, 3183, 15),
    (3191, 3199, 15),
    (3200, 3212, 16),
    (3214, 3216, 16),
    (3218, 3240, 16),
    (3242, 3251, 16),
    (3253, 3257, 16),
    (3260, 3268, 16),
    (3270, 3272, 16),
    (3274, 3277, 16),
    (3285, 3286, 16),
    (3293, 3294, 16),
    (3296, 3299, 16),
    (3302, 3311, 16),
    (3313, 3315, 16),
    (3328, 3340, 17),
    (3342, 3344, 17),
    (3346, 3396, 17),
    (3398, 3400, 17),
    (3402, 3407, 17),
    (3412, 3427, 17),
    (3430, 3455, 17),
    (3457, 3459, 18),
    (3461, 3478, 18),
    (3482, 3505, 18),
    (3507, 3515, 18),
    (3517, 3517, 18),
    (3520, 3526, 18),
    (3530, 3530, 18),
    (3535, 3540, 18),
    (3542, 3542, 18),
    (3544, 3551, 18),
    (3558, 3567, 18),
    (3570, 3572, 18),
    (3585, 3642, 19),
    (3647, 3647, 0),
    (3648, 3675, 19),
    (3713, 3714, 20),
    (3716, 3716, 20),
    (3718, 3722, 20),
    (3724, 3747, 20),
    (3749, 3749, 20),
    (3751, 3773, 20),
    (3776, 3780, 20),
    (3782, 3782, 20),
    (3784, 3790, 20),
    (3792, 3801, 20),
    (3804, 3807, 20),
    (3840, 3911, 21),
    (3913, 3948, 21),
    (3953, 3991, 21),
    (3993, 4028, 21),
    (4030, 4044, 21),
    (4046, 4052, 21),
    (4053, 4056, 0),
    (4057, 4058, 21),
    (4096, 4255, 22),
    (4256, 4293, 23),
    (4295, 4295, 23),
    (4301, 4301, 23),
    (4304, 4346, 23),
    (4347, 4347, 0),
    (4348, 4351, 23),
    (4352, 4607, 24),
    (4608, 4680, 25),
    (4682, 4685, 25),
    (4688, 4694, 25),
    (4696, 4696, 25),
    (4698, 4701, 25),
    (4704, 4744, 25),
    (4746, 4749, 25),
    (4752, 4784, 25),
    (4786, 4789, 25),
    (4792, 4798, 25),
    (4800, 4800, 25),
    (4802, 4805, 25),
    (4808, 4822, 25),
    (4824, 4880, 25),
    (4882, 4885, 25),
    (4888, 4954, 25),
    (4957, 4988, 25),
    (4992, 5017, 25),
    (5024, 5109, 26),
    (5112, 5117, 26),
    (5120, 5759, 27),
    (5760, 5788, 28),
    (5792, 5866, 29),
    (5867, 5869, 0),
    (5870, 5880, 29),
    (5888, 5909, 41),
    (5919, 5919, 41),
    (5920, 5940, 42),
    (5941, 5942, 0),
    (5952, 5971, 43),
    (5984, 5996, 44),
    (5998, 6000, 44),
    (6002, 6003, 44),
    (6016, 6109, 30),
    (6112, 6121, 30),
    (6128, 6137, 30),
    (6144, 6145, 31),
    (6146, 6147, 0),
    (6148, 6148, 31),
    (6149, 6149, 0),
    (6150, 6169, 31),
    (6176, 6264, 31),
    (6272, 6314, 31),
    (6320, 6389, 27),
    (6400, 6430, 45),
    (6432, 6443, 45),
    (6448, 6459, 45),
    (6464, 6464, 45),
    (6468, 6479, 45),
    (6480, 6509, 46),
    (6512, 6516, 46),
    (6528, 6571, 55),
    (6576, 6601, 55),
    (6608, 6618, 55),
    (6622, 6623, 55),
    (6624, 6655, 30),
    (6656, 6683, 53),
    (6686, 6687, 53),
    (6688, 6750, 77),
    (6752, 6780, 77),
    (6783, 6793, 77),
    (6800, 6809, 77),
    (6816, 6829, 77),
    (6832, 6862, 40),
    (6912, 6988, 61),
    (6990, 7039, 61),
    (7040, 7103, 66),
    (7104, 7155, 92),
    (7164, 7167, 92),
    (7168, 7223, 67),
    (7227, 7241, 67),
    (7245, 7247, 67),
    (7248, 7295, 68),
    (7296, 7306, 3),
    (7312, 7354, 23),
    (7357, 7359, 23),
    (7360, 7367, 66),
    (7376, 7378, 40),
    (7379, 7379, 0),
    (7380, 7392, 40),
    (7393, 7393, 0),
    (7394, 7400, 40),
    (7401, 7404, 0),
    (7405, 7405, 40),
    (7406, 7411, 0),
    (7412, 7412, 40),
    (7413, 7415, 0),
    (7416, 7417, 40),
    (7418, 7418, 0),
    (7424, 7461, 1),
    (7462, 7466, 2),
    (7467, 7467, 3),
    (7468, 7516, 1),
    (7517, 7521, 2),
    (7522, 7525, 1),
    (7526, 7530, 2),
    (7531, 7543, 1),
    (7544, 7544, 3),
    (7545, 7614, 1),
    (7615, 7615, 2),
    (7616, 7679, 40),
    (7680, 7935, 1),
    (7936, 7957, 2),
    (7960, 7965, 2),
    (7968, 8005, 2),
    (8008, 8013, 2),
    (8016, 8023, 2),
    (8025, 8025, 2),
    (8027, 8027, 2),
    (8029, 8029, 2),
    (8031, 8061, 2),
    (8064, 8116, 2),
    (8118, 8132, 2),
    (8134, 8147, 2),
    (8150, 8155, 2),
    (8157, 8175, 2),
    (8178, 8180, 2),
    (8182, 8190, 2),
    (8192, 8203, 0),
    (8204, 8205, 40),
    (8206, 8292, 0),
    (8294, 8304, 0),
    (8305, 8305, 1),
    (8308, 8318, 0),
    (8319, 8319, 1),
    (8320, 8334, 0),
    (8336, 8348, 1),
    (8352, 8384, 0),
    (8400, 8432, 40),
    (8448, 8485, 0),
    (8486, 8486, 2),
    (8487, 8489, 0),
    (8490, 8491, 1),
    (8492, 8497, 0),
    (8498, 8498, 1),
    (8499, 8525, 0),
    (8526, 8526, 1),
    (8527, 8543, 0),
    (8544, 8584, 1),
    (8585, 8587, 0),
    (8592, 9257, 0),
    (9280, 9290, 0),
    (9312, 10239, 0),
    (10240, 10495, 52),
    (10496, 11123, 0),
    (11126, 11157, 0),
    (11159, 11263, 0),
    (11264, 11359, 56),
    (11360, 11391, 1),
    (11392, 11507, 54),
    (11513, 11519, 54),
    (11520, 11557, 23),
    (11559, 11559, 23),
    (11565, 11565, 23),
    (11568, 11623, 57),
    (11631, 11632, 57),
    (11647, 11647, 57),
    (11648, 11670, 25),
    (11680, 11686, 25),
    (11688, 11694, 25),
    (11696, 11702, 25),
    (11704, 11710, 25),
    (11712, 11718, 25),
    (11720, 11726, 25),
    (11728, 11734, 25),
    (11736, 11742, 25),
    (11744, 11775, 3),
    (11776, 11869, 0),
    (11904, 11929, 35),
    (11931, 12019, 35),
    (12032, 12245, 35),
    (12272, 12292, 0),
    (12293, 12293, 35),
    (12294, 12294, 0),
    (12295, 12295, 35),
    (12296, 12320, 0),
    (12321, 12329, 35),
    (12330, 12333, 40),
    (12334, 12335, 24),
    (12336, 12343, 0),
    (12344, 12347, 35),
    (12348, 12351, 0),
    (12353, 12438, 32),
    (12441, 12442, 40),
    (12443, 12444, 0),
    (12445, 12447, 32),
    (12448, 12448, 0),
    (12449, 12538, 33),
    (12539, 12540, 0),
    (12541, 12543, 33),
    (12549, 12591, 34),
    (12593, 12686, 24),
    (12688, 12703, 0),
    (12704, 12735, 34),
    (12736, 12773, 0),
    (12783, 12783, 0),
    (12784, 12799, 33),
    (12800, 12830, 24),
    (12832, 12895, 0),
    (12896, 12926, 24),
    (12927, 13007, 0),
    (13008, 13054, 33),
    (13055, 13055, 0),
    (13056, 13143, 33),
    (13144, 13311, 0),
    (13312, 19903, 35),
    (19904, 19967, 0),
    (19968, 40959, 35),
    (40960, 42124, 36),
    (42128, 42182, 36),
    (42192, 42239, 82),
    (42240, 42539, 69),
    (42560, 42655, 3),
    (42656, 42743, 83),
    (42752, 42785, 0),
    (42786, 42887, 1),
    (42888, 42890, 0),
    (42891, 42957, 1),
    (42960, 42961, 1),
    (42963, 42963, 1),
    (42965, 42972, 1),
    (42994, 43007, 1),
    (43008, 43052, 58),
    (43056, 43065, 0),
    (43072, 43127, 64),
    (43136, 43205, 70),
    (43214, 43225, 70),
    (43232, 43263, 9),
    (43264, 43309, 71),
    (43310, 43310, 0),
    (43311, 43311, 71),
    (43312, 43347, 72),
    (43359, 43359, 72),
    (43360, 43388, 24),
    (43392, 43469, 84),
    (43471, 43471, 0),
    (43472, 43481, 84),
    (43486, 43487, 84),
    (43488, 43518, 22),
    (43520, 43574, 76),
    (43584, 43597, 76),
    (43600, 43609, 76),
    (43612, 43615, 76),
    (43616, 43647, 22),
    (43648, 43714, 78),
    (43739, 43743, 78),
    (43744, 43766, 85),
    (43777, 43782, 25),
    (43785, 43790, 25),
    (43793, 43798, 25),
    (43808, 43814, 25),
    (43816, 43822, 25),
    (43824, 43866, 1),
    (43867, 43867, 0),
    (43868, 43876, 1),
    (43877, 43877, 2),
    (43878, 43881, 1),
    (43882, 43883, 0),
    (43888, 43967, 26),
    (43968, 44013, 85),
    (44016, 44025, 85),
    (44032, 55203, 24),
    (55216, 55238, 24),
    (55243, 55291, 24),
    (63744, 64109, 35),
    (64112, 64217, 35),
    (64256, 64262, 1),
    (64275, 64279, 4),
    (64285, 64310, 5),
    (64312, 64316, 5),
    (64318, 64318, 5),
    (64320, 64321, 5),
    (64323, 64324, 5),
    (64326, 64335, 5),
    (64336, 64450, 6),
    (64467, 64829, 6),
    (64830, 64831, 0),
    (64832, 64911, 6),
    (64914, 64967, 6),
    (64975, 64975, 6),
    (65008, 65023, 6),
    (65024, 65039, 40),
    (65040, 65049, 0),
    (65056, 65069, 40),
    (65070, 65071, 3),
    (65072, 65106, 0),
    (65108, 65126, 0),
    (65128, 65131, 0),
    (65136, 65140, 6),
    (65142, 65276, 6),
    (65279, 65279, 0),
    (65281, 65312, 0),
    (65313, 65338, 1),
    (65339, 65344, 0),
    (65345, 65370, 1),
    (65371, 65381, 0),
    (65382, 65391, 33),
    (65392, 65392, 0),
    (65393, 65437, 33),
    (65438, 65439, 0),
    (65440, 65470, 24),
    (65474, 65479, 24),
    (65482, 65487, 24),
    (65490, 65495, 24),
    (65498, 65500, 24),
    (65504, 65510, 0),
    (65512, 65518, 0),
    (65529, 65533, 0),
    (65536, 65547, 47),
    (65549, 65574, 47),
    (65576, 65594, 47),
    (65596, 65597, 47),
    (65599, 65613, 47),
    (65616, 65629, 47),
    (65664, 65786, 47),
    (65792, 65794, 0),
    (65799, 65843, 0),
    (65847, 65855, 0),
    (65856, 65934, 2),
    (65936, 65948, 0),
    (65952, 65952, 2),
    (66000, 66044, 0),
    (66045, 66045, 40),
    (66176, 66204, 73),
    (66208, 66256, 74),
    (66272, 66272, 40),
    (66273, 66299, 0),
    (66304, 66339, 37),
    (66349, 66351, 37),
    (66352, 66378, 38),
    (66384, 66426, 119),
    (66432, 66461, 48),
    (66463, 66463, 48),
    (66464, 66499, 59),
    (66504, 66517, 59),
    (66560, 66639, 39),
    (66640, 66687, 49),
    (66688, 66717, 50),
    (66720, 66729, 50),
    (66736, 66771, 135),
    (66776, 66811, 135),
    (66816, 66855, 105),
    (66864, 66915, 102),
    (66927, 66927, 102),
    (66928, 66938, 160),
    (66940, 66954, 160),
    (66956, 66962, 160),
    (66964, 66965, 160),
    (66967, 66977, 160),
    (66979, 66993, 160),
    (66995, 67001, 160),
    (67003, 67004, 160),
    (67008, 67059, 168),
    (67072, 67382, 109),
    (67392, 67413, 109),
    (67424, 67431, 109),
    (67456, 67461, 1),
    (67463, 67504, 1),
    (67506, 67514, 1),
    (67584, 67589, 51),
    (67592, 67592, 51),
    (67594, 67637, 51),
    (67639, 67640, 51),
    (67644, 67644, 51),
    (67647, 67647, 51),
    (67648, 67669, 86),
    (67671, 67679, 86),
    (67680, 67711, 117),
    (67712, 67742, 116),
    (67751, 67759, 116),
    (67808, 67826, 127),
    (67828, 67829, 127),
    (67835, 67839, 127),
    (67840, 67867, 63),
    (67871, 67871, 63),
    (67872, 67897, 75),
    (67903, 67903, 75),
    (67968, 67999, 97),
    (68000, 68023, 96),
    (68028, 68047, 96),
    (68050, 68095, 96),
    (68096, 68099, 60),
    (68101, 68102, 60),
    (68108, 68115, 60),
    (68117, 68119, 60),
    (68121, 68149, 60),
    (68152, 68154, 60),
    (68159, 68168, 60),
    (68176, 68184, 60),
    (68192, 68223, 87),
    (68224, 68255, 115),
    (68288, 68326, 111),
    (68331, 68342, 111),
    (68352, 68405, 79),
    (68409, 68415, 79),
    (68416, 68437, 88),
    (68440, 68447, 88),
    (68448, 68466, 89),
    (68472, 68479, 89),
    (68480, 68497, 120),
    (68505, 68508, 120),
    (68521, 68527, 120),
    (68608, 68680, 90),
    (68736, 68786, 129),
    (68800, 68850, 129),
    (68858, 68863, 129),
    (68864, 68903, 145),
    (68912, 68921, 145),
    (68928, 68965, 163),
    (68969, 68997, 163),
    (69006, 69007, 163),
    (69216, 69246, 6),
    (69248, 69289, 155),
    (69291, 69293, 155),
    (69296, 69297, 155),
    (69314, 69316, 6),
    (69372, 69375, 6),
    (69376, 69415, 147),
    (69424, 69465, 146),
    (69488, 69513, 157),
    (69552, 69579, 152),
    (69600, 69622, 148),
    (69632, 69709, 93),
    (69714, 69749, 93),
    (69759, 69759, 93),
    (69760, 69826, 91),
    (69837, 69837, 91),
    (69840, 69864, 100),
    (69872, 69881, 100),
    (69888, 69940, 95),
    (69942, 69959, 95),
    (69968, 70006, 110),
    (70016, 70111, 99),
    (70113, 70132, 18),
    (70144, 70161, 108),
    (70163, 70209, 108),
    (70272, 70278, 128),
    (70280, 70280, 128),
    (70282, 70285, 128),
    (70287, 70301, 128),
    (70303, 70313, 128),
    (70320, 70378, 122),
    (70384, 70393, 122),
    (70400, 70403, 106),
    (70405, 70412, 106),
    (70415, 70416, 106),
    (70419, 70440, 106),
    (70442, 70448, 106),
    (70450, 70451, 106),
    (70453, 70457, 106),
    (70459, 70459, 40),
    (70460, 70468, 106),
    (70471, 70472, 106),
    (70475, 70477, 106),
    (70480, 70480, 106),
    (70487, 70487, 106),
    (70493, 70499, 106),
    (70502, 70508, 106),
    (70512, 70516, 106),
    (70528, 70537, 169),
    (70539, 70539, 169),
    (70542, 70542, 169),
    (70544, 70581, 169),
    (70583, 70592, 169),
    (70594, 70594, 169),
    (70597, 70597, 169),
    (70599, 70602, 169),
    (70604, 70613, 169),
    (70615, 70616, 169),
    (70625, 70626, 169),
    (70656, 70747, 134),
    (70749, 70753, 134),
    (70784, 70855, 123),
    (70864, 70873, 123),
    (71040, 71093, 121),
    (71096, 71133, 121),
    (71168, 71236, 113),
    (71248, 71257, 113),
    (71264, 71276, 31),
    (71296, 71353, 101),
    (71360, 71369, 101),
    (71376, 71395, 22),
    (71424, 71450, 125),
    (71453, 71467, 125),
    (71472, 71494, 125),
    (71680, 71739, 141),
    (71840, 71922, 124),
    (71935, 71935, 124),
    (71936, 71942, 153),
    (71945, 71945, 153),
    (71948, 71955, 153),
    (71957, 71958, 153),
    (71960, 71989, 153),
    (71991, 71992, 153),
    (71995, 72006, 153),
    (72016, 72025, 153),
    (72096, 72103, 149),
    (72106, 72151, 149),
    (72154, 72164, 149),
    (72192, 72263, 140),
    (72272, 72354, 139),
    (72368, 72383, 27),
    (72384, 72440, 118),
    (72448, 72457, 9),
    (72640, 72673, 167),
    (72688, 72697, 167),
    (72704, 72712, 132),
    (72714, 72758, 132),
    (72760, 72773, 132),
    (72784, 72812, 132),
    (72816, 72847, 133),
    (72850, 72871, 133),
    (72873, 72886, 133),
    (72960, 72966, 137),
    (72968, 72969, 137),
    (72971, 73014, 137),
    (73018, 73018, 137),
    (73020, 73021, 137),
    (73023, 73031, 137),
    (73040, 73049, 137),
    (73056, 73061, 142),
    (73063, 73064, 142),
    (73066, 73102, 142),
    (73104, 73105, 142),
    (73107, 73112, 142),
    (73120, 73129, 142),
    (73440, 73464, 143),
    (73472, 73488, 161),
    (73490, 73530, 161),
    (73534, 73562, 161),
    (73648, 73648, 82),
    (73664, 73713, 14),
    (73727, 73727, 14),
    (73728, 74649, 62),
    (74752, 74862, 62),
    (74864, 74868, 62),
    (74880, 75075, 62),
    (77712, 77810, 156),
    (77824, 78933, 80),
    (78944, 82938, 80),
    (82944, 83526, 126),
    (90368, 90425, 164),
    (92160, 92728, 83),
    (92736, 92766, 114),
    (92768, 92777, 114),
    (92782, 92783, 114),
    (92784, 92862, 158),
    (92864, 92873, 158),
    (92880, 92909, 103),
    (92912, 92917, 103),
    (92928, 92997, 107),
    (93008, 93017, 107),
    (93019, 93025, 107),
    (93027, 93047, 107),
    (93053, 93071, 107),
    (93504, 93561, 165),
    (93760, 93850, 144),
    (93952, 94026, 98),
    (94031, 94087, 98),
    (94095, 94111, 98),
    (94176, 94176, 136),
    (94177, 94177, 138),
    (94178, 94179, 35),
    (94180, 94180, 154),
    (94192, 94193, 35),
    (94208, 100343, 136),
    (100352, 101119, 136),
    (101120, 101589, 154),
    (101631, 101631, 154),
    (101632, 101640, 136),
    (110576, 110579, 33),
    (110581, 110587, 33),
    (110589, 110590, 33),
    (110592, 110592, 33),
    (110593, 110879, 32),
    (110880, 110882, 33),
    (110898, 110898, 32),
    (110928, 110930, 32),
    (110933, 110933, 33),
    (110948, 110951, 33),
    (110960, 111355, 138),
    (113664, 113770, 104),
    (113776, 113788, 104),
    (113792, 113800, 104),
    (113808, 113817, 104),
    (113820, 113823, 104),
    (113824, 113827, 0),
    (117760, 118009, 0),
    (118016, 118451, 0),
    (118528, 118573, 40),
    (118576, 118598, 40),
    (118608, 118723, 0),
    (118784, 119029, 0),
    (119040, 119078, 0),
    (119081, 119142, 0),
    (119143, 119145, 40),
    (119146, 119162, 0),
    (119163, 119170, 40),
    (119171, 119172, 0),
    (119173, 119179, 40),
    (119180, 119209, 0),
    (119210, 119213, 40),
    (119214, 119274, 0),
    (119296, 119365, 2),
    (119488, 119507, 0),
    (119520, 119539, 0),
    (119552, 119638, 0),
    (119648, 119672, 0),
    (119808, 119892, 0),
    (119894, 119964, 0),
    (119966, 119967, 0),
    (119970, 119970, 0),
    (119973, 119974, 0),
    (119977, 119980, 0),
    (119982, 119993, 0),
    (119995, 119995, 0),
    (119997, 120003, 0),
    (120005, 120069, 0),
    (120071, 120074, 0),
    (120077, 120084, 0),
    (120086, 120092, 0),
    (120094, 120121, 0),
    (120123, 120126, 0),
    (120128, 120132, 0),
    (120134, 120134, 0),
    (120138, 120144, 0),
    (120146, 120485, 0),
    (120488, 120779, 0),
    (120782, 120831, 0),
    (120832, 121483, 130),
    (121499, 121503, 130),
    (121505, 121519, 130),
    (122624, 122654, 1),
    (122661, 122666, 1),
    (122880, 122886, 56),
    (122888, 122904, 56),
    (122907, 122913, 56),
    (122915, 122916, 56),
    (122918, 122922, 56),
    (122928, 122989, 3),
    (123023, 123023, 3),
    (123136, 123180, 150),
    (123184, 123197, 150),
    (123200, 123209, 150),
    (123214, 123215, 150),
    (123536, 123566, 159),
    (123584, 123641, 151),
    (123647, 123647, 151),
    (124112, 124153, 162),
    (124368, 124410, 166),
    (124415, 124415, 166),
    (124896, 124902, 25),
    (124904, 124907, 25),
    (124909, 124910, 25),
    (124912, 124926, 25),
    (124928, 125124, 112),
    (125127, 125142, 112),
    (125184, 125259, 131),
    (125264, 125273, 131),
    (125278, 125279, 131),
    (126065, 126132, 0),
    (126209, 126269, 0),
    (126464, 126467, 6),
    (126469, 126495, 6),
    (126497, 126498, 6),
    (126500, 126500, 6),
    (126503, 126503, 6),
    (126505, 126514, 6),
    (126516, 126519, 6),
    (126521, 126521, 6),
    (126523, 126523, 6),
    (126530, 126530, 6),
    (126535, 126535, 6),
    (126537, 126537, 6),
    (126539, 126539, 6),
    (126541, 126543, 6),
    (126545, 126546, 6),
    (126548, 126548, 6),
    (126551, 126551, 6),
    (126553, 126553, 6),
    (126555, 126555, 6),
    (126557, 126557, 6),
    (126559, 126559, 6),
    (126561, 126562, 6),
    (126564, 126564, 6),
    (126567, 126570, 6),
    (126572, 126578, 6),
    (126580, 126583, 6),
    (126585, 126588, 6),
    (126590, 126590, 6),
    (126592, 126601, 6),
    (126603, 126619, 6),
    (126625, 126627, 6),
    (126629, 126633, 6),
    (126635, 126651, 6),
    (126704, 126705, 6),
    (126976, 127019, 0),
    (127024, 127123, 0),
    (127136, 127150, 0),
    (127153, 127167, 0),
    (127169, 127183, 0),
    (127185, 127221, 0),
    (127232, 127405, 0),
    (127462, 127487, 0),
    (127488, 127488, 32),
    (127489, 127490, 0),
    (127504, 127547, 0),
    (127552, 127560, 0),
    (127568, 127569, 0),
    (127584, 127589, 0),
    (127744, 128727, 0),
    (128732, 128748, 0),
    (128752, 128764, 0),
    (128768, 128886, 0),
    (128891, 128985, 0),
    (128992, 129003, 0),
    (129008, 129008, 0),
    (129024, 129035, 0),
    (129040, 129095, 0),
    (129104, 129113, 0),
    (129120, 129159, 0),
    (129168, 129197, 0),
    (129200, 129211, 0),
    (129216, 129217, 0),
    (129280, 129619, 0),
    (129632, 129645, 0),
    (129648, 129660, 0),
    (129664, 129673, 0),
    (129679, 129734, 0),
    (129742, 129756, 0),
    (129759, 129769, 0),
    (129776, 129784, 0),
    (129792, 129938, 0),
    (129940, 130041, 0),
    (131072, 173791, 35),
    (173824, 177977, 35),
    (177984, 178205, 35),
    (178208, 183969, 35),
    (183984, 191456, 35),
    (191472, 192093, 35),
    (194560, 195101, 35),
    (196608, 201546, 35),
    (201552, 205743, 35),
    (917505, 917505, 0),
    (917536, 917631, 0),
    (917760, 917999, 40),
)


@lru_cache(maxsize=None)
def get_unicode_script(char: str) -> UnicodeScript:
    chr_search = ord(char)
    for range_start, range_end, script_code in UNICODE_RANGE_TO_SCRIPT:
        if range_start <= chr_search <= range_end:
            return UnicodeScript(script_code)
        if chr_search < range_start:
            break
    return UnicodeScript(999)  # default to Unknown
