/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include "sh4lt/c/clogger.h"
#include "sh4lt/c/cfollower.h"
#include "sh4lt/c/cshtype.h"
#include "sh4lt/c/cwriter.h"

static char label[16] = "check-c-shtype";
static char *group_label = sh4lt_shtype_default_group();
static Sh4ltShType shtype = nullptr;


int main () {

  // create a spec for the sh4lt
  shtype = sh4lt_make_shtype("application/x-check-c-shtype", label, group_label);
  assert(shtype);
  sh4lt_shtype_set_str_prop(shtype, "4-letters", "abcd");
  sh4lt_shtype_set_int_prop(shtype, "count", 12345);
  sh4lt_shtype_set_bool_prop(shtype, "is-something", 0);
  sh4lt_shtype_set_str_prop(shtype, "share-ratio", "40/39");
  sh4lt_shtype_set_custom_type(shtype, "share-ratio", "fraction");
  sh4lt_shtype_set_float_prop(shtype, "temperature", 37.2);
  sh4lt_shtype_set_custom_type(shtype, "temperature", "Celsius degree");
  
  // check get functions
  char *tmpmedia = sh4lt_shtype_media(shtype);
  assert(0 == strcmp("application/x-check-c-shtype", tmpmedia));
  free(tmpmedia);
  char *tmplabel = sh4lt_shtype_label(shtype);
  assert(0 == strcmp(label, tmplabel));
  free(tmplabel);
  char *tmpgroup = sh4lt_shtype_group(shtype);
  assert(0 == strcmp(group_label, tmpgroup));
  free(tmpgroup);
  char *lettersprop = sh4lt_shtype_get_str_prop(shtype, "4-letters");
  assert(0 == strcmp("abcd", lettersprop));
  free(lettersprop);
  assert(12345 == sh4lt_shtype_get_int_prop(shtype, "count"));
  assert(0 == sh4lt_shtype_get_bool_prop(shtype, "is-something"));
  char *ratioprop = sh4lt_shtype_get_str_prop(shtype, "share-ratio");
  assert(0 == strcmp("40/39", ratioprop));
  free(ratioprop);
  char *ratiotype = sh4lt_shtype_get_custom_type(shtype, "share-ratio");
  assert(0 == strcmp("fraction", ratiotype));
  free(ratiotype);
  double temperatureprop = sh4lt_shtype_get_float_prop(shtype, "temperature");
  assert(37.2 == temperatureprop);
  char *temperaturetype = sh4lt_shtype_get_custom_type(shtype, "temperature");
  assert(0 == strcmp("Celsius degree", temperaturetype));
  free(temperaturetype);

  // check list of properties
  char *properties[10];
  size_t propsize;
  sh4lt_shtype_get_properties(shtype, properties, &propsize);
  assert(5 == propsize);
  for (size_t i = 0; i< propsize; ++i) {
    assert(0 == strcmp(properties[i], "4-letters") ||
           0 == strcmp(properties[i], "count") ||
           0 == strcmp(properties[i], "is-something") ||
           0 == strcmp(properties[i], "share-ratio") ||
           0 == strcmp(properties[i], "temperature"));
    free(properties[i]);
  }

  // check list of custom types
  char *custom_types[10];
  size_t custom_types_size;
  sh4lt_shtype_get_custom_types(shtype, custom_types, &custom_types_size);
  assert(2 == custom_types_size);
  for (size_t i = 0; i< custom_types_size; ++i) {
    assert(0 == strcmp(custom_types[i], "share-ratio") ||
           0 == strcmp(custom_types[i], "temperature"));
    free(custom_types[i]);
  }

  // check shtype from basic GStreamer caps 
  const char* caps = "video/x-raw, format=(string)I420, width=(int)320, height=(int)240, "
      "framerate=(fraction)30/1, multiview-mode=(string)mono, pixel-aspect-ratio=(fraction)1/1, "
      "interlace-mode=(string)progressive  , label=(string)\"\\(unexpected\\) \\= chars\\,\"";
  Sh4ltShType from_gst = sh4lt_shtype_from_gst_caps(caps, "from gst", "GSstreamer");
  assert(from_gst);
  char *gstmedia = sh4lt_shtype_media(from_gst);
  assert(0 == strcmp("video/x-raw", gstmedia));
  free(gstmedia);
  char *gstlabel = sh4lt_shtype_label(from_gst);
  assert(0 == strcmp("from gst", gstlabel));
  free(gstlabel);
  char *gstgroup = sh4lt_shtype_group(from_gst);
  assert(0 == strcmp("GSstreamer", gstgroup));
  free(gstgroup);
  char *formatprop = sh4lt_shtype_get_str_prop(from_gst, "format");
  assert(0 == strcmp("I420", formatprop));
  free(formatprop);
  sh4lt_delete_shtype(from_gst);
  
  // check shtype to basic GStreamer caps
  char *to_caps=sh4lt_shtype_to_gst_caps(shtype);
  assert(to_caps);
  // assert(0 == strcmp("application/x-check-c-shtype, 4-letters=abcd,
  // count=12345, is-something=false, share-ratio=(fraction)40/39,
  // temperature=(Celsius degree)37.200000", to_caps));
  free(to_caps);

  // check serialization/deserialization
  char *serialized = sh4lt_shtype_serialize(shtype);
  assert(serialized);
  assert(strlen(serialized) > 10); // 10 ensures there is "some" data
  Sh4ltShType deserialized = sh4lt_shtype_deserialize(serialized);
  assert(deserialized);
  char *reserialized = sh4lt_shtype_serialize(deserialized);
  assert(0 == strcmp(serialized, reserialized));
  free(serialized);
  free(deserialized);
  free(reserialized);
  
  // clean before quit
  sh4lt_delete_shtype(shtype);
  free(group_label);
  return 0;
}

