/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#include "./monitor.hpp"

#include "../shtype/shtype.hpp"

namespace sh4lt::monitor {

void Monitor::update_with_new_sh4lt(const std::vector<fs::path>& paths) {
  // prepare the ".created" subtree
  auto created = InfoTree::make();
  info_->graft("created", created);
  info_->tag_as_array("created", true);
  // update the tree
  for (auto& it : paths) {
    // check if the path is not already followed
    auto already_followed = false;
    for (const auto& foll : follower_stats_) {
      if (info_->branch_get_value("ids." + std::to_string(foll->id()) + ".path")
              .as<std::string>() == it) {
        already_followed = true;
        break;
      }
    }
    if (already_followed) continue;
    // create the follower
    follower_stats_.push_back(std::make_unique<FollowerStat>(log_, it));
    const auto& back = follower_stats_.back();
    auto id = std::to_string(back->id());
    info_->vgraft("created." + id, id);
    // save id/group/label/path
    auto idskey = "ids." + id;
    info_->vgraft(idskey + ".group", back->group());
    info_->vgraft(idskey + ".label", back->label());
    info_->vgraft(idskey + ".path", back->path().string());
  }
}

Monitor::Monitor(logger::Logger::ptr log, fs::path path)
    : log_(std::move(log)), dir_watcher_(std::move(path)), info_(InfoTree::make()) {
  // start Followers for currently active Sh4lts
  dir_watcher_.update();
  update_with_new_sh4lt(dir_watcher_.get_currents());
}

Monitor::Monitor(logger::Logger::ptr log) : Monitor(std::move(log), ShType::get_socket_dir()) {}

auto Monitor::update() -> void {
  dir_watcher_.update();
  update_with_new_sh4lt(dir_watcher_.get_created());
}

auto Monitor::get_stats() -> const InfoTree* {
  for (const auto& fstat : follower_stats_) {
    auto id = std::to_string(fstat->id());
    // save informations
    auto infokey = "infos." + id;
    info_->graft(infokey, fstat->stats());
  }
  return info_.get();
}

}  // namespace sh4lt::monitor
