// Copyright (c) 2022-25 Cesanta Software Limited
// All rights reserved
// https://www.st.com/resource/en/reference_manual/dm00124865-stm32f75xxx-and-stm32f74xxx-advanced-arm-based-32-bit-mcus-stmicroelectronics.pdf
// https://www.st.com/resource/en/datasheet/stm32f746zg.pdf

#pragma once

#include <stm32f746xx.h>

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#define BIT(x) (1UL << (x))
#define SETBITS(R, CLEARMASK, SETMASK) (R) = ((R) & ~(CLEARMASK)) | (SETMASK)
#define PIN(bank, num) ((((bank) - 'A') << 8) | (num))
#define PINNO(pin) (pin & 255)
#define PINBANK(pin) (pin >> 8)

#define LED1 PIN('B', 0)   // On-board LED pin (green)
#define LED2 PIN('B', 7)   // On-board LED pin (blue)
#define LED3 PIN('B', 14)  // On-board LED pin (red)

#define LED LED2  // Use blue LED for blinking

/* System clock
5.3.3: APB1 clock <= 54MHz; APB2 clock <= 108MHz
3.3.2, Table 5: configure flash latency (WS) in accordance to clock freq
38.4: The AHB clock frequency must be at least 25 MHz when the Ethernet
controller is used */
enum { APB1_PRE = 5 /* AHB clock / 4 */, APB2_PRE = 4 /* AHB clock / 2 */ };
enum { PLL_HSI = 16, PLL_M = 8, PLL_N = 216, PLL_P = 2 };  // Run at 216 Mhz
#define FLASH_LATENCY 7
#define SYS_FREQUENCY ((PLL_HSI * PLL_N / PLL_M / PLL_P) * 1000000)
#define APB2_FREQUENCY (SYS_FREQUENCY / (BIT(APB2_PRE - 3)))
#define APB1_FREQUENCY (SYS_FREQUENCY / (BIT(APB1_PRE - 3)))

static inline void spin(volatile uint32_t count) {
  while (count--) (void) 0;
}

enum { GPIO_MODE_INPUT, GPIO_MODE_OUTPUT, GPIO_MODE_AF, GPIO_MODE_ANALOG };
enum { GPIO_OTYPE_PUSH_PULL, GPIO_OTYPE_OPEN_DRAIN };
enum { GPIO_SPEED_LOW, GPIO_SPEED_MEDIUM, GPIO_SPEED_HIGH, GPIO_SPEED_INSANE };
enum { GPIO_PULL_NONE, GPIO_PULL_UP, GPIO_PULL_DOWN };
#define GPIO(N) ((GPIO_TypeDef *) (0x40020000 + 0x400 * (N)))

static GPIO_TypeDef *gpio_bank(uint16_t pin) {
  return GPIO(PINBANK(pin));
}
static inline void gpio_toggle(uint16_t pin) {
  GPIO_TypeDef *gpio = gpio_bank(pin);
  uint32_t mask = BIT(PINNO(pin));
  gpio->BSRR = mask << (gpio->ODR & mask ? 16 : 0);
}
static inline int gpio_read(uint16_t pin) {
  return gpio_bank(pin)->IDR & BIT(PINNO(pin)) ? 1 : 0;
}
static inline void gpio_write(uint16_t pin, bool val) {
  GPIO_TypeDef *gpio = gpio_bank(pin);
  gpio->BSRR = BIT(PINNO(pin)) << (val ? 0 : 16);
}
static inline void gpio_init(uint16_t pin, uint8_t mode, uint8_t type,
                             uint8_t speed, uint8_t pull, uint8_t af) {
  GPIO_TypeDef *gpio = gpio_bank(pin);
  uint8_t n = (uint8_t) (PINNO(pin));
  RCC->AHB1ENR |= BIT(PINBANK(pin));  // Enable GPIO clock
  SETBITS(gpio->OTYPER, 1UL << n, ((uint32_t) type) << n);
  SETBITS(gpio->OSPEEDR, 3UL << (n * 2), ((uint32_t) speed) << (n * 2));
  SETBITS(gpio->PUPDR, 3UL << (n * 2), ((uint32_t) pull) << (n * 2));
  SETBITS(gpio->AFR[n >> 3], 15UL << ((n & 7) * 4),
          ((uint32_t) af) << ((n & 7) * 4));
  SETBITS(gpio->MODER, 3UL << (n * 2), ((uint32_t) mode) << (n * 2));
}
static inline void gpio_input(uint16_t pin) {
  gpio_init(pin, GPIO_MODE_INPUT, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_HIGH,
            GPIO_PULL_NONE, 0);
}
static inline void gpio_output(uint16_t pin) {
  gpio_init(pin, GPIO_MODE_OUTPUT, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_HIGH,
            GPIO_PULL_NONE, 0);
}

static inline void irq_exti_attach(uint16_t pin) {
  uint8_t bank = (uint8_t) (PINBANK(pin)), n = (uint8_t) (PINNO(pin));
  SYSCFG->EXTICR[n / 4] &= ~(15UL << ((n % 4) * 4));
  SYSCFG->EXTICR[n / 4] |= (uint32_t) (bank << ((n % 4) * 4));
  EXTI->IMR |= BIT(n);
  EXTI->RTSR |= BIT(n);
  EXTI->FTSR |= BIT(n);
  int irqvec = n < 5 ? 6 + n : n < 10 ? 23 : 40;  // IRQ vector index, 10.1.2
  NVIC_SetPriority(irqvec, 3);
  NVIC_EnableIRQ(irqvec);
}

#ifndef UART_DEBUG
#define UART_DEBUG USART3
#endif

static inline void uart_init(USART_TypeDef *uart, unsigned long baud) {
  uint8_t af = 7;           // Alternate function
  uint16_t rx = 0, tx = 0;  // pins
  uint32_t freq = 0;        // Bus frequency. UART1 is on APB2, rest on APB1

  if (uart == USART1) freq = APB2_FREQUENCY, RCC->APB2ENR |= BIT(4);
  if (uart == USART2) freq = APB1_FREQUENCY, RCC->APB1ENR |= BIT(17);
  if (uart == USART3) freq = APB1_FREQUENCY, RCC->APB1ENR |= BIT(18);

  if (uart == USART1) tx = PIN('A', 9), rx = PIN('A', 10);
  if (uart == USART2) tx = PIN('A', 2), rx = PIN('A', 3);
  if (uart == USART3) tx = PIN('D', 8), rx = PIN('D', 9);

  gpio_init(tx, GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_HIGH, 0, af);
  gpio_init(rx, GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_HIGH, 0, af);
  uart->CR1 = 0;                          // Disable this UART
  uart->BRR = freq / baud;                // Set baud rate
  uart->CR1 |= BIT(0) | BIT(2) | BIT(3);  // Set UE, RE, TE
}
static inline void uart_write_byte(USART_TypeDef *uart, uint8_t byte) {
  uart->TDR = byte;
  while ((uart->ISR & BIT(7)) == 0) spin(1);
}
static inline void uart_write_buf(USART_TypeDef *uart, char *buf, size_t len) {
  while (len-- > 0) uart_write_byte(uart, *(volatile uint8_t *) buf++);
}
static inline int uart_read_ready(USART_TypeDef *uart) {
  return uart->ISR & BIT(5);  // If RXNE bit is set, data is ready
}
static inline uint8_t uart_read_byte(USART_TypeDef *uart) {
  return (uint8_t) (uart->RDR & 255);
}

static inline void rng_init(void) {
  RCC->AHB2ENR |= RCC_AHB2ENR_RNGEN;
  RNG->CR |= RNG_CR_RNGEN;
}
static inline uint32_t rng_read(void) {
  while ((RNG->SR & RNG_SR_DRDY) == 0) (void) 0;
  return RNG->DR;
}

// Hw pull-ups on PHY RXD0,1,DV to enable autonegotiation
static inline void ethernet_init(void) {
  // Initialise Ethernet. Enable MAC GPIO pins, see
  // https://www.farnell.com/datasheets/2014265.pdf section 6.10
  uint16_t pins[] = {PIN('A', 1),  PIN('A', 2),  PIN('A', 7),
                     PIN('B', 13), PIN('C', 1),  PIN('C', 4),
                     PIN('C', 5),  PIN('G', 11), PIN('G', 13)};
  for (size_t i = 0; i < sizeof(pins) / sizeof(pins[0]); i++) {
    gpio_init(pins[i], GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_INSANE,
              GPIO_PULL_NONE, 11);  // 11 is the Ethernet function
  }
  NVIC_EnableIRQ(ETH_IRQn);                // Setup Ethernet IRQ handler
  SYSCFG->PMC |= SYSCFG_PMC_MII_RMII_SEL;  // Use RMII. Goes first!
  RCC->AHB1ENR |=
      RCC_AHB1ENR_ETHMACEN | RCC_AHB1ENR_ETHMACTXEN | RCC_AHB1ENR_ETHMACRXEN;
}

#define UUID ((uint8_t *) UID_BASE)  // Unique 96-bit chip ID. TRM 41.1

// Helper macro for MAC generation
#define GENERATE_LOCALLY_ADMINISTERED_MAC()                        \
  {                                                                \
    2, UUID[0] ^ UUID[1], UUID[2] ^ UUID[3], UUID[4] ^ UUID[5],    \
        UUID[6] ^ UUID[7] ^ UUID[8], UUID[9] ^ UUID[10] ^ UUID[11] \
  }

// DS 2.25: SPI1 works up to 50MHz, SPI3 up to 25MHz
// UM1974 6.14 Table 19: PA7 (SPI1) is shared with the PHY chip, unless JP6 is
// removed or a solder jumper rewired to use PA5; DS 3 Table 12: SPI1 and SPI3
// share pins in CN7 SPIB section
#ifndef CYWSPI
#define CYWSPI SPI1
#endif

// pins valid for SPI1 and SPI3 in CN7, SPIB section
#define DATAPIN PIN('B', 5)
#define MISOPIN PIN('B', 4)  // wire this pin to DATAPIN above
#define CLKPIN PIN('B', 3)
#define CSPIN PIN('A', 4)  // manually controlled

#define PWRPIN PIN('C', 7)

// RM 32.5.7
static inline void spi_setup(SPI_TypeDef *spi, uint8_t div) {
  spi->CR1 = 0;                    // disable before any config change
  spi->CR1 = (div << 3) | BIT(2);          // 32.9.1 MSTR, CPOL=0, CPHA=0
  spi->CR2 = BIT(12) | (7 << 8) | BIT(2);  // 32.9.2 FRXTH, 8-bit frame, SSOE
  spi->CR1 |= BIT(6);                      // enable after fully configured
}
// DS 3 Table 12
static inline void spi_init(SPI_TypeDef *spi, uint8_t div) {
  uint16_t cs = CSPIN, miso = MISOPIN, mosi = DATAPIN, sck = CLKPIN;
  uint8_t af;
  // Bus clock. SPI2,3 on APB1, rest on APB2
  if (spi == SPI1) RCC->APB2ENR |= BIT(12);
  if (spi == SPI3) RCC->APB1ENR |= BIT(15);

  if (spi == SPI1) af = 5;
  if (spi == SPI3) af = 6;

  gpio_output(cs);    // manual control
  gpio_write(cs, 0);  // no power at startup
  gpio_init(miso, GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_INSANE,
            GPIO_PULL_NONE, af);
  gpio_init(mosi, GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_INSANE,
            GPIO_PULL_NONE, af);
  gpio_output(sck);  // RM 32.5.8
  gpio_write(sck, 0);
  gpio_init(sck, GPIO_MODE_AF, GPIO_OTYPE_PUSH_PULL, GPIO_SPEED_INSANE,
            GPIO_PULL_NONE, af);
  spi_setup(spi, div);
}
static inline void spi_done(SPI_TypeDef *spi) {
  while (spi->SR & BIT(7)) spin(1);  // BSY
}
static inline void spi_rx(SPI_TypeDef *spi) {
  GPIO_TypeDef *gpio = gpio_bank(DATAPIN);
  uint8_t n = (uint8_t) (PINNO(DATAPIN));
  while ((spi->SR & BIT(0)) != 0) {
    uint32_t d = spi->DR;
    (void) d;
  }
  // set as input to disable output
  SETBITS(gpio->MODER, 3UL << (n * 2), GPIO_MODE_INPUT << (n * 2));
}
static inline void spi_rxdone(SPI_TypeDef *spi) {
  GPIO_TypeDef *gpio = gpio_bank(DATAPIN);
  uint8_t n = (uint8_t) (PINNO(DATAPIN));
  while (spi->SR & BIT(7)) spin(1);  // BSY
  // set back as AF to enable output
  SETBITS(gpio->MODER, 3UL << (n * 2), GPIO_MODE_AF << (n * 2));
}
static inline void spi_write_byte(SPI_TypeDef *spi, uint8_t byte) {
  while ((spi->SR & BIT(1))== 0) spin(1);   // TXE
  *(volatile uint8_t *) (&spi->DR) = byte;
}
static inline uint8_t spi_read_byte(SPI_TypeDef *spi) {
  *(volatile uint8_t *) (&spi->DR) = 0;     // dummy byte to start transaction
  while ((spi->SR & BIT(0)) == 0) spin(1);  // RXNE
  return *(volatile uint8_t *) (&spi->DR);
}
static inline void spi_select(SPI_TypeDef *spi, bool s) {
  gpio_write(CSPIN, !s);
  (void) spi;
}
