%feature("docstring") OT::DirectionalSampling
"Directional sampling algorithm.

Refer to :ref:`directional_simulation`.

Available constructors:
    DirectionalSampling(*event*)

    DirectionalSampling(*event, rootStrategy, samplingStrategy*)

Parameters
----------
event : :class:`~openturns.RandomVector`
    Event we are computing the probability of.
rootStrategy : :class:`~openturns.RootStrategy`
    Strategy used to evaluate the frontiers of
    the event along each direction in the standard space.

    By default :class:`~openturns.SafeAndSlow`.
samplingStrategy : :class:`~openturns.SamplingStrategy`
    Strategy used to sample directions in the standard space.

    By default :class:`~openturns.RandomDirection`.

Notes
-----
Let :math:`\\inputRV` be a random vector of dimension :math:`\\inputDim`, distributed according to
the measure :math:`\\inputMeasure`, and :math:`\\model: \\Rset^\\inputDim \\rightarrow \\Rset`
be the *limit state function* (where we only wrote the random input parameters). We define the event
:math:`\\cD_f` by:

.. math::

    \\cD_f = \\{\\vect{x} \\in \\Rset^{\\inputDim} \\, | \\,\\model(\\vect{x}) \\leq 0\\}

The directional sampling algorithm estimates the probability of the event :math:`\\cD_f`:

.. math::

    P_f = \\Prob{\\model\\left( \\inputRV \\right) \\leq 0}
        = \\int_{\\Rset^{\\inputDim}} \\mathbf{1}_{\\{\\model(\\vect{x}) \\leq 0 \\}}\\inputMeasure(\\vect{x})\\di{\\vect{x}}


The directional sampling algorithm is a variance reduction sampling method. As it is performed
in the standard space, it uses the iso-probabilistic transformation associated to :math:`\\inputMeasure` (see  :ref:`isoprobabilistic_transformation` to get more details).

Let :math:`T` be the iso-probabilistic transformation: :math:`T(\\vect{X}) = \\vect{Z}`. The
random vector :math:`\\vect{Z}` follows a spherical distribution, which means that it is written
as:

.. math::

    \\vect{Z} = R \\vect{U}

where :math:`R` is a scalar random variable following the radial distribution, :math:`\\vect{U}` is a random
vector that follows the uniform distribution on the unit sphere of dimension :math:`\\inputDim`, 
denoted by :math:`\\cS_\\inputDim`, and :math:`(R, \\vect{U})` are independent.

Then the probability :math:`P_f` can be written as:

.. math::

    P_f & = \\dfrac{1}{|\\cS_\\inputDim|} \\int_{\\cS_\\inputDim} \\left \\{ \\int_{\\Rset^+} 1_{\\left\\{\\model \\circ T^{-1}(r\\vect{u}) \\leq 0 \\right\\}} p_R(r) dr \\right\\} \\, d\\vect{u} \\\\

For a given direction :math:`\\vect{u}`, let :math:`I_{\\vect{u}}` be:

.. math::

   I_{\\vect{u}} = \\{r \\in \\Rset^+ \\, | \\, \\model \\circ T^{-1}(r\\vect{u})\\leq 0 \\}

Then, we have: 

.. math::

    \\int_{\\Rset^+} 1_{\\left\\{\\model \\circ T^{-1}(r\\vect{u}) \\leq 0\\right\\}} p_R(r) dr = \\Prob{R \\in I_{\\vect{u}}}
    
As the radial distribution of :math:`R` is known, we can evaluate exactly  the probability :math:`\\Prob{R \\in I_{\\vect{u}}}` as soon as we know the direction :math:`\\vect{u}`.

The integral with respect to the directions is evaluated using a sampling method, defined by the
class:`~openturns.SamplingStrategy`.

The directional sampling algorithm follows the steps:

- A sample is generated from the uniform distribution on 
  :math:`\\cS_\\inputDim`: from this sample, some directions are defined, according to the
  specified :class:`~openturns.SamplingStrategy`. We denote by :math:`(\\vect{u}_i)` the sample of the directions;

- Along each direction :math:`i`, we find the frontiers of :math:`\\cD_f`, i.e. the solutions
  of :math:`\\model \\circ T^{-1}(r\\vect{u}_i) = 0` according to the :class:`~openturns.RootStrategy`.
  From these solutions, we deduce the subset :math:`I_i \\subset \\Rset^+` that belongs to
  :math:`\\cD_f`;

- For each subset :math:`I_i`, we compute the probability
  :math:`q_i = \\Prob{R \\in I_{\\vect{u}_i}}`.

The probability :math:`P_f` is computed from all the  :math:`q_i` probabilities. If :math:`N` directions have been used, then:

.. math::

    \\widehat{P}_{f,DS} = \\frac{1}{N} \\sum_{i=1}^N q_i

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)

Create the output random vector :math:`Y = \\model(\\inputRV)`:

>>> myFunction = ot.SymbolicFunction(['E', 'F', 'L', 'I'], ['-F*L^3/(3*E*I)'])
>>> myDistribution = ot.Normal([50.0, 1.0, 10.0, 5.0], [1.0]*4, ot.IdentityMatrix(4))
>>> input_RV = ot.RandomVector(myDistribution)
>>> output_RV = ot.CompositeRandomVector(myFunction, input_RV)

Create the event :math:`\\cD_f = \\{\\vect{x} \\in \\Rset^{\\inputDim} \\, | \\,\\model(\\vect{x}) \\leq -3.0\\}`:

>>> threshold = -3.0
>>> event = ot.ThresholdEvent(output_RV, ot.Less(), -3.0)

Create the directional sampling algorithm and the convergence criteria: 

>>> myAlgo = ot.DirectionalSampling(event, ot.MediumSafe(), ot.OrthogonalDirection())
>>> myAlgo.setMaximumOuterSampling(150)
>>> myAlgo.setBlockSize(4)
>>> myAlgo.setMaximumCoefficientOfVariation(0.1)
>>> # Perform the simulation

Run the algorithm and extract the estimated probability:

>>> myAlgo.run()
>>> print('Probability estimate=%.6f' % myAlgo.getResult().getProbabilityEstimate())
Probability estimate=0.169716"

// ---------------------------------------------------------------------

%feature("docstring") OT::DirectionalSampling::getRootStrategy
"Get the root strategy.

Returns
-------
strategy : :class:`~openturns.RootStrategy`
    Root strategy used to evaluate the frontiers of
    the event along each direction in the standard space."

// ---------------------------------------------------------------------

%feature("docstring") OT::DirectionalSampling::setRootStrategy
"Set the root strategy.

Parameters
----------
strategy : :class:`~openturns.RootStrategy`
    Root strategy to evaluate the frontiers of
    the event along each direction in the standard space."

// ---------------------------------------------------------------------

%feature("docstring") OT::DirectionalSampling::getSamplingStrategy
"Get the direction sampling strategy.

Returns
-------
strategy : :class:`~openturns.SamplingStrategy`
    Direction sampling strategy used to sample
    directions in the standard space."

// ---------------------------------------------------------------------

%feature("docstring") OT::DirectionalSampling::setSamplingStrategy
"Set the direction sampling strategy.

Parameters
----------
strategy : :class:`~openturns.SamplingStrategy`
    Direction sampling strategy used
    to sample directions in the standard space."
