%feature("docstring") OT::PhysicalSpaceCrossEntropyImportanceSampling
"Physical Space Cross-Entropy Importance Sampling.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Parameters
----------
event : :class:`~openturns.ThresholdEvent`
    Event we are computing the probability of.

activeParameters : sequence of integers
    List of active parameters indices for the auxiliary distribution.

initialAuxiliaryDistributionParameters : sequence of floats
    Initial value of active parameters of the auxiliary distribution.

bounds : :class:`~openturns.Interval`
    Bounds on the active parameters of the auxiliary distribution.

auxiliaryDistribution : :class:`~openturns.Distribution`
    Auxiliary distribution for the Cross Entropy Importance Sampling algorithm.

quantileLevel : float  :math:`0<quantileLevel<1`
    Intermediate quantile level. The default number can be tweaked with the `CrossEntropyImportanceSampling-DefaultQuantileLevel` key from :class:`~openturns.ResourceMap`.
    
Notes
-----    

The following explanations are given for a failure event defined as :math:`g(\\vect{X})<T` with :math:`\\vect{X}` 
a random vector following a joint PDF :math:`f_\\vect{X}`, :math:`T` a threshold and :math:`g` a limit state function,
without loss of generality.

The Importance Sampling (IS) probability estimate :math:`\\widehat{P}^\\text{IS}` is given by:

.. math::

    \\widehat{P}^\\text{IS}=\\frac{1}{N} \\sum_{i=1}^{N} {\\vect{1}}_{g(\\vect{x}_i)<T} \\frac{f_\\vect{X}(\\vect{x}_i)}{h(\\vect{x}_i)},

with :math:`h` the auxiliary PDF of Importance Sampling, 
:math:`N` the number of independent samples generated with :math:`h` and :math:`{\\vect{1}}_{g(\\vect{x}_i)<T}` the 
indicator function of the failure domain. 

The optimal density :math:`h_{opt}` minimizing the variance of the estimator is defined as:

.. math::

    h_{opt}=\\frac{{\\vect{1}}_{g(\\vect{x})<T}f_\\vect{X}}{P},

with :math:`P` the failure probability which is inaccessible in practice since this probability is the quantity of interest
and unknown.

The Physical Space Cross-Entropy Importance Sampling algorithm [rubinstein2017]_ uses a parametric auxiliary distribution :math:`h_\\vect{\\lambda}` in order to optimize its parameters to compute the probability of interest with accuracy.

It involves an auxiliary optimization problem to find the auxiliary distribution parameters :math:`\\vect{\\lambda}` minimizing the Kullback-Leibler divergence with respect to :math:`h_{opt}`. The following algorithm is used:

1. :math:`k=1`, set the quantile level :math:`\\rho \\in [0,1]` and :math:`h_0 = f_{\\vect{X}}`

2. Generate the population :math:`\\vect{x}_1^{(k)},...,\\vect{x}_N^{(k)}` according to the PDF :math:`h_{k-1}`, apply the 
   function :math:`g` in order to have :math:`y_1^{(k)}=g(\\vect{x}_1^{(k)}),...,y_N^{(k)} = g(\\vect{x}_N^{(k)})`

3. Compute the empirical :math:`\\rho`-quantile :math:`q_k=\\max(T,y^{(k)}_{\\left \\lfloor \\rho N \\right\\rfloor})` using the floor of :math:`\\rho N`.

4. If :math:`q_k>T`: go to Step 7

5. Estimate the auxiliary distribution parameters:

   .. math::

       \\vect{\\lambda}_{k}= \\textrm{argmax}_{\\lambda} \\frac{1}{N} \\sum_{i=1}^{N}{\\vect{1}}_{g(\\vect{x}_i^{(k)}) \\leq q_k} \\frac{f_\\vect{X}(\\vect{x}_i^{(k)})}{h_{\\vect{\\lambda}_{k-1}}(\\vect{x}_i^{(k)})} \\log(h_{\\vect{\\lambda}}(\\vect{x}_i^{(k)}))

6. :math:`k\\leftarrow k+1`, go to Step 2

7. Estimate the probability :math:`\\widehat{P}^{CE}(g(\\vect{\\vect{X}})<T)=\\frac{1}{N}\\displaystyle \\sum_{i=1}^{N} \\vect{1}_{g(\\vect{x}_i^{(k)})<T} \\frac{f_\\vect{X}(\\vect{x}_i^{(k)})}{h_{\\vect{\\lambda}_{k-1}(\\vect{x}_i^{(k)})}}`.


Default solver for the optimization of auxiliary distribution is `LD_LBFGS` from :class:`~openturns.NLopt`.

See also
--------
StandardSpaceCrossEntropyImportanceSampling, CrossEntropyImportanceSampling, CrossEntropyResult


Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution_R = ot.LogNormalMuSigma(300.0, 30.0, 0.0).getDistribution()
>>> distribution_F = ot.Normal(75e3, 5e3)
>>> marginals = [distribution_R, distribution_F]
>>> distribution = ot.JointDistribution(marginals)
>>> # create the model
>>> model = ot.SymbolicFunction(['R', 'F'], ['R-F/(pi_*100.0)'])   
>>> # create the event 
>>> vect = ot.RandomVector(distribution)
>>> G = ot.CompositeRandomVector(model, vect)
>>> event = ot.ThresholdEvent(G, ot.Less(), 0.0)
>>> distribution_marginR = ot.LogNormalMuSigma().getDistribution()
>>> distribution_marginF = ot.Normal()
>>> aux_marginals = [distribution_marginR, distribution_marginF]
>>> aux_distribution = ot.JointDistribution(aux_marginals)
>>> # definition of auxiliary distribution parameters to be optimized
>>> active_parameters = [0,1,2,3,4]
>>> ### WARNING : native parameters of distribution have to be considered
>>> bounds = ot.Interval([3,0.09,0.,50e3,2e3], [7,0.5,0.5,100e3,10e3])# bounds on the active parameters
>>> initial_theta= [5.70,0.1,0.,75e3,5e3] # initial value of the active parameters
>>> my_IS_physical = otexp.PhysicalSpaceCrossEntropyImportanceSampling(event,aux_distribution,active_parameters,initial_theta,bounds,0.3)
>>> my_IS_physical.run()
>>> myResult = my_IS_physical.getResult()"

// ---------------------------------------------------------------------------

%feature("docstring") OT::PhysicalSpaceCrossEntropyImportanceSampling::setOptimizationAlgorithm
"Accessor to the optimization algorithm used to update auxiliary distribution parameters.

Parameters
----------
algorithm : :class:`~openturns.OptimizationAlgorithm`"




// ---------------------------------------------------------------------------

%feature("docstring") OT::PhysicalSpaceCrossEntropyImportanceSampling::getOptimizationAlgorithm
"Accessor to the optimization algorithm used to update auxiliary distribution parameters.

Returns
-------
algorithm : :class:`~openturns.OptimizationAlgorithm`"




