/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { WebFetchTool } from './web-fetch.js';
import { ApprovalMode } from '../config/config.js';
import { ToolConfirmationOutcome } from './tools.js';
import { ToolErrorType } from './tool-error.js';
import * as fetchUtils from '../utils/fetch.js';
const mockGenerateContent = vi.fn();
const mockGetGeminiClient = vi.fn(() => ({
    generateContent: mockGenerateContent,
}));
vi.mock('../utils/fetch.js', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        fetchWithTimeout: vi.fn(),
        isPrivateIp: vi.fn(),
    };
});
describe('WebFetchTool', () => {
    let mockConfig;
    beforeEach(() => {
        vi.resetAllMocks();
        mockConfig = {
            getApprovalMode: vi.fn(),
            setApprovalMode: vi.fn(),
            getProxy: vi.fn(),
            getGeminiClient: mockGetGeminiClient,
        };
    });
    describe('execute', () => {
        it('should throw validation error when url parameter is missing', async () => {
            const tool = new WebFetchTool(mockConfig);
            const params = { prompt: 'no url here' };
            /* @ts-expect-error - we are testing validation */
            expect(() => tool.build(params)).toThrow("params must have required property 'url'");
        });
        it('should return WEB_FETCH_FALLBACK_FAILED on fetch failure', async () => {
            vi.spyOn(fetchUtils, 'isPrivateIp').mockReturnValue(true);
            vi.spyOn(fetchUtils, 'fetchWithTimeout').mockRejectedValue(new Error('fetch failed'));
            const tool = new WebFetchTool(mockConfig);
            const params = { url: 'https://private.ip', prompt: 'summarize this' };
            const invocation = tool.build(params);
            const result = await invocation.execute(new AbortController().signal);
            expect(result.error?.type).toBe(ToolErrorType.WEB_FETCH_FALLBACK_FAILED);
        });
        it('should return WEB_FETCH_FALLBACK_FAILED on API processing failure', async () => {
            vi.spyOn(fetchUtils, 'isPrivateIp').mockReturnValue(false);
            vi.spyOn(fetchUtils, 'fetchWithTimeout').mockResolvedValue({
                ok: true,
                text: () => Promise.resolve('<html><body>Test content</body></html>'),
            });
            mockGenerateContent.mockRejectedValue(new Error('API error'));
            const tool = new WebFetchTool(mockConfig);
            const params = { url: 'https://public.ip', prompt: 'summarize this' };
            const invocation = tool.build(params);
            const result = await invocation.execute(new AbortController().signal);
            expect(result.error?.type).toBe(ToolErrorType.WEB_FETCH_FALLBACK_FAILED);
        });
    });
    describe('shouldConfirmExecute', () => {
        it('should return confirmation details with the correct prompt and urls', async () => {
            const tool = new WebFetchTool(mockConfig);
            const params = {
                url: 'https://example.com',
                prompt: 'summarize this page',
            };
            const invocation = tool.build(params);
            const confirmationDetails = await invocation.shouldConfirmExecute(new AbortController().signal);
            expect(confirmationDetails).toEqual({
                type: 'info',
                title: 'Confirm Web Fetch',
                prompt: 'Fetch content from https://example.com and process with: summarize this page',
                urls: ['https://example.com'],
                onConfirm: expect.any(Function),
            });
        });
        it('should return github urls as-is in confirmation details', async () => {
            const tool = new WebFetchTool(mockConfig);
            const params = {
                url: 'https://github.com/google/gemini-react/blob/main/README.md',
                prompt: 'summarize the README',
            };
            const invocation = tool.build(params);
            const confirmationDetails = await invocation.shouldConfirmExecute(new AbortController().signal);
            expect(confirmationDetails).toEqual({
                type: 'info',
                title: 'Confirm Web Fetch',
                prompt: 'Fetch content from https://github.com/google/gemini-react/blob/main/README.md and process with: summarize the README',
                urls: ['https://github.com/google/gemini-react/blob/main/README.md'],
                onConfirm: expect.any(Function),
            });
        });
        it('should return false if approval mode is AUTO_EDIT', async () => {
            const tool = new WebFetchTool({
                ...mockConfig,
                getApprovalMode: () => ApprovalMode.AUTO_EDIT,
            });
            const params = {
                url: 'https://example.com',
                prompt: 'summarize this page',
            };
            const invocation = tool.build(params);
            const confirmationDetails = await invocation.shouldConfirmExecute(new AbortController().signal);
            expect(confirmationDetails).toBe(false);
        });
        it('should call setApprovalMode when onConfirm is called with ProceedAlways', async () => {
            const setApprovalMode = vi.fn();
            const testConfig = {
                ...mockConfig,
                setApprovalMode,
            };
            const tool = new WebFetchTool(testConfig);
            const params = {
                url: 'https://example.com',
                prompt: 'summarize this page',
            };
            const invocation = tool.build(params);
            const confirmationDetails = await invocation.shouldConfirmExecute(new AbortController().signal);
            if (confirmationDetails &&
                typeof confirmationDetails === 'object' &&
                'onConfirm' in confirmationDetails) {
                await confirmationDetails.onConfirm(ToolConfirmationOutcome.ProceedAlways);
            }
            expect(setApprovalMode).toHaveBeenCalledWith(ApprovalMode.AUTO_EDIT);
        });
    });
});
//# sourceMappingURL=web-fetch.test.js.map