// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Chrome-WebUI-side of the Glic API.
// Communicates with the web client side in
// glic_api_host/glic_api_impl.ts.

// TODO(crbug.com/379677413): Add tests for the API host.

import {loadTimeData} from '//resources/js/load_time_data.js';
import type {BigBuffer} from '//resources/mojo/mojo/public/mojom/base/big_buffer.mojom-webui.js';
import type {BitmapN32} from '//resources/mojo/skia/public/mojom/bitmap.mojom-webui.js';
import {AlphaType} from '//resources/mojo/skia/public/mojom/image_info.mojom-webui.js';
import type {Origin} from '//resources/mojo/url/mojom/origin.mojom-webui.js';
import type {Url} from '//resources/mojo/url/mojom/url.mojom-webui.js';

import type {BrowserProxy} from '../browser_proxy.js';
import type {PanelState as PanelStateMojo, TabData as TabDataMojo, WebClientHandlerInterface, WebClientInterface} from '../glic.mojom-webui.js';
import {GetTabContextErrorReason as MojoGetTabContextErrorReason, WebClientHandlerRemote, WebClientMode, WebClientReceiver} from '../glic.mojom-webui.js';
import type {DraggableArea, PanelState, Screenshot, TabContextOptions, WebPageData} from '../glic_api/glic_api.js';
import {CaptureScreenshotErrorReason, DEFAULT_PDF_SIZE_LIMIT, GetTabContextErrorReason} from '../glic_api/glic_api.js';
import type {GlicAppController} from '../glic_app_controller.js';

import type {PostMessageRequestHandler} from './post_message_transport.js';
import {PostMessageRequestReceiver, PostMessageRequestSender} from './post_message_transport.js';
import type {AnnotatedPageDataPrivate, HostRequestTypes, PdfDocumentDataPrivate, RgbaImage, TabContextResultPrivate, TabDataPrivate, UserProfileInfoPrivate} from './request_types.js';
import {ImageAlphaType, ImageColorType} from './request_types.js';

// Turn everything except void into a promise.
type Promisify<T> = T extends void ? void : Promise<T>;

// A type which the host should implement. This helps verify that
// `HostMessageHandler` is implemented with the correct parameter and return
// types.
type HostMessageHandlerInterface = {
  [Property in keyof HostRequestTypes]:
      // `payload` is the message payload.
      // `responseTransfer` is populated by objects that should be transferred
      // when sending the message.
  (payload: HostRequestTypes[Property]['request'],
   responseTransfer: Transferable[]) =>
      Promisify<HostRequestTypes[Property]['response']>;
};

class WebClientImpl implements WebClientInterface {
  constructor(
      private sender: PostMessageRequestSender,
      private appController: GlicAppController) {}

  notifyPanelOpened(attachedToWindowId: (number|null)): void {
    this.sender.requestNoResponse('glicWebClientNotifyPanelOpened', {
      attachedToWindowId: optionalWindowIdToClient(attachedToWindowId),
    });
  }

  async notifyPanelClosed(): Promise<void> {
    await this.sender.requestWithResponse('glicWebClientNotifyPanelClosed', {});
  }

  async notifyPanelWillOpen(panelState: PanelStateMojo):
      Promise<{webClientMode: WebClientMode}> {
    const result = await this.sender.requestWithResponse(
        'glicWebClientNotifyPanelWillOpen',
        {panelState: panelStateToClient(panelState)});

    // The web client is ready to show, ensure the webview is
    // displayed.
    this.appController.webClientReady();

    return {
      webClientMode:
          (result.openPanelInfo?.startingMode as WebClientMode | undefined) ??
          WebClientMode.kUnknown,
    };
  }

  notifyPanelWasClosed(): Promise<void> {
    return this.sender.requestWithResponse(
        'glicWebClientNotifyPanelWasClosed', {});
  }

  notifyPanelStateChange(panelState: PanelStateMojo) {
    this.sender.requestNoResponse('glicWebClientPanelStateChanged', {
      panelState: panelStateToClient(panelState),
    });
  }

  notifyMicrophonePermissionStateChanged(enabled: boolean): void {
    this.sender.requestNoResponse(
        'glicWebClientNotifyMicrophonePermissionStateChanged', {
          enabled: enabled,
        });
  }

  notifyLocationPermissionStateChanged(enabled: boolean): void {
    this.sender.requestNoResponse(
        'glicWebClientNotifyLocationPermissionStateChanged', {
          enabled: enabled,
        });
  }

  notifyTabContextPermissionStateChanged(enabled: boolean): void {
    this.sender.requestNoResponse(
        'glicWebClientNotifyTabContextPermissionStateChanged', {
          enabled: enabled,
        });
  }

  notifyFocusedTabChanged(focusedTab: (TabDataMojo|null)): void {
    const transfer: Transferable[] = [];
    this.sender.requestNoResponse(
        'glicWebClientNotifyFocusedTabChanged', {
          focusedTab: tabDataToClient(focusedTab, transfer),
        },
        transfer);
  }
  notifyPanelActiveChange(panelActive: boolean): void {
    this.sender.requestNoResponse(
        'glicWebClientNotifyPanelActiveChanged', {panelActive});
  }
}

// Handles all requests to the host.
class HostMessageHandler implements HostMessageHandlerInterface {
  // Undefined until the web client is initialized.
  private receiver: WebClientReceiver|undefined;

  constructor(
      private handler: WebClientHandlerInterface,
      private sender: PostMessageRequestSender,
      private appController: GlicAppController) {}

  destroy() {
    if (this.receiver) {
      this.receiver.$.close();
    }
  }

  async glicBrowserWebClientCreated({}, transfer: Transferable[]) {
    this.receiver = new WebClientReceiver(
        new WebClientImpl(this.sender, this.appController));
    const {initialState} = await this.handler.webClientCreated(
        this.receiver.$.bindNewPipeAndPassRemote());
    const chromeVersion = initialState.chromeVersion.components;

    return {
      panelState: panelStateToClient(initialState.panelState),
      focusedTab: tabDataToClient(initialState.focusedTab, transfer),
      microphonePermissionEnabled: initialState.microphonePermissionEnabled,
      locationPermissionEnabled: initialState.locationPermissionEnabled,
      tabContextPermissionEnabled: initialState.tabContextPermissionEnabled,
      chromeVersion: {
        major: chromeVersion[0] || 0,
        minor: chromeVersion[1] || 0,
        build: chromeVersion[2] || 0,
        patch: chromeVersion[3] || 0,
      },
      panelIsActive: initialState.panelIsActive,
    };
  }

  glicBrowserWebClientInitialized(request: {success: boolean}) {
    // The webview may have been re-shown by webui, having previously been
    // opened by the browser. In that case, show the guest frame again.
    this.appController.showGuest();

    if (request.success) {
      this.handler.webClientInitialized();
    } else {
      this.handler.webClientInitializeFailed();
    }
  }

  async glicBrowserCreateTab(request: {
    url: string,
    options: {openInBackground?: boolean, windowId?: string},
  }) {
    const response = await this.handler.createTab(
        urlFromClient(request.url),
        request.options.openInBackground !== undefined ?
            request.options.openInBackground :
            false,
        optionalWindowIdFromClient(request.options.windowId));
    const tabData = response.tabData;
    if (tabData) {
      return {
        tabData: {
          tabId: tabIdToClient(tabData.tabId),
          windowId: windowIdToClient(tabData.windowId),
          url: urlToClient(tabData.url),
          title: optionalToClient(tabData.title),
        },
      };
    }
    return {};
  }

  glicBrowserOpenGlicSettingsPage() {
    this.handler.openGlicSettingsPage();
  }

  glicBrowserClosePanel() {
    return this.handler.closePanel();
  }

  glicBrowserAttachPanel(): void {
    this.handler.attachPanel();
  }

  glicBrowserDetachPanel(): void {
    this.handler.detachPanel();
  }

  glicBrowserShowProfilePicker() {
    return this.handler.showProfilePicker();
  }

  async glicBrowserGetContextFromFocusedTab(
      request: {options: TabContextOptions},
      transfer: Transferable[]): Promise<{
    tabContextResult?: TabContextResultPrivate,
    error?: GetTabContextErrorReason,
  }> {
    const {
      result: {errorReason, tabContext},
    } = await this.handler.getContextFromFocusedTab({
      includeInnerText: request.options.innerText || false,
      includeViewportScreenshot: request.options.viewportScreenshot || false,
      includePdf: request.options.pdfData || false,
      includeAnnotatedPageContent:
          request.options.annotatedPageContent || false,
      pdfSizeLimit: request.options.pdfSizeLimit === undefined ?
          DEFAULT_PDF_SIZE_LIMIT :
          Math.min(Number.MAX_SAFE_INTEGER, request.options.pdfSizeLimit),
    });
    if (!tabContext) {
      let error = GetTabContextErrorReason.UNKNOWN;
      if (errorReason === MojoGetTabContextErrorReason.kWebContentsChanged) {
        error = GetTabContextErrorReason.WEB_CONTENTS_CHANGED;
      }
      return {error};
    }
    const tabData = tabContext.tabData;
    let favicon: RgbaImage|undefined = undefined;
    if (tabData.favicon) {
      favicon = bitmapN32ToRGBAImage(tabData.favicon);
      if (favicon) {
        transfer.push(favicon.dataRGBA);
      }
    }

    const tabDataResult: TabDataPrivate = {
      tabId: tabIdToClient(tabData.tabId),
      windowId: windowIdToClient(tabData.windowId),
      url: urlToClient(tabData.url),
      title: optionalToClient(tabData.title),
      favicon,
    };
    const webPageData = tabContext.webPageData;
    let webPageDataResult: WebPageData|undefined = undefined;
    if (webPageData) {
      webPageDataResult = {
        mainDocument: {
          origin: originToClient(webPageData.mainDocument.origin),
          innerText: webPageData.mainDocument.innerText,
        },
      };
    }
    const viewportScreenshot = tabContext.viewportScreenshot;
    let viewportScreenshotResult: Screenshot|undefined = undefined;
    if (viewportScreenshot) {
      const screenshotArray = new Uint8Array(viewportScreenshot.data);
      viewportScreenshotResult = {
        widthPixels: viewportScreenshot.widthPixels,
        heightPixels: viewportScreenshot.heightPixels,
        data: screenshotArray.buffer,
        mimeType: viewportScreenshot.mimeType,
        originAnnotations: {},
      };
      transfer.push(screenshotArray.buffer);
    }
    let pdfDocumentData: PdfDocumentDataPrivate|undefined = undefined;
    if (tabContext.pdfDocumentData) {
      const pdfData = tabContext.pdfDocumentData.pdfData ?
          new Uint8Array(tabContext.pdfDocumentData.pdfData).buffer :
          undefined;
      if (pdfData) {
        transfer.push(pdfData);
      }
      pdfDocumentData = {
        origin: originToClient(tabContext.pdfDocumentData.origin),
        pdfSizeLimitExceeded: tabContext.pdfDocumentData.sizeLimitExceeded,
        pdfData,
      };
    }
    let annotatedPageData: AnnotatedPageDataPrivate|undefined = undefined;
    if (tabContext.annotatedPageData) {
      const annotatedPageContent =
          tabContext.annotatedPageData.annotatedPageContent ?
          getArrayBufferFromBigBuffer(
              tabContext.annotatedPageData.annotatedPageContent.smuggled) :
          undefined;
      if (annotatedPageContent) {
        transfer.push(annotatedPageContent);
      }
      annotatedPageData = {annotatedPageContent};
    }

    return {
      tabContextResult: {
        tabData: tabDataResult,
        webPageData: webPageDataResult,
        viewportScreenshot: viewportScreenshotResult,
        pdfDocumentData,
        annotatedPageData,
      },
    };
  }

  async glicBrowserResizeWindow(request: {
    size: {width: number, height: number},
    options?: {durationMs?: number},
  }) {
    this.appController.onGuestResizeRequest(request.size);
    const durationMs = request.options?.durationMs || 0;
    return await this.handler.resizeWidget(request.size, {
      microseconds: BigInt(Math.floor(durationMs * 1000)),
    });
  }

  async glicBrowserCaptureScreenshot(_request: {}, transfer: Transferable[]):
      Promise<{
        screenshot?: Screenshot,
        errorReason?: CaptureScreenshotErrorReason,
      }> {
    const {
      result: {screenshot, errorReason},
    } = await this.handler.captureScreenshot();
    if (!screenshot) {
      const returnedErrorReason =
          (errorReason as CaptureScreenshotErrorReason | undefined) ??
          CaptureScreenshotErrorReason.SCREEN_CAPTURE_FAILED_FOR_UNKNOWN_REASON;
      return {errorReason: returnedErrorReason};
    }
    const screenshotArray = new Uint8Array(screenshot.data);
    transfer.push(screenshotArray.buffer);
    return {
      screenshot: {
        widthPixels: screenshot.widthPixels,
        heightPixels: screenshot.heightPixels,
        data: screenshotArray.buffer,
        mimeType: screenshot.mimeType,
        originAnnotations: {},
      },
    };
  }

  glicBrowserSetWindowDraggableAreas(request: {areas: DraggableArea[]}) {
    return this.handler.setPanelDraggableAreas(request.areas);
  }

  glicBrowserSetMicrophonePermissionState(request: {enabled: boolean}) {
    return this.handler.setMicrophonePermissionState(request.enabled);
  }

  glicBrowserSetLocationPermissionState(request: {enabled: boolean}) {
    return this.handler.setLocationPermissionState(request.enabled);
  }

  glicBrowserSetTabContextPermissionState(request: {enabled: boolean}) {
    return this.handler.setTabContextPermissionState(request.enabled);
  }

  async glicBrowserGetUserProfileInfo(_request: {}, transfer: Transferable[]) {
    const {profileInfo: mojoProfileInfo} =
        await this.handler.getUserProfileInfo();
    if (!mojoProfileInfo) {
      return {};
    }
    const {displayName, email, avatarIcon} = mojoProfileInfo;
    const profileInfo: UserProfileInfoPrivate = {displayName, email};
    if (avatarIcon) {
      profileInfo.avatarIcon = bitmapN32ToRGBAImage(avatarIcon);
      if (profileInfo.avatarIcon) {
        transfer.push(profileInfo.avatarIcon.dataRGBA);
      }
    }
    return {profileInfo};
  }

  glicBrowserRefreshSignInCookies(): Promise<{success: boolean}> {
    return this.handler.syncCookies();
  }

  glicBrowserSetContextAccessIndicator(request: {show: boolean}) {
    this.handler.setContextAccessIndicator(request.show);
  }

  glicBrowserSetAudioDucking(request: {enabled: boolean}) {
    this.handler.setAudioDucking(request.enabled);
  }

  glicBrowserOnUserInputSubmitted(request: {mode: number}) {
    this.handler.onUserInputSubmitted(request.mode);
  }

  glicBrowserOnResponseStarted() {
    this.handler.onResponseStarted();
  }

  glicBrowserOnResponseStopped() {
    this.handler.onResponseStopped();
  }

  glicBrowserOnSessionTerminated() {
    this.handler.onSessionTerminated();
  }

  glicBrowserOnResponseRated(request: {positive: boolean}) {
    this.handler.onResponseRated(request.positive);
  }
}

export class GlicApiHost implements PostMessageRequestHandler {
  private messageHandler: HostMessageHandler;
  private readonly postMessageReceiver: PostMessageRequestReceiver;
  private sender: PostMessageRequestSender;
  private handler: WebClientHandlerRemote;
  private bootstrapPingIntervalId: number|undefined;

  constructor(
      private browserProxy: BrowserProxy, private windowProxy: WindowProxy,
      private embeddedOrigin: string, appController: GlicAppController) {
    this.postMessageReceiver =
        new PostMessageRequestReceiver(embeddedOrigin, windowProxy, this);
    this.sender = new PostMessageRequestSender(windowProxy, embeddedOrigin);
    this.handler = new WebClientHandlerRemote();
    this.browserProxy.handler.createWebClient(
        this.handler.$.bindNewPipeAndPassReceiver());
    this.messageHandler =
        new HostMessageHandler(this.handler, this.sender, appController);

    this.bootstrapPingIntervalId =
        window.setInterval(this.bootstrapPing.bind(this), 50);
    this.bootstrapPing();
  }

  destroy() {
    window.clearInterval(this.bootstrapPingIntervalId);
    this.postMessageReceiver.destroy();
    this.messageHandler.destroy();
    this.sender.destroy();
  }

  // Called when the webview page is loaded.
  contentLoaded() {
    // Send the ping message one more time. At this point, the webview should
    // be able to handle the message, if it hasn't already.
    this.bootstrapPing();
    this.stopBootstrapPing();
  }

  // Sends a message to the webview which is required to initialize the client.
  // Because we don't know when the client will be ready to receive this
  // message, we start sending this every 50ms as soon as navigation commits on
  // the webview, and stop sending this when the page loads, or we receive a
  // request from the client.
  bootstrapPing() {
    if (this.bootstrapPingIntervalId === undefined) {
      return;
    }
    this.windowProxy.postMessage(
        {
          type: 'glic-bootstrap',
          glicApiSource: loadTimeData.getString('glicGuestAPISource'),
        },
        this.embeddedOrigin);
  }

  stopBootstrapPing() {
    if (this.bootstrapPingIntervalId !== undefined) {
      window.clearInterval(this.bootstrapPingIntervalId);
      this.bootstrapPingIntervalId = undefined;
    }
  }

  async openLinkInNewTab(url: string) {
    await this.handler.createTab(urlFromClient(url), false, null);
  }

  // PostMessageRequestHandler implementation.
  async handleRawRequest(type: string, payload: any):
      Promise<{payload: any, transfer: Transferable[]}|undefined> {
    const handlerFunction = (this.messageHandler as any)[type];
    if (typeof handlerFunction !== 'function') {
      console.error(`GlicApiHost: Unknown message type ${type}`);
      return;
    }

    this.stopBootstrapPing();
    const transfer: Transferable[] = [];
    const response =
        await handlerFunction.call(this.messageHandler, payload, transfer);
    if (!response) {
      // Not all request types require a return value.
      return;
    }
    return {payload: response, transfer};
  }
}


// Utility functions for converting from mojom types to message types.
// Summary of changes:
// * Window and tab IDs are sent using int32 in mojo, but made opaque
//   strings for the public API. This allows Chrome to change the ID
//   representation later.
// * Optional types in Mojo use null, but optional types in the public API use
//   undefined.
function windowIdToClient(windowId: number): string {
  return `${windowId}`;
}

function windowIdFromClient(windowId: string): number {
  return parseInt(windowId);
}

function tabIdToClient(tabId: number): string {
  return `${tabId}`;
}

function optionalWindowIdToClient(windowId: number|null): string|undefined {
  if (windowId === null) {
    return undefined;
  }
  return windowIdToClient(windowId);
}

function optionalWindowIdFromClient(windowId: string|undefined): number|null {
  if (windowId === undefined) {
    return null;
  }
  return windowIdFromClient(windowId);
}

function optionalToClient<T>(value: T|null) {
  if (value === null) {
    return undefined;
  }
  return value;
}

function urlToClient(url: Url): string {
  return url.url;
}

function urlFromClient(url: string): Url {
  return {url};
}

function originToClient(origin: Origin): string {
  if (!origin.scheme) {
    return '';
  }
  const originBase = `${origin.scheme}://${origin.host}`;
  if (origin.port) {
    return `${originBase}:${origin.port}`;
  }
  return originBase;
}

function tabDataToClient(tabData: TabDataMojo|null, transfer: Transferable[]):
    TabDataPrivate|undefined {
  if (!tabData) {
    return undefined;
  }

  let favicon: RgbaImage|undefined = undefined;
  if (tabData.favicon) {
    favicon = bitmapN32ToRGBAImage(tabData.favicon);
    if (favicon) {
      transfer.push(favicon.dataRGBA);
    }
  }

  return {
    tabId: tabIdToClient(tabData.tabId),
    windowId: windowIdToClient(tabData.windowId),
    url: urlToClient(tabData.url),
    title: optionalToClient(tabData.title),
    favicon,
    documentMimeType: tabData.documentMimeType,
  };
}

function getArrayBufferFromBigBuffer(bigBuffer: BigBuffer): ArrayBuffer|
    undefined {
  if (bigBuffer.bytes !== undefined) {
    return new Uint8Array(bigBuffer.bytes).buffer;
  }
  return bigBuffer.sharedMemory?.bufferHandle
      .mapBuffer(0, bigBuffer.sharedMemory.size)
      .buffer;
}

function bitmapN32ToRGBAImage(bitmap: BitmapN32): RgbaImage|undefined {
  const bytes = getArrayBufferFromBigBuffer(bitmap.pixelData);
  if (!bytes) {
    return undefined;
  }
  // We don't transmit ColorType over mojo, because it's determined by the
  // endianness of the platform. Chromium only supports little endian, which
  // maps to BGRA. See third_party/skia/include/core/SkColorType.h.
  const colorType = ImageColorType.BGRA;

  return {
    width: bitmap.imageInfo.width,
    height: bitmap.imageInfo.height,
    dataRGBA: bytes,
    alphaType: bitmap.imageInfo.alphaType === AlphaType.PREMUL ?
        ImageAlphaType.PREMUL :
        ImageAlphaType.UNPREMUL,
    colorType,
  };
}

function panelStateToClient(panelState: PanelStateMojo): PanelState {
  return {
    kind: panelState.kind as number,
    windowId: optionalWindowIdToClient(panelState.windowId),
  };
}
