<?php

/*
 * Copyright (C) 2019 Michael Muenz <m.muenz@gmail.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

namespace OPNsense\Maltrail\Api;

use OPNsense\Base\ApiMutableModelControllerBase;
use OPNsense\Firewall\Alias;

class ServerController extends ApiMutableModelControllerBase
{
    protected static $internalModelClass = '\OPNsense\Maltrail\Server';
    protected static $internalModelName = 'server';

    protected function setActionHook()
    {
        $model = $this->getModel();

        // Handle addition/removal of "BlocklistMaltrail"
        if (strval($model->addblocklistalias) == "1") {
            $enabled = strval($model->enabled) == "1";
            $address = trim(strval($model->listenaddress));

            if ($address == "0.0.0.0") {
                $address = "127.0.0.1";
            } elseif ($address == "::") {
                $address = "::1";
            }

            if (strpos($address, ':') !== false) {
                $address = "[$address]";
            }

            $address = $address . ':' . strval($model->listenport);

            self::toggleBlocklistAlias(true, $enabled, $address);
        } else {
            self::toggleBlocklistAlias(false);
        }
    }

    const BLOCKLIST_ALIAS_NAME = "BlocklistMaltrail";
    const BLOCKLIST_ALIAS_DESCRIPTION = "Autogenerated alias for Maltrail's fail2ban feature";
    const BLOCKLIST_ALIAS_UPDATE_FREQ = (1 / 24 / 60 * 5); // (1) is 24 hours | (1 / 24 / 60 * 5) is every 5 minutes

    private static function toggleBlocklistAlias(bool $add, bool $enabled = false, string $addressAndPort = "")
    {
        $model = new Alias();

        // Search for existing alias
        $blocklist = null;
        $blocklistIndex = null;
        foreach ($model->aliases->alias->iterateItems() as $index => $alias) {
            if (strval($alias->name) == self::BLOCKLIST_ALIAS_NAME) {
                $blocklist = $alias;
                $blocklistIndex = $index;
                break;
            }
        }

        // Add, update or remove the alias
        if ($add) {
            if ($blocklist === null) {
                $blocklist = $model->aliases->alias->add();
                $blocklist->name = self::BLOCKLIST_ALIAS_NAME;
                $blocklist->description = self::BLOCKLIST_ALIAS_DESCRIPTION;
                $blocklist->updatefreq = strval(self::BLOCKLIST_ALIAS_UPDATE_FREQ);
            }

            $url = "http://$addressAndPort/fail2ban";
            $enabled = ($enabled ? "1" : "0");

            $needsUpdate = strval($blocklist->type) != "urltable"
                || strval($blocklist->content) != $url
                || strval($blocklist->enabled) != $enabled;

            if ($needsUpdate) {
                $blocklist->type = "urltable";
                $blocklist->content = $url;
                $blocklist->enabled = $enabled;

                $model->serializeToConfig();
            }
        } elseif ($blocklistIndex !== null) {
            if ($model->aliases->alias->del($blocklistIndex)) {
                $model->serializeToConfig();
            }
        }
    }
}
