<?php

/*
 * Copyright (C) 2004-2008 Scott Ullrich <sullrich@gmail.com>
 * Copyright (C) 2008-2009 Ermal Luçi
 * Copyright (C) 2005 Espen Johansen
 * Copyright (C) 2003-2004 Manuel Kasper <mk@neon1.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

require_once("interfaces.lib.inc");
require_once("gwlb.inc");

/*
 * converts a string like "a,b,c,d"
 * into an array like array("a" => "b", "c" => "d")
 */
function explode_assoc($delimiter, $string)
{
    $array = explode($delimiter, $string);
    $result = array();
    $numkeys = floor(count($array) / 2);
    for ($i = 0; $i < $numkeys; $i += 1) {
        $result[$array[$i * 2]] = $array[$i * 2 + 1];
    }
    return $result;
}


function return_hex_ipv4($ipv4)
{
    if (!is_ipaddrv4($ipv4)) {
        return false;
    }

    /* we need the hex form of the interface IPv4 address */
    $ip4arr = explode(".", $ipv4);
    return (sprintf("%02x%02x%02x%02x", $ip4arr[0], $ip4arr[1], $ip4arr[2], $ip4arr[3]));
}

function convert_ipv6_to_128bit($ipv6)
{
    if (!is_ipaddrv6($ipv6)) {
        return false;
    }

    $ip6arr = array();
    $ip6prefix = Net_IPv6::uncompress($ipv6);
    $ip6arr = explode(":", $ip6prefix);
    /* binary presentation of the prefix for all 128 bits. */
    $ip6prefixbin = "";
    foreach ($ip6arr as $element) {
        $ip6prefixbin .= sprintf("%016b", hexdec($element));
    }
    return $ip6prefixbin;
}

function convert_128bit_to_ipv6($ip6bin)
{
    if (strlen($ip6bin) <> 128) {
        return false;
    }

    $ip6arr = array();
    $ip6binarr = array();
    $ip6binarr = str_split($ip6bin, 16);
    foreach ($ip6binarr as $binpart) {
        $ip6arr[] = dechex(bindec($binpart));
    }
    $ip6addr = Net_IPv6::compress(implode(":", $ip6arr));

    return $ip6addr;
}

function match_wireless_interface($int)
{
    $wireless_regex = '/^(ndis|wi|ath|an|ral|ural|iwi|wlan|rum|run|bwn|zyd|mwl|bwi|ipw|iwn|malo|uath|upgt|urtw|wpi)/';
    return preg_match($wireless_regex, $int);
}

function interfaces_bring_up($interface)
{
    if (!$interface) {
        log_error('interfaces_bring_up() was called but no variable defined.');
        log_error('Backtrace: ' . debug_backtrace());
        return;
    }

    legacy_interface_flags($interface, 'up');
}

function does_interface_exist($interface)
{
    $ints = legacy_interface_listget();
    if (empty($interface) || $ints == null || !in_array($interface, $ints)) {
        return false;
    } else {
        return true;
    }
}

function interfaces_loopback_configure($verbose = false)
{
    if ($verbose) {
        echo 'Configuring loopback interface...';
        flush();
    }

    legacy_interface_setaddress('lo0', '127.0.0.1');
    legacy_interface_flags('lo0', 'up');

    if ($verbose) {
        echo "done.\n";
    }
}

function interfaces_vlan_priorities()
{
    $priorities = array();

    $priorities['1'] = gettext('Background (1, lowest)');
    $priorities['0'] = gettext('Best Effort (0, default)');
    $priorities['2'] = gettext('Excellent Effort (2)');
    $priorities['3'] = gettext('Critical Applications (3)');
    $priorities['4'] = gettext('Video (4)');
    $priorities['5'] = gettext('Voice (5)');
    $priorities['6'] = gettext('Internetwork Control (6)');
    $priorities['7'] = gettext('Network Control (7, highest)');

    return $priorities;
}

function interfaces_vlan_configure($realif = '', $verbose = false)
{
    global $config;

    if (!isset($config['vlans']['vlan'])) {
        return;
    }

    if ($verbose) {
        echo 'Configuring VLAN interfaces...';
        flush();
    }

    foreach ($config['vlans']['vlan'] as $vlan) {
        if (empty($vlan['vlanif'])) {
            $vlan['vlanif'] = "{$vlan['if']}_vlan{$vlan['tag']}";
        }
        if (!empty($realif) && $realif != $vlan['vlanif']) {
            continue;
        }
        interface_vlan_configure($vlan);
    }

    if ($verbose) {
        echo "done.\n";
    }
}

function interface_vlan_configure(&$vlan)
{
    global $config;

    if (!is_array($vlan)) {
        log_error('VLAN: called with wrong options. Problems with config!');
        return;
    }
    $if = $vlan['if'];
    $vlanif = empty($vlan['vlanif']) ? "{$if}_vlan{$vlan['tag']}" : $vlan['vlanif'];

    if (empty($if)) {
        log_error('interface_vlan_configure called with if undefined.');
        return;
    }

    interfaces_bring_up($if);

    $members = array($if);

    /* VLAN may be on top of another type of aggregation */
    if (isset($config['laggs']['lagg'])) {
        foreach ($config['laggs']['lagg'] as $lagg) {
            if ($if == $lagg['laggif']) {
                $members = explode(',', $lagg['members']);
                break;
            }
        }
    }

    // disable/enable hardware vlan tags, will be skipped when "Leave default" (option 2) is selected
    if (!isset($config['system']['disablevlanhwfilter']) || $config['system']['disablevlanhwfilter'] == 1) {
        foreach ($members as $member) {
            // probe already selected options
            $intf_details = legacy_interface_details($member);
            $selected_opts = array();
            if (!empty($intf_details['options'])) {
                foreach ($intf_details['options'] as $opt) {
                    if ($opt == 'vlan_hwtagging') {
                        $selected_opts[] = 'vlanhwtag';
                    } else {
                        $selected_opts[] = str_replace('_', '', $opt);
                    }
                }
            }
            // set one tag at a time to avoid driver issues
            foreach (array('vlanhwtag', 'vlanhwfilter', 'vlanhwtso')  as $tag) {
                if (!isset($config['system']['disablevlanhwfilter'])) {
                    if (!in_array($tag, $selected_opts)) {
                        legacy_interface_flags($member, $tag);
                    }
                } elseif (in_array($tag, $selected_opts)) {
                    legacy_interface_flags($member, '-'.$tag);
                }
            }
        }
    }

    if (!empty($vlanif) && does_interface_exist($vlanif)) {
        legacy_interface_destroy($vlanif);
    }

    legacy_interface_create('vlan', $vlanif);
    $pcp = isset($vlan['pcp']) ? $vlan['pcp'] : 0;
    legacy_vlan_tag($vlanif, $if, $vlan['tag'], $pcp);

    interfaces_bring_up($vlanif);
    interfaces_bring_up($if);

    return $vlanif;
}

function interfaces_test_wireless_capability($if, $cap)
{
    $caps = array('hostap' => 'HOSTAP', 'adhoc' => 'IBSS');

    if (!isset($caps[$cap])) {
        return false;
    }

    exec(sprintf('/sbin/ifconfig %s list caps', escapeshellarg($if)), $lines);

    foreach ($lines as $line) {
        if (preg_match("/^drivercaps=.*<.*{$caps[$cap]}.*>$/", $line)) {
            return true;
        }
    }

    return false;
}

function interfaces_create_wireless_clones($verbose = false)
{
    global $config;

    if ($verbose) {
        echo 'Creating wireless clone interfaces...';
        flush();
    }

    $iflist = get_configured_interface_list();

    foreach ($iflist as $if) {
        $realif = $config['interfaces'][$if]['if'];
        if (is_interface_wireless($realif)) {
            interface_wireless_clone(interface_get_wireless_clone($realif), $config['interfaces'][$if]);
        }
    }

    if (isset($config['wireless']['clone'])) {
        foreach ($config['wireless']['clone'] as $clone) {
            if (empty($clone['cloneif'])) {
                continue;
            }
            if (does_interface_exist($clone['cloneif'])) {
                continue;
            }
            interface_wireless_clone($clone['cloneif'], $clone);
        }
    }

    if ($verbose) {
        echo "done.\n";
    }
}

function interfaces_bridge_configure($checkmember = 0, $realif = '')
{
    global $config;

    $i = 0;
    if (isset($config['bridges']['bridged'])) {
        foreach ($config['bridges']['bridged'] as $bridge) {
            if (empty($bridge['bridgeif'])) {
                $bridge['bridgeif'] = "bridge{$i}";
            }
            if (!empty($realif) && $realif != $bridge['bridgeif']) {
                continue;
            }

            if ($checkmember == 1) {
                if (strstr($bridge['if'], "_vip")) {
                    continue;
                }
                $members = explode(',', $bridge['members']);
                foreach ($members as $member) {
                    if (!empty($config['interfaces'][$bridge['if']]) && $config['interfaces'][$bridge['if']]['ipaddrv6'] == "track6") {
                        continue 2;
                    }
                }
            } elseif ($checkmember == 2) {
                if (strstr($bridge['if'], "_vip")) {
                    continue;
                }
                $members = explode(',', $bridge['members']);
                foreach ($members as $member) {
                    if (empty($config['interfaces'][$bridge['if']]) || $config['interfaces'][$bridge['if']]['ipaddrv6'] != "track6") {
                        continue 2;
                    }
                }
            }
            interface_bridge_configure($bridge, $checkmember);
            $i++;
        }
    }
}

function interface_bridge_configure(&$bridge, $checkmember = 0)
{
    global $config;

    if (!is_array($bridge)) {
        return;
    }

    if (empty($bridge['members'])) {
        log_error("No members found on {$bridge['bridgeif']}");
        return;
    }

    $members = explode(',', $bridge['members']);
    if (!count($members)) {
        return;
    }

    /* Calculate smaller mtu and enforce it */
    $mtu = null;
    $foundgif = false;
    foreach ($members as $member) {
        $realif = get_real_interface($member);
        $opts = legacy_interface_stats($realif);
        if (substr($realif, 0, 3) == "gif") {
            $foundgif = true;
            if ($checkmember == 1) {
                return;
            }
            if ($opts['mtu'] <= 1500) {
                continue;
            }
        }
        if (!empty($opts['mtu']) && ($mtu == null || $opts['mtu'] < $mtu)) {
            $mtu = $opts['mtu'];
        }
    }

    if (!$foundgif && $checkmember == 2) {
        return;
    }

    if (file_exists("/var/run/booting") || !empty($bridge['bridgeif'])) {
        legacy_interface_destroy($bridge['bridgeif']);
        legacy_interface_create($bridge['bridgeif']);
        $bridgeif = $bridge['bridgeif'];
    } else {
        $bridgeif = legacy_interface_create('bridge');
        $bridge['bridgeif'] = $bridgeif;
    }

    $checklist = get_configured_interface_list();

    /* Add interfaces to bridge */
    foreach ($members as $member) {
        if (empty($checklist[$member])) {
            continue;
        }
        $realif = get_real_interface($member);
        if (!$realif) {
            log_error('realif not defined in interfaces bridge - up');
            continue;
        }
        /* make sure the parent interface is up */
        legacy_interface_mtu($realif, $mtu);
        configure_interface_hardware($realif);
        interfaces_bring_up($realif);
        legacy_bridge_member($bridge['bridgeif'], $realif);
    }

    if (isset($bridge['enablestp'])) {
        /* Choose spanning tree proto */
        mwexec("/sbin/ifconfig {$bridgeif} proto " . escapeshellarg($bridge['proto']));

        if (!empty($bridge['stp'])) {
            $stpifs = explode(',', $bridge['stp']);
            foreach ($stpifs as $stpif) {
                $realif = get_real_interface($stpif);
                mwexec("/sbin/ifconfig {$bridgeif} stp {$realif}");
            }
        }
        if (!empty($bridge['maxage'])) {
            mwexec("/sbin/ifconfig {$bridgeif} maxage " . escapeshellarg($bridge['maxage']));
        }
        if (!empty($bridge['fwdelay'])) {
            mwexec("/sbin/ifconfig {$bridgeif} fwddelay " . escapeshellarg($bridge['fwdelay']));
        }
        if (!empty($bridge['hellotime'])) {
            mwexec("/sbin/ifconfig {$bridgeif} hellotime " . escapeshellarg($bridge['hellotime']));
        }
        if (!empty($bridge['priority'])) {
            mwexec("/sbin/ifconfig {$bridgeif} priority " . escapeshellarg($bridge['priority']));
        }
        if (!empty($bridge['holdcnt'])) {
            mwexec("/sbin/ifconfig {$bridgeif} holdcnt " . escapeshellarg($bridge['holdcnt']));
        }
        if (!empty($bridge['ifpriority'])) {
            $pconfig = explode(",", $bridge['ifpriority']);
            $ifpriority = array();
            foreach ($pconfig as $cfg) {
                $embcfg = explode_assoc(":", $cfg);
                foreach ($embcfg as $key => $value) {
                    $ifpriority[$key] = $value;
                }
            }
            foreach ($ifpriority as $key => $value) {
                $realif = get_real_interface($key);
                mwexec("/sbin/ifconfig ${bridgeif} ifpriority {$realif} " . escapeshellarg($value));
            }
        }
        if (!empty($bridge['ifpathcost'])) {
            $pconfig = explode(",", $bridge['ifpathcost']);
            $ifpathcost = array();
            foreach ($pconfig as $cfg) {
                $embcfg = explode_assoc(":", $cfg);
                foreach ($embcfg as $key => $value) {
                    $ifpathcost[$key] = $value;
                }
            }
            foreach ($ifpathcost as $key => $value) {
                $realif = get_real_interface($key);
                mwexec("/sbin/ifconfig ${bridgeif} ifpathcost {$realif} " . escapeshellarg($value));
            }
        }
    }

    if ($bridge['maxaddr'] <> "") {
        mwexec("/sbin/ifconfig {$bridgeif} maxaddr " . escapeshellarg($bridge['maxaddr']));
    }
    if ($bridge['timeout'] <> "") {
        mwexec("/sbin/ifconfig {$bridgeif} timeout " . escapeshellarg($bridge['timeout']));
    }
    if (!empty($bridge['span'])) {
        $realif = get_real_interface($bridge['span']);
        mwexec("/sbin/ifconfig {$bridgeif} span {$realif}");
    }
    if (!empty($bridge['edge'])) {
        $edgeifs = explode(',', $bridge['edge']);
        foreach ($edgeifs as $edgeif) {
            $realif = get_real_interface($edgeif);
            mwexec("/sbin/ifconfig {$bridgeif} edge {$realif}");
        }
    }
    if (!empty($bridge['autoedge'])) {
        $edgeifs = explode(',', $bridge['autoedge']);
        foreach ($edgeifs as $edgeif) {
            $realif = get_real_interface($edgeif);
            mwexec("/sbin/ifconfig {$bridgeif} -autoedge {$realif}");
        }
    }
    if (!empty($bridge['ptp'])) {
        $ptpifs = explode(',', $bridge['ptp']);
        foreach ($ptpifs as $ptpif) {
            $realif = get_real_interface($ptpif);
            mwexec("/sbin/ifconfig {$bridgeif} ptp {$realif}");
        }
    }
    if (!empty($bridge['autoptp'])) {
        $ptpifs = explode(',', $bridge['autoptp']);
        foreach ($ptpifs as $ptpif) {
            $realif = get_real_interface($ptpif);
            mwexec("/sbin/ifconfig {$bridgeif} -autoptp {$realif}");
        }
    }
    if (!empty($bridge['static'])) {
        $stickyifs = explode(',', $bridge['static']);
        foreach ($stickyifs as $stickyif) {
            $realif = get_real_interface($stickyif);
            mwexec("/sbin/ifconfig {$bridgeif} sticky {$realif}");
        }
    }
    if (!empty($bridge['private'])) {
        $privateifs = explode(',', $bridge['private']);
        foreach ($privateifs as $privateif) {
            $realif = get_real_interface($privateif);
            mwexec("/sbin/ifconfig {$bridgeif} private {$realif}");
        }
    }

    if ($bridge['bridgeif']) {
        interfaces_bring_up($bridge['bridgeif']);
    } else {
        log_error('bridgeif not defined -- could not bring interface up');
    }
}

function interface_bridge_add_member($bridgeif, $interface)
{
    if (!does_interface_exist($bridgeif) || !does_interface_exist($interface)) {
        return;
    }

    $mtu = legacy_interface_stats($bridgeif)['mtu'];
    $mtum = legacy_interface_stats($interface)['mtu'];

    if ($mtu != $mtum && !(substr($interface, 0, 3) == "gif" && $mtu <= 1500)) {
        legacy_interface_mtu($interface, $mtu);
    }

    configure_interface_hardware($interface);
    interfaces_bring_up($interface);
    legacy_bridge_member($bridgeif, $interface);
}

function interfaces_lagg_configure($realif = '', $verbose = false)
{
    global $config;

    if (!isset($config['laggs']['lagg'])) {
        return;
    }

    if ($verbose) {
        echo 'Configuring LAGG interfaces...';
        flush();
    }

    $i = 0;

    foreach ($config['laggs']['lagg'] as $lagg) {
        if (empty($lagg['laggif'])) {
            $lagg['laggif'] = "lagg{$i}";
        }
        if (!empty($realif) && $realif != $lagg['laggif']) {
            continue;
        }
        interface_lagg_configure($lagg);
        $i++;
    }

    if ($verbose) {
        echo "done.\n";
    }
}

function interface_lagg_configure(&$lagg)
{
    global $config;

    if (!is_array($lagg)) {
        return -1;
    }

    $members = explode(',', $lagg['members']);
    if (!count($members)) {
        return -1;
    }
    $interface_stats = legacy_interfaces_details();

    if (!empty($lagg['laggif'])) {
        $laggif = $lagg['laggif'];
        if (empty($interface_stats[$laggif])) {
            legacy_interface_create($lagg['laggif']);
        } else {
            // Already configured, detach child interfaces before attempting to configure.
            // Prevents vlans to loose parent.
            if (!empty($interface_stats[$laggif]['laggport'])) {
                foreach ($interface_stats[$laggif]['laggport'] as $laggport) {
                    mwexec("/sbin/ifconfig {$laggif} -laggport {$laggport}");
                }
            }
        }
    } else {
        $laggif = legacy_interface_create('lagg');
    }


    // determine mtu value to use, either the provided one for the lagg or smallest of it's children
    $mtu = null;
    if (!empty($lagg['mtu'])) {
        // mtu provided for lagg
        $mtu = $lagg['mtu'];
    } else {
        // min() mtu of children
        foreach ($members as $member) {
            if (!empty($interface_stats[$member]['mtu']) && ($mtu == null || $interface_stats[$member]['mtu'] < $mtu)) {
                $mtu = $interface_stats[$member]['mtu'];
            }
        }
    }

    foreach ($members as $member) {
        if (!empty($interface_stats[$member])) {
            legacy_interface_mtu($member, $mtu);
            configure_interface_hardware($member);
            interfaces_bring_up($member);
            mwexec("/sbin/ifconfig {$laggif} laggport {$member}");
        }
    }

    mwexec("/sbin/ifconfig {$laggif} laggproto " . escapeshellarg($lagg['proto']));
    if (!empty($lagg['lacp_fast_timeout'])) {
        mwexec("/sbin/ifconfig {$laggif} lacp_fast_timeout");
    } elseif ($lagg['proto'] == 'lacp') {
        mwexec("/sbin/ifconfig {$laggif} -lacp_fast_timeout");
    }

    interfaces_bring_up($laggif);

    return $laggif;
}

function interfaces_gre_configure($checkparent = 0, $verbose = false, $realif = '')
{
    global $config;

    if (!isset($config['gres']['gre'])) {
        return;
    }

    if ($verbose) {
        echo "Configuring GRE interfaces ($checkparent)...";
        flush();
    }

    foreach ($config['gres']['gre'] as $i => $gre) {
        if (empty($gre['greif'])) {
            $gre['greif'] = "gre{$i}";
        }
        if (!empty($realif) && $realif != $gre['greif']) {
            continue;
        }

        if ($checkparent == 1) {
            if (strstr($gre['if'], '_vip')) {
                continue;
            }
            if (!empty($config['interfaces'][$gre['if']]) && $config['interfaces'][$gre['if']]['ipaddrv6'] == 'track6') {
                continue;
            }
        } elseif ($checkparent == 2) {
            if (strstr($gre['if'], '_vip')) {
                continue;
            }
            if (empty($config['interfaces'][$gre['if']]) || $config['interfaces'][$gre['if']]['ipaddrv6'] != 'track6') {
                continue;
            }
        }

        interface_gre_configure($gre);
    }

    if ($verbose) {
        echo "done.\n";
    }
}

/* NOTE: $grekey is not used but useful for passing this function to array_walk. */
function interface_gre_configure(&$gre, $grekey = "")
{
    global $config;

    if (!is_array($gre)) {
        return -1;
    }

    $realif = get_real_interface($gre['if']);
    $realifip = get_interface_ip($gre['if']);

    /* make sure the parent interface is up */
    interfaces_bring_up($realif);

    if (file_exists("/var/run/booting") || !(empty($gre['greif']))) {
        legacy_interface_destroy($gre['greif']);
        legacy_interface_create($gre['greif']);
        $greif = $gre['greif'];
    } else {
        $greif = legacy_interface_create('gre');
    }

    /* Do not change the order here for more see gre(4) NOTES section. */
    mwexec("/sbin/ifconfig {$greif} tunnel {$realifip} " . escapeshellarg($gre['remote-addr']));
    if ((is_ipaddrv6($gre['tunnel-local-addr'])) || (is_ipaddrv6($gre['tunnel-remote-addr']))) {
        /* XXX: The prefixlen argument for tunnels of ipv6 is useless since it needs to be 128 as enforced by kernel */
        //mwexec("/sbin/ifconfig {$greif} inet6 " . escapeshellarg($gre['tunnel-local-addr']) . " " . escapeshellarg($gre['tunnel-remote-addr']) . " prefixlen /" . escapeshellarg($gre['tunnel-remote-net']));
        mwexec("/sbin/ifconfig {$greif} inet6 " . escapeshellarg($gre['tunnel-local-addr']) . " " . escapeshellarg($gre['tunnel-remote-addr']) . " prefixlen 128");
    } else {
        mwexec("/sbin/ifconfig {$greif} " . escapeshellarg($gre['tunnel-local-addr']) . " " . escapeshellarg($gre['tunnel-remote-addr']) . " netmask " . gen_subnet_mask($gre['tunnel-remote-net']));
    }

    if (isset($gre['link0'])) {
        legacy_interface_flags($greif, 'link0');
    }
    if (isset($gre['link1'])) {
        legacy_interface_flags($greif, 'link1');
    }
    if (isset($gre['link2'])) {
        legacy_interface_flags($greif, 'link2');
    }

    interfaces_bring_up($greif);

    if (isset($gre['link1']) && $gre['link1']) {
        mwexecf(
            '/sbin/route add %s/%s %s',
            array(
                $gre['tunnel-remote-addr'],
                $gre['tunnel-remote-net'],
                $gre['tunnel-local-addr']
            )
        );
    }

    if (is_ipaddrv4($gre['tunnel-remote-addr'])) {
        file_put_contents("/tmp/{$greif}_router", $gre['tunnel-remote-addr']);
    }
    if (is_ipaddrv6($gre['tunnel-remote-addr'])) {
        file_put_contents("/tmp/{$greif}_routerv6", $gre['tunnel-remote-addr']);
    }

    interfaces_bring_up($greif);

    return $greif;
}

function interfaces_gif_configure($checkparent = 0, $verbose = false, $realif = '')
{
    global $config;

    if (!isset($config['gifs']['gif'])) {
        return;
    }

    if ($verbose) {
        echo "Configuring GIF interfaces ($checkparent)...";
        flush();
    }

    foreach ($config['gifs']['gif'] as $i => $gif) {
        if (empty($gif['gifif'])) {
            $gre['gifif'] = "gif{$i}";
        }
        if (!empty($realif) && $realif != $gif['gifif']) {
            continue;
        }

        if ($checkparent == 1) {
            if (strstr($gif['if'], '_vip')) {
                continue;
            }
            if (!empty($config['interfaces'][$gif['if']]) && $config['interfaces'][$gif['if']]['ipaddrv6'] == 'track6') {
                continue;
            }
        } elseif ($checkparent == 2) {
            if (strstr($gif['if'], '_vip')) {
                continue;
            }
            if (empty($config['interfaces'][$gif['if']]) || $config['interfaces'][$gif['if']]['ipaddrv6'] != 'track6') {
                continue;
            }
        }

        interface_gif_configure($gif);
    }

    if ($verbose) {
        echo "done.\n";
    }
}

/* NOTE: $gifkey is not used but useful for passing this function to array_walk. */
function interface_gif_configure(&$gif, $gifkey = "")
{
    global $config;

    if (!is_array($gif)) {
        return -1;
    }

    $realif = get_real_interface($gif['if']);
    $ipaddr = $gif['ipaddr'];

    if (is_ipaddrv4($gif['remote-addr'])) {
        if (is_ipaddrv4($ipaddr)) {
            $realifip = $ipaddr;
        } else {
            $realifip = get_interface_ip($gif['if']);
        }
        $realifgw = get_interface_gateway($gif['if']);
    } elseif (is_ipaddrv6($gif['remote-addr'])) {
        if (is_ipaddrv6($ipaddr)) {
            $realifip = $ipaddr;
        } else {
            $realifip = get_interface_ipv6($gif['if']);
        }
        $realifgw = get_interface_gateway_v6($gif['if']);
    }
    /* make sure the parent interface is up */
    if ($realif) {
        interfaces_bring_up($realif);
    } else {
        log_error('could not bring realif up -- variable not defined -- interface_gif_configure()');
    }

    if (file_exists("/var/run/booting") || !(empty($gif['gifif']))) {
        legacy_interface_destroy($gif['gifif']);
        legacy_interface_create($gif['gifif']);
        $gifif = $gif['gifif'];
    } else {
        $gifif = legacy_interface_create('gif');
    }

    /* Do not change the order here for more see gif(4) NOTES section. */
    mwexec("/sbin/ifconfig {$gifif} tunnel {$realifip} " . escapeshellarg($gif['remote-addr']));
    if ((is_ipaddrv6($gif['tunnel-local-addr'])) || (is_ipaddrv6($gif['tunnel-remote-addr']))) {
        /* XXX: The prefixlen argument for tunnels of ipv6 is useless since it needs to be 128 as enforced by kernel */
        //mwexec("/sbin/ifconfig {$gifif} inet6 " . escapeshellarg($gif['tunnel-local-addr']) . " " . escapeshellarg($gif['tunnel-remote-addr']) . " prefixlen /" . escapeshellarg($gif['tunnel-remote-net']));
        mwexec("/sbin/ifconfig {$gifif} inet6 " . escapeshellarg($gif['tunnel-local-addr']) . " " . escapeshellarg($gif['tunnel-remote-addr']) . " prefixlen 128");
    } else {
        mwexec("/sbin/ifconfig {$gifif} " . escapeshellarg($gif['tunnel-local-addr']) . " " . escapeshellarg($gif['tunnel-remote-addr']) . " netmask " . gen_subnet_mask($gif['tunnel-remote-net']));
    }

    if (isset($gif['link0'])) {
        legacy_interface_flags($gifif, 'link0');
    }
    if (isset($gif['link1'])) {
        legacy_interface_flags($gifif, 'link1');
    }

    interfaces_bring_up($gifif);

    $iflist = get_configured_interface_list();
    foreach ($iflist as $ifname) {
        if ($config['interfaces'][$ifname]['if'] == $gifif) {
            if (get_interface_gateway($ifname) || get_interface_gateway_v6($ifname)) {
                system_routing_configure(false, $ifname);
                break;
            }
        }
    }

    if (is_ipaddrv4($gif['tunnel-remote-addr'])) {
        file_put_contents("/tmp/{$gifif}_router", $gif['tunnel-remote-addr']);
    }
    if (is_ipaddrv6($gif['tunnel-remote-addr'])) {
        file_put_contents("/tmp/{$gifif}_routerv6", $gif['tunnel-remote-addr']);
    }

    system_host_route($gif['remote-addr'], $realifgw);
    interfaces_bring_up($gifif);

    return $gifif;
}

function interfaces_configure($verbose = false)
{
    global $config;

    interfaces_loopback_configure($verbose);
    interfaces_create_wireless_clones($verbose);
    interfaces_lagg_configure('', $verbose);
    interfaces_vlan_configure('', $verbose);

    $iflist = get_configured_interface_with_descr();
    $delayed_list = array();
    $bridge_list = array();
    $track6_list = array();

    /* This is needed to speedup interfaces on bootup. */
    $reload = !file_exists('/var/run/booting');

    foreach ($iflist as $if => $ifname) {
        $realif = $config['interfaces'][$if]['if'];
        if (strstr($realif, "bridge")) {
            $bridge_list[$if] = $ifname;
        } elseif (strstr($realif, 'gre') || strstr($realif, 'gif') || strstr($realif, 'ovpn')) {
            $delayed_list[$if] = $ifname;
        } elseif (!empty($config['interfaces'][$if]['ipaddrv6']) && $config['interfaces'][$if]['ipaddrv6'] == 'track6') {
            $track6_list[$if] = $ifname;
        } else {
            interface_configure($if, $reload, false, $verbose);
        }
    }

    /*
     * NOTE: The following function parameter consists of
     *  1 - Do not load gre/gif/bridge with parent/member as vip
     *  2 - Do load gre/gif/bridge with parent/member as vip
     */

    interfaces_gre_configure(1, $verbose);
    interfaces_gif_configure(1, $verbose);
    interfaces_bridge_configure(1);

    foreach ($track6_list as $if => $ifname) {
        interface_configure($if, $reload, false, $verbose);
    }

    interfaces_vips_configure('', $verbose);
    interfaces_gre_configure(2, $verbose);
    interfaces_gif_configure(2, $verbose);

    foreach ($delayed_list as $if => $ifname) {
        interface_configure($if, $reload, false, $verbose);
    }

    interfaces_bridge_configure(2);

    foreach ($bridge_list as $if => $ifname) {
        interface_configure($if, $reload, false, $verbose);
    }

    interfaces_group_setup();

    if ($reload) {
        system_routing_configure($verbose);
        ipsec_configure_do($verbose);
        plugins_configure('dns', $verbose);
        services_dhcpd_configure('all', array(), $verbose);
    }
}

function interface_vip_bring_down($vip)
{
    $vipif = get_real_interface($vip['interface']);
    switch ($vip['mode']) {
        case 'proxyarp':
            killbypid("/var/run/choparp_{$vipif}.pid");
            break;
        case 'ipalias':
            if (does_interface_exist($vipif)) {
                if (is_ipaddrv6($vip['subnet'])) {
                    mwexec("/sbin/ifconfig {$vipif} inet6 " . escapeshellarg($vip['subnet']) . " -alias");
                } else {
                    legacy_interface_deladdress($vipif, $vip['subnet']);
                }
            }
            break;
        case 'carp':
            if (does_interface_exist($vipif)) {
                legacy_interface_deladdress($vipif, $vip['subnet']);
            }
            break;
    }
}

function interface_bring_down($interface = "wan", $ifacecfg = false)
{
    global $config;

    if (!isset($config['interfaces'][$interface]) || ($ifacecfg !== false && !is_array($ifacecfg))) {
        // exit on invalid input
        return;
    }

    if ($ifacecfg === false) {
        $realif = get_real_interface($interface);
        $realifv6 = get_real_interface($interface, "inet6");
        $ifcfg = $config['interfaces'][$interface];
        $ppps = isset($config['ppps']['ppp']) ? $config['ppps']['ppp'] : array();
    } else {
        $ifcfg = $ifacecfg['ifcfg'];
        $ppps = $ifacecfg['ppps'];
        /* When $ifacecfg is passed, it should contain the original interfaces */
        $realif = $ifacecfg['ifcfg']['realif'];
        $realifv6 = $ifacecfg['ifcfg']['realifv6'];
    }
    $ipaddrv6 = empty($ifcfg['ipaddrv6']) ? null : $ifcfg['ipaddrv6'];
    if (!in_array($interface, array('openvpn', 'ppp', 'ipsec', 'enc0'))) {
        /**
         * NOTE: The $realifv6 is needed when WANv4 is type PPP and v6 is DHCP and the option v6 from v4 is used.
         * XXX: we kept this in place due to dropping the $realv6iface parameter from get_real_interface(),
        */
        if (!in_array($ipaddrv6, array('6rd', '6to4', 'pppoe', 'ppp', 'l2tp', 'pptp'))) {
            $ipaddr = empty($ifcfg['ipaddr']) ? null : $ifcfg['ipaddr'];
            if (in_array($ipaddr, array('pppoe', 'ppp', 'l2tp', 'pptp'))) {
                foreach ($ppps as $ppp) {
                    if ($ifcfg['if'] == $ppp['if']) {
                        $ports = explode(',', $ppp['ports']);
                        foreach ($ports as $pid => $parent_if) {
                            $realifv6 = $parent_if;
                            break;
                        }
                    }
                }
            }
        }
    }

    // force VIPs to go down
    if (isset($config['virtualip']['vip'])) {
        foreach ($config['virtualip']['vip'] as $vip) {
            if ($vip['interface'] == $interface) {
                interface_vip_bring_down($vip);
            }
        }
    }

    switch ($ifcfg['ipaddr']) {
        case "ppp":
        case "pppoe":
        case "pptp":
        case "l2tp":
            if (!empty($ppps)) {
                foreach ($ppps as $ppp) {
                    if ($ifcfg['if'] == $ppp['if']) {
                        if (isset($ppp['ondemand']) && $ifacecfg === false) {
                            configd_run("interface reconfigure {$interface}");
                            break;
                        }
                        if (isvalidpid("/var/run/{$ppp['type']}_{$interface}.pid")) {
                            killbypid("/var/run/{$ppp['type']}_{$interface}.pid");
                            sleep(2);
                        }
                        @unlink("/var/etc/mpd_{$interface}.conf");
                        break;
                    }
                }
                return;
            }
            break;
        case "dhcp":
            kill_dhclient_process($realif);
            @unlink("/var/etc/dhclient_{$interface}.conf");
            if (does_interface_exist("$realif")) {
                mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " delete", true);
                if ($destroy) {
                    legacy_interface_flags($realif, 'down');
                }
                mwexec("/usr/sbin/arp -d -i " . escapeshellarg($realif) . " -a");
            }
            break;
        default:
            if (does_interface_exist("$realif")) {
                mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " delete", true);
                if ($destroy) {
                    legacy_interface_flags($realif, 'down');
                }
                mwexec("/usr/sbin/arp -d -i " . escapeshellarg($realif) . " -a");
            }
            break;
    }

    $track6 = array();
    switch ($ipaddrv6) {
        case "slaac":
        case "dhcp6":
            killbypid("/var/run/dhcp6c_{$realif}.pid", 'TERM', true);
            if (does_interface_exist($realifv6)) {
                $ip6 = find_interface_ipv6($realifv6);
                if (is_ipaddrv6($ip6) && $ip6 != "::") {
                    mwexec("/sbin/ifconfig " . escapeshellarg($realifv6) . " inet6 {$ip6} delete", true);
                }
                if ($destroy) {
                    legacy_interface_flags($realif, 'down');
                }
                mwexec("/usr/sbin/arp -d -i " . escapeshellarg($realif) . " -a");
            }
            $track6 = link_interface_to_track6($interface);
            break;
        case "6rd":
        case "6to4":
            $realif = "{$interface}_stf";
            if (does_interface_exist("$realif")) {
                $ip6 = get_interface_ipv6($interface);
                if (is_ipaddrv6($ip6)) {
                    mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 {$ip6} delete", true);
                }
                if ($ifacecfg !== false) {
                    legacy_interface_flags($realif, 'down');
                }
            }
            $track6 = link_interface_to_track6($interface);
            break;
        default:
            if (does_interface_exist("$realif")) {
                $ip6 = get_interface_ipv6($interface);
                if (is_ipaddrv6($ip6)) {
                    mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 {$ip6} delete", true);
                }
                if (!empty($ifcfg['ipaddrv6']) && is_ipaddrv6($ifcfg['ipaddrv6'])) {
                    mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 {$ifcfg['ipaddrv6']} delete", true);
                }
                if ($destroy) {
                    legacy_interface_flags($realif, 'down');
                }
                mwexec("/usr/sbin/arp -d -i " . escapeshellarg($realif) . " -a");
            }
            $track6 = link_interface_to_track6($interface);
            break;
    }

    if (!empty($track6)) {
        /* Bring down radvd and dhcp6 on these interfaces */
        services_dhcpd_configure('inet6', $track6);
    }

    if (file_exists("/tmp/{$realif}_router")) {
        $old_router = trim(file_get_contents("/tmp/{$realif}_router"));
        if (!empty($old_router)) {
            log_error("Clearing states to old gateway {$old_router}.");
            mwexec("/sbin/pfctl -i " . escapeshellarg($realif) . " -Fs");
        }
    }

    @unlink("/var/db/{$interface}_ip");
    @unlink("/var/db/{$interface}_ipv6");
    @unlink("/var/db/{$interface}_cacheip");
    @unlink("/var/db/{$interface}_cacheipv6");
    @unlink("/var/etc/nameserver_{$realif}");
    @unlink("/var/etc/nameserver_v6{$realif}");
    @unlink("/var/etc/searchdomain_{$realif}");
    @unlink("/var/etc/searchdomain_v6{$realif}");
    @unlink("/tmp/{$realif}up");
    @unlink("/tmp/{$realif}upv6");
    @unlink("/tmp/{$realif}_router");
    @unlink("/tmp/{$realif}_routerv6");

    /* hostapd and wpa_supplicant do not need to be running when the interface is down.
     * They will also use 100% CPU if running after the wireless clone gets deleted. */
    if (!empty($ifcfg['wireless'])) {
        kill_hostapd($realif);
        mwexec(kill_wpasupplicant($realif));
    }

    return;
}

function interfaces_ptpid_used($ptpid)
{
    global $config;

    if (isset($config['ppps']['ppp'])) {
        foreach ($config['ppps']['ppp'] as & $settings) {
            if ($ptpid == $settings['ptpid']) {
                return true;
            }
        }
    }

    return false;
}

function interfaces_ptpid_next()
{

    $ptpid = 0;
    while (interfaces_ptpid_used($ptpid)) {
        $ptpid++;
    }

    return $ptpid;
}

/*
 * This function can configure PPPoE, MLPPP (PPPoE), PPTP.
 * It writes the mpd config file to /var/etc every time the link is opened.
 */
function interface_ppps_configure($interface)
{
    global $config;

    /* Return for unassigned interfaces. This is a minimum requirement. */
    if (empty($config['interfaces'][$interface])) {
        return 0;
    }
    $ifcfg = $config['interfaces'][$interface];
    if (!isset($ifcfg['enable'])) {
        return 0;
    }

    // mpd5 requires a /var/spool/lock directory for PPP modem links.
    @mkdir("/var/spool/lock", 0777, true);

    // mpd5 modem chat script expected in the same directory as the mpd_xxx.conf files
    if (!file_exists('/var/etc/mpd.script')) {
        symlink('/usr/local/sbin/mpd.script', '/var/etc/mpd.script');
    }

    if (isset($config['ppps']['ppp'])) {
        foreach ($config['ppps']['ppp'] as $pppid => $ppp) {
            if ($ifcfg['if'] == $ppp['if']) {
                break;
            }
        }
    }
    if (!isset($ppp) || $ifcfg['if'] != $ppp['if']) {
        log_error("Can't find PPP config for {$ifcfg['if']} in interface_ppps_configure().");
        return 0;
    }

    if (file_exists("/var/run/booting")) {
        // Do not re-configure the interface if we are booting and it's already been started
        if (isvalidpid("/var/run/{$ppp['type']}_{$interface}.pid")) {
            return 0;
        }
    }

    $ports = explode(',', $ppp['ports']);
    if ($ppp['type'] != "ppp") {
        foreach ($ports as $pid => $port) {
            $ports[$pid] = get_real_interface($port);
            if (empty($ports[$pid])) {
                return 0;
            }
        }
    }
    $localips = isset($ppp['localip']) ? explode(',', $ppp['localip']) : array();
    $gateways = isset($ppp['gateway']) ? explode(',', $ppp['gateway']) : array();
    $subnets = isset($ppp['subnet']) ? explode(',', $ppp['subnet']) : array();

    /* We bring up the parent interface first because if DHCP is configured on the parent we need
     * to obtain an address first so we can write it in the mpd .conf file for PPTP and L2TP configs
     */
    foreach ($ports as $pid => $port) {
        switch ($ppp['type']) {
            case "pppoe":
                /* Bring the parent interface up */
                interfaces_bring_up($port);
                /* Enable setautosrc to automatically change mac address if parent interface's changes */
                mwexecf('/usr/sbin/ngctl msg %s: setautosrc 1', array($port));
                break;
            case "pptp":
            case "l2tp":
                /* configure interface */
                if (is_ipaddr($localips[$pid])) {
                    // Manually configure interface IP/subnet
                    legacy_interface_setaddress($port, "{$localips[$pid]}/{$subnets[$pid]}");
                    interfaces_bring_up($port);
                } elseif (empty($localips[$pid])) {
                    $localips[$pid] = get_interface_ip($port); // try to get the interface IP from the port
                }

                if (!is_ipaddr($localips[$pid])) {
                    log_error("Could not get a Local IP address for PPTP/L2TP link on {$port}. Using 0.0.0.0!");
                    $localips[$pid] = "0.0.0.0";
                }
                if (!is_ipaddr($gateways[$pid])) {
                    log_error("Could not get a Remote IP address for PPTP/L2TP link on {$port}.");
                    return 0;
                }
                break;
            case "ppp":
                if (!file_exists("{$port}")) {
                    log_error("Device {$port} does not exist. PPP link cannot start without the modem device.");
                    return 0;
                }
                break;
            default:
                log_error("Unkown {$ppp['type']} configured as ppp interface.");
                break;
        }
    }

    // Construct the mpd.conf file
    $mpdconf = <<<EOD
startup:
  # configure the console
  set console close
  # configure the web server
  set web close

default:
{$ppp['type']}client:
  create bundle static {$interface}
  set bundle enable ipv6cp
  set iface name {$ifcfg['if']}

EOD;
    $setdefaultgw = false;
    $founddefaultgw = false;
    if (isset($config['gateways']['gateway_item'])) {
        foreach ($config['gateways']['gateway_item'] as $gateway) {
            if ($interface == $gateway['interface'] && isset($gateway['defaultgw'])) {
                $setdefaultgw = true;
                break;
            } elseif (isset($gateway['defaultgw']) && !empty($gateway['interface'])) {
                $founddefaultgw = true;
                break;
            }
        }
    }

    if (!$founddefaultgw || $setdefaultgw) {
        $mpdconf .= "  set iface route default\n";
    }
    if (isset($ppp['ondemand'])) {
        $mpdconf .= "  set iface enable on-demand\n";
    } else {
        $mpdconf .= "  set iface disable on-demand\n";
    }
    if (!isset($ppp['idletimeout'])) {
        $mpdconf .= "  set iface idle 0\n";
    } else {
        $mpdconf .= "  set iface idle {$ppp['idletimeout']}\n";
    }

    if (isset($ppp['ondemand'])) {
        $mpdconf .= "  set iface addrs 10.10.1.1 10.10.1.2\n";
    }

    if (isset($ppp['tcpmssfix'])) {
        $mpdconf .= "  set iface disable tcpmssfix\n";
    } else {
        $mpdconf .= "  set iface enable tcpmssfix\n";
    }

    $mpdconf .= "  set iface up-script /usr/local/sbin/ppp-linkup\n";
    $mpdconf .= "  set iface down-script /usr/local/sbin/ppp-linkdown\n";
    if ($ppp['type'] == "ppp") {
        if (is_ipaddr($ppp['localip'])) {
            $localip = $ppp['localip'];
        } else {
            $localip = '0.0.0.0';
        }
        if (is_ipaddr($ppp['gateway'])) {
            $gateway = $ppp['gateway'];
        } else {
            $gateway = "10.64.64.{$pppid}";
        }
        $mpdconf .= "  set ipcp ranges {$localip}/0 {$gateway}/0\n";
    } else {
        $mpdconf .= "  set ipcp ranges 0.0.0.0/0 0.0.0.0/0\n";
    }

    if (isset($ppp['vjcomp'])) {
        $mpdconf .= "  set ipcp no vjcomp\n";
    }

    if (isset($config['system']['dnsallowoverride'])) {
        $mpdconf .= "  set ipcp enable req-pri-dns\n";
        $mpdconf .= "  set ipcp enable req-sec-dns\n";
    }
    foreach ($ports as $pid => $port) {
        $mpdconf_arr = array();
        $port = get_real_interface($port);
        if ($ppp['type'] == "ppp") {
            $mpdconf_arr[] = "create link static {$interface}_link{$pid} modem";
        } else {
            $mpdconf_arr[] = "create link static {$interface}_link{$pid} {$ppp['type']}";
        }
        $mpdconf_arr[] = "set link action bundle {$interface}";
        if (count($ports) > 1) {
            $mpdconf_arr[] = "set link enable multilink";
        } else {
            $mpdconf_arr[] = "set link disable multilink";
        }
        $mpdconf_arr[] = "set link keep-alive 10 60";
        $mpdconf_arr[] = "set link max-redial 0";
        if (isset($ppp['shortseq'])) {
            $mpdconf_arr[] = "set link no shortseq";
        }
        if (isset($ppp['acfcomp'])) {
            $mpdconf_arr[] = "set link no acfcomp";
        }
        if (isset($ppp['protocomp'])) {
            $mpdconf_arr[] = "set link no protocomp";
        }
        $mpdconf_arr[] = "set link disable chap pap";
        $mpdconf_arr[] = "set link accept chap pap eap";
        $mpdconf_arr[] = "set link disable incoming";
        $bandwidths = !empty($ppp['bandwidth']) ? explode(',', $ppp['bandwidth']) : null;
        if (!empty($bandwidths[$pid])) {
            $mpdconf_arr[] = "set link bandwidth {$bandwidths[$pid]}";
        }

        $mtus = !empty($ppp['mtu']) ? explode(',', $ppp['mtu']) : null;
        if (empty($mtus[$pid])) {
            $mtus[$pid] = !empty($ifcfg['mtu']) ? intval($ifcfg['mtu']) : 1500;
        }
        if ($ppp['type'] == "pppoe" && $mtus[$pid] > 1500 ) {
            // use pppoe max-payload if mtu we're about to set > 1492
            $mpdconf_arr[] = "set pppoe max-payload " . ($mtus[$pid]-8);
        } else {
            $mpdconf_arr[] = "set link mtu " . ($mtus[$pid]-8);
        }
        $mrus = !empty($ppp['mtu']) ? explode(',', $ppp['mru']) : null;
        if (!empty($mrus[$pid])) {
            $mpdconf_arr[] = "set link mru {$mrus[$pid]}";
        }
        $mrrus = !empty($ppp['mrru']) ? explode(',', $ppp['mrru']) : null;
        if (!empty($mrrus[$pid])) {
            $mpdconf_arr[] = "set link mrru {$mrrus[$pid]}";
        }

        if (empty($ppp['username']) && $ppp['type'] == "ppp") {
            $mpdconf_arr[] = "set auth authname \"user\"";
        } else {
            $mpdconf_arr[] = "set auth authname \"{$ppp['username']}\"";
        }
        if (empty($ppp['password']) && $ppp['type'] == "ppp") {
            $mpdconf_arr[] = "set auth password " .  base64_decode('none');
        } else {
            $mpdconf_arr[] = "set auth password " . base64_decode($ppp['password']);
        }

        if ($ppp['type'] == "ppp") {
            // ppp, modem connections
            $mpdconf_arr[] = "set modem device {$ppp['ports']}";
            $mpdconf_arr[] = "set modem script DialPeer";
            $mpdconf_arr[] = "set modem idle-script Ringback";
            $mpdconf_arr[] = "set modem watch -cd";
            $mpdconf_arr[] = "set modem var \$DialPrefix \"DT\"";
            $mpdconf_arr[] = "set modem var \$Telephone \"{$ppp['phone']}\"";
            if (isset($ppp['connect-timeout'])) {
                $mpdconf_arr[] = "set modem var \$ConnectTimeout \"{$ppp['connect-timeout']}\"";
            }
            if (isset($ppp['initstr'])) {
                $initstr = base64_decode($ppp['initstr']);
                $mpdconf_arr[] = "set modem var \$InitString \"{$initstr}\"";
            }
            if (isset($ppp['simpin'])) {
                $mpdconf_arr[] = "set modem var \$SimPin \"{$ppp['simpin']}\"";
                if (!empty($ppp['pin-wait'])) {
                    $mpdconf_arr[] = "set modem var \$PinWait \"{$ppp['pin-wait']}\"";
                } else {
                    $mpdconf_arr[] = "set modem var \$PinWait \"0\"";
                }
            }
            if (isset($ppp['apn'])) {
                $mpdconf_arr[] = "set modem var \$APN \"{$ppp['apn']}\"";
                if (empty($ppp['apnum'])) {
                    $mpdconf_arr[] = "set modem var \$APNum \"1\"";
                } else {
                    $mpdconf_arr[] = "set modem var \$APNum \"{$ppp['apnum']}\"";
                }
            }
        } elseif ($ppp['type'] == "pppoe") {
            $provider = isset($ppp['provider']) ? $ppp['provider'] : "";
            $hostuniq = '';
            if (!empty($ppp['hostuniq'])) {
                $hostuniq = '0x' . strtolower(array_shift(unpack('H*', $ppp['hostuniq']))) . '|';
            }
            $mpdconf_arr[] = "set pppoe service \"{$hostuniq}{$provider}\"";
            $mpdconf_arr[] = "set pppoe iface {$port}";
        } elseif ($ppp['type'] == "pptp" || $ppp['type'] == "l2tp") {
            $mpdconf_arr[] = "set {$ppp['type']} self {$localips[$pid]}";
            $mpdconf_arr[] = "set {$ppp['type']} peer {$gateways[$pid]}";
        }

        foreach ($mpdconf_arr as $mpdconf_opt) {
            $mpdconf .= "  " . $mpdconf_opt . "\n";
        }

        $mpdconf .= "\topen\n";
    }


    /* Generate mpd.conf. If mpd_[interface].conf exists in the conf path, then link to it instead of generating a fresh conf file. */
    if (file_exists("/conf/mpd_{$interface}.conf")) {
        @symlink("/conf/mpd_{$interface}.conf", "/var/etc/mpd_{$interface}.conf");
    } else {
        $fd = fopen("/var/etc/mpd_{$interface}.conf", "w");
        if (!$fd) {
            log_error("Error: cannot open mpd_{$interface}.conf in interface_ppps_configure().");
            return 0;
        }
        // Write out mpd_ppp.conf
        fwrite($fd, $mpdconf);
        fclose($fd);
        unset($mpdconf);
    }

    // Create the uptime log if requested and if it doesn't exist already, or delete it if it is no longer requested.
    if (isset($ppp['uptime'])) {
        if (!file_exists("/conf/{$ifcfg['if']}.log")) {
            file_put_contents("/conf/{$ifcfg['if']}.log", '');
        }
    } else {
        if (file_exists("/conf/{$ifcfg['if']}.log")) {
            @unlink("/conf/{$ifcfg['if']}.log");
        }
    }

    /* clean up old lock files */
    foreach ($ports as $port) {
        @unlink("/var/spool/lock/LCK..{$port}");
    }

    /* fire up mpd */
    mwexecf(
        '/usr/local/sbin/mpd5 -b -k -d /var/etc -f %s -p %s -s ppp %s',
        array("mpd_{$interface}.conf", "/var/run/{$ppp['type']}_{$interface}.pid", "{$ppp['type']}client")
    );

    /* wait for up to 10 seconds for the interface to appear (ppp(oe)) */
    $i = 0;
    while ($i < 10) {
        exec("/sbin/ifconfig " . escapeshellarg($ppp['if']) . " 2>&1", $out, $ret);
        if ($ret == 0) {
            break;
        }
        sleep(1);
        $i++;
    }

    /* we only support the 3gstats.php for huawei modems for now. Will add more later. */
    /* We should be able to launch the right version for each modem */
    /* We can also guess the mondev from the manufacturer */
    exec("usbconfig | egrep -ie '(huawei)'", $usbmodemoutput);
    mwexec("/bin/ps auxww|grep \"{$interface}\" |grep \"[3]gstats\" | awk '{print $2}' |xargs kill");
    foreach ($ports as $port) {
        $mondev = '';
        if (preg_match("/huawei/i", implode("\n", $usbmodemoutput))) {
            $mondev  = substr(basename($port), 0, -1);
            $devlist = glob("/dev/{$mondev}?");
            $mondev = basename(end($devlist));
        }
        if (preg_match("/zte/i", implode("\n", $usbmodemoutput))) {
            $mondev  = substr(basename($port), 0, -1) . "1";
        }
        if ($mondev != '') {
            log_error("Starting 3gstats.php on device '{$mondev}' for interface '{$interface}'");
            mwexec_bg("/usr/local/sbin/3gstats.php {$mondev} {$interface}");
        }
    }

    return 1;
}

function interfaces_carp_setup($verbose = false)
{
    global $config;

    if ($verbose) {
        echo 'Configuring CARP settings...';
        flush();
    }

    if (!empty($config['hasync']['pfsyncinterface'])) {
        $carp_sync_int = get_real_interface($config['hasync']['pfsyncinterface']);
    }

    if (!empty($carp_sync_int) && isset($config['hasync']['pfsyncenabled'])) {
        if (!empty($config['hasync']['pfsyncpeerip']) && is_ipaddrv4($config['hasync']['pfsyncpeerip'])) {
            $syncpeer = "syncpeer {$config['hasync']['pfsyncpeerip']}";
        } else {
            $syncpeer = "-syncpeer";
        }

        mwexec("/sbin/ifconfig pfsync0 syncdev {$carp_sync_int} {$syncpeer} up", false);
    } else {
        mwexec("/sbin/ifconfig pfsync0 -syncdev -syncpeer down", false);
    }

    if ($verbose) {
        echo "done.\n";
    }
}

function interface_proxyarp_configure($interface = '')
{
    global $config;

    /* kill any running choparp, on restart "all" */
    if (empty($interface)) {
        killbyname('choparp');
    }

    $paa = array();
    if (isset($config['virtualip']['vip'])) {
        /* group by interface */
        foreach ($config['virtualip']['vip'] as $vipent) {
            if ($vipent['mode'] === "proxyarp") {
                if (empty($interface) || $interface == $vipent['interface']) {
                    if (empty($paa[$vipent['interface']])) {
                        $paa[$proxyif] = array();
                    }
                    $paa[$vipent['interface']][] = $vipent;
                }
            }
        }
    }

    foreach ($paa as $paif => $paents) {
        $paaifip = get_interface_ip($paif);
        if (!is_ipaddr($paaifip)) {
            continue;
        }
        $vipif = get_real_interface($paif);
        $pid_filename = "/var/run/choparp_{$vipif}.pid";
        $args = "-p {$pid_filename} {$vipif} auto";
        foreach ($paents as $paent) {
            $args .= " " . escapeshellarg("{$paent['subnet']}/{$paent['subnet_bits']}");
        }
        if (!empty($interface)) {
            killbypid($pid_filename);
        }
        mwexec_bg("/usr/local/sbin/choparp " . $args);
    }
}


function interfaces_vips_configure($interface = '', $verbose = false)
{
    global $config;

    $paa = array();

    if (isset($config['virtualip']['vip'])) {
        $carp_setuped = false;
        $anyproxyarp = false;
        foreach ($config['virtualip']['vip'] as $vip) {
            if ($interface == "" || $vip['interface'] == $interface) {
                switch ($vip['mode']) {
                    case "proxyarp":
                        $anyproxyarp = true;
                        break;
                    case "ipalias":
                        interface_ipalias_configure($vip);
                        break;
                    case "carp":
                        if ($carp_setuped == false) {
                            $carp_setuped = true;
                        }
                        interface_carp_configure($vip);
                        break;
                }
            }
        }
        if ($carp_setuped == true) {
            interfaces_carp_setup($verbose);
        }
        if ($anyproxyarp == true) {
            interface_proxyarp_configure();
        }
    }
}

function interface_ipalias_configure(&$vip)
{
    global $config;

    if ($vip['mode'] != "ipalias") {
        return;
    }

    if ($vip['interface'] != 'lo0' && !isset($config['interfaces'][$vip['interface']]['enable'])) {
        return;
    }

    $if = escapeshellarg(get_real_interface($vip['interface']));
    $af = "inet";
    if (is_ipaddrv6($vip['subnet'])) {
        $af = "inet6";
    }
    $vhid = !empty($vip['vhid']) ? "vhid " . escapeshellarg($vip['vhid']) : "";
    $gateway = !empty($vip['gateway']) ? escapeshellarg($vip['gateway']) . " " : "";
    mwexec("/sbin/ifconfig " . $if ." {$af} ". escapeshellarg($vip['subnet']) ."/" . escapeshellarg($vip['subnet_bits']) . " alias ".$gateway. $vhid);
}

function interface_carp_configure(&$vip)
{
    global $config;

    if ($vip['mode'] != 'carp') {
        return;
    }

    // when CARP is temporary disabled, don't try to configure on any interface-up events
    if (get_single_sysctl('net.inet.carp.allow') == '0') {
        return;
    }

    /* NOTE: Maybe its useless nowdays */
    $realif = get_real_interface($vip['interface']);
    if (!does_interface_exist($realif)) {
        file_notice("CARP", sprintf(gettext("Interface specified for the virtual IP address %s does not exist. Skipping this VIP."), $vip['subnet']), "Firewall: Virtual IP", "");
        return;
    }

    if (is_ipaddrv4($vip['subnet'])) {
        /* Ensure a IP on this interface exists prior to configuring CARP. */
        $ww_subnet_ip = find_interface_ip($realif);
        if (!is_ipaddrv4($ww_subnet_ip)) {
            file_notice("CARP", sprintf(gettext("Sorry but we could not find a required assigned ip address on the interface for the virtual IP address %s."), $vip['subnet']), "Firewall: Virtual IP", "");
            return;
        }
    } elseif (is_ipaddrv6($vip['subnet'])) {
        /* Ensure a IP on this interface exists prior to configuring CARP. */
        $ww_subnet_ip = find_interface_ipv6($realif);
        if (!is_ipaddrv6($ww_subnet_ip)) {
            file_notice("CARP", sprintf(gettext("Sorry but we could not find a required assigned ip address on the interface for the virtual IPv6 address %s."), $vip['subnet']), "Firewall: Virtual IP", "");
            return;
        }
    }

    $vip_password = $vip['password'];
    $vip_password = escapeshellarg(addslashes(str_replace(" ", "", $vip_password)));
    if ($vip['password'] != "") {
        $password = " pass {$vip_password}";
    }

    $advbase = "";
    if (!empty($vip['advbase'])) {
        $advbase = "advbase " . escapeshellarg($vip['advbase']);
    }

    $carp_maintenancemode = isset($config["virtualip_carp_maintenancemode"]);
    if ($carp_maintenancemode) {
        $advskew = "advskew 254";
    } else {
        $advskew = "advskew " . escapeshellarg($vip['advskew']);
    }

    mwexec("/sbin/ifconfig {$realif} vhid " . escapeshellarg($vip['vhid']) . " {$advskew} {$advbase} {$password}");

    if (is_ipaddrv4($vip['subnet'])) {
        mwexec("/sbin/ifconfig {$realif} " . escapeshellarg($vip['subnet']) . "/" . escapeshellarg($vip['subnet_bits']) . " alias vhid " . escapeshellarg($vip['vhid']));
    } elseif (is_ipaddrv6($vip['subnet'])) {
        mwexec("/sbin/ifconfig {$realif} inet6 " . escapeshellarg($vip['subnet']) . " prefixlen " . escapeshellarg($vip['subnet_bits']) . " alias vhid " . escapeshellarg($vip['vhid']));
    }

    return $realif;
}

function interface_wireless_clone($realif, $wlcfg)
{
    global $config;

    /*   Check to see if interface has been cloned as of yet.
     *   If it has not been cloned then go ahead and clone it.
     */
    $needs_clone = false;
    if (isset($wlcfg['wireless']) && is_array($wlcfg['wireless'])) {
        $wlcfg_mode = $wlcfg['wireless']['mode'];
    } else {
        $wlcfg_mode = $wlcfg['mode'];
    }
    switch($wlcfg_mode) {
        case "hostap":
            $mode = "wlanmode hostap";
            break;
        case "adhoc":
            $mode = "wlanmode adhoc";
            break;
        default:
            $mode = "";
            break;
    }
    $baseif = interface_get_wireless_base($wlcfg['if']);
    if (does_interface_exist($realif)) {
        exec("/sbin/ifconfig " . escapeshellarg($realif), $output, $ret);
        $ifconfig_str = implode($output);
        if (($wlcfg_mode == "hostap") && (! preg_match("/hostap/si", $ifconfig_str))) {
            log_error("Interface {$realif} changed to hostap mode");
            $needs_clone = true;
        }
        if (($wlcfg_mode == "adhoc") && (! preg_match("/adhoc/si", $ifconfig_str))) {
            log_error("Interface {$realif} changed to adhoc mode");
            $needs_clone = true;
        }
        if (($wlcfg_mode == "bss") && (preg_match("/hostap|adhoc/si", $ifconfig_str))) {
            log_error("Interface {$realif} changed to infrastructure mode");
            $needs_clone = true;
        }
    } else {
        $needs_clone = true;
    }

    if ($needs_clone) {
        /* remove previous instance if it exists */
        if (does_interface_exist($realif)) {
            legacy_interface_destroy($realif);
        }

        log_error("Cloning new wireless interface {$realif}");
        exec("/sbin/ifconfig wlan create wlandev {$baseif} {$mode} bssid name {$realif} 2>&1", $out, $ret);
        if ($ret <> 0) {
            log_error("Failed to clone interface {$baseif} with error code {$ret}, output {$out[0]}");
            return false;
        }
        file_put_contents("/tmp/{$realif}_oldmac", get_interface_mac($realif));
    }
    return true;
}

function interface_sync_wireless_clones(&$ifcfg, $sync_changes = false)
{
    global $config;

    $shared_settings = array(
        'channel',
        'diversity',
        'protmode',
        'regcountry',
        'regdomain',
        'reglocation',
        'rxantenna',
        'standard',
        'turbo',
        'txantenna',
        'txpower',
    );

    if (!is_interface_wireless($ifcfg['if'])) {
        return;
    }

    $baseif = interface_get_wireless_base($ifcfg['if']);

    // Sync shared settings for assigned clones
    $iflist = get_configured_interface_list(false, true);
    foreach ($iflist as $if) {
        if ($baseif == interface_get_wireless_base($config['interfaces'][$if]['if']) && $ifcfg['if'] != $config['interfaces'][$if]['if']) {
            if (isset($config['interfaces'][$if]['wireless']['standard']) || $sync_changes) {
                foreach ($shared_settings as $setting) {
                    if ($sync_changes) {
                        if (isset($ifcfg['wireless'][$setting])) {
                            $config['interfaces'][$if]['wireless'][$setting] = $ifcfg['wireless'][$setting];
                        } elseif (isset($config['interfaces'][$if]['wireless'][$setting])) {
                            unset($config['interfaces'][$if]['wireless'][$setting]);
                        }
                    } else {
                        if (isset($config['interfaces'][$if]['wireless'][$setting])) {
                            $ifcfg['wireless'][$setting] = $config['interfaces'][$if]['wireless'][$setting];
                        } elseif (isset($ifcfg['wireless'][$setting])) {
                            unset($ifcfg['wireless'][$setting]);
                        }
                    }
                }
                if (!$sync_changes) {
                    break;
                }
            }
        }
    }

    // Read or write settings at shared area
    if (isset($config['wireless']['interfaces'][$baseif]) && is_array($config['wireless']['interfaces'][$baseif])) {
        foreach ($shared_settings as $setting) {
            if ($sync_changes) {
                if (isset($ifcfg['wireless'][$setting])) {
                    $config['wireless']['interfaces'][$baseif][$setting] = $ifcfg['wireless'][$setting];
                } elseif (isset($config['wireless']['interfaces'][$baseif][$setting])) {
                    unset($config['wireless']['interfaces'][$baseif][$setting]);
                }
            } elseif (isset($config['wireless']['interfaces'][$baseif][$setting])) {
                if (isset($config['wireless']['interfaces'][$baseif][$setting])) {
                    $ifcfg['wireless'][$setting] = $config['wireless']['interfaces'][$baseif][$setting];
                } elseif (isset($ifcfg['wireless'][$setting])) {
                    unset($ifcfg['wireless'][$setting]);
                }
            }
        }
    }

    // Sync the mode on the clone creation page with the configured mode on the interface
    if (stristr($ifcfg['if'], "_wlan") && isset($config['wireless']['clone']) && is_array($config['wireless']['clone'])) {
        foreach ($config['wireless']['clone'] as &$clone) {
            if ($clone['cloneif'] == $ifcfg['if']) {
                if ($sync_changes) {
                    $clone['mode'] = $ifcfg['wireless']['mode'];
                } else {
                    $ifcfg['wireless']['mode'] = $clone['mode'];
                }
                break;
            }
        }
        unset($clone);
    }
}

function interface_wireless_configure($if, &$wl, &$wlcfg)
{
    global $config;

   /*    open up a shell script that will be used to output the commands.
    *    since wireless is changing a lot, these series of commands are fragile
    *    and will sometimes need to be verified by a operator by executing the command
    *    and returning the output of the command to the developers for inspection.  please
    *    do not change this routine from a shell script to individual exec commands.  -sullrich
    */

    // Remove script file
    @unlink("/tmp/{$if}_setup.sh");

    // Clone wireless nic if needed.
    interface_wireless_clone($if, $wl);

    // Reject inadvertent changes to shared settings in case the interface hasn't been configured.
    interface_sync_wireless_clones($wl, false);

    $fd_set = fopen("/tmp/{$if}_setup.sh", "w");
    fwrite($fd_set, "#!/bin/sh\n");
    fwrite($fd_set, "# wireless configuration script.\n\n");

    /* set values for /path/program */
    $wpa_supplicant = '/usr/local/sbin/wpa_supplicant';
    $hostapd = '/usr/local/sbin/hostapd';
    $killall = '/usr/bin/killall';
    $ifconfig = '/sbin/ifconfig';
    $sysctl = '/sbin/sysctl';

    /* Set all wireless ifconfig variables (split up to get rid of needed checking) */

    $wlcmd = array();
    $wl_sysctl = array();
    /* Make sure it's up */
    $wlcmd[] = "up";
    /* Set a/b/g standard */
    $standard = str_replace(" Turbo", "", $wlcfg['standard']);
    $wlcmd[] = "mode " . escapeshellarg($standard);

    /* XXX: Disable ampdu for now on mwl when running in 11n mode
     * to prevent massive packet loss under certain conditions. */
    if (preg_match("/^mwl/i", $if) && ($standard == "11ng" || $standard == "11na")) {
        $wlcmd[] = "-ampdu";
    }

    /* Set ssid */
    if ($wlcfg['ssid']) {
        $wlcmd[] = "ssid " . escapeshellarg($wlcfg['ssid']);
    }

    /* Set 802.11g protection mode */
    $wlcmd[] = "protmode " . escapeshellarg($wlcfg['protmode']);

    /* set wireless channel value */
    if (isset($wlcfg['channel'])) {
        if ($wlcfg['channel'] == "0") {
            $wlcmd[] = "channel any";
        } else {
            $wlcmd[] = "channel " . escapeshellarg($wlcfg['channel']);
        }
    }

    /* Set antenna diversity value */
    if (isset($wlcfg['diversity'])) {
        $wl_sysctl[] = "diversity=" . escapeshellarg($wlcfg['diversity']);
    }

    /* Set txantenna value */
    if (isset($wlcfg['txantenna'])) {
        $wl_sysctl[] = "txantenna=" . escapeshellarg($wlcfg['txantenna']);
    }

    /* Set rxantenna value */
    if (isset($wlcfg['rxantenna'])) {
        $wl_sysctl[] = "rxantenna=" . escapeshellarg($wlcfg['rxantenna']);
    }

    /* Set wireless hostap mode */
    if ($wlcfg['mode'] == "hostap") {
        $wlcmd[] = "mediaopt hostap";
    } else {
        $wlcmd[] = "-mediaopt hostap";
    }

    /* Set wireless adhoc mode */
    if ($wlcfg['mode'] == "adhoc") {
        $wlcmd[] = "mediaopt adhoc";
    } else {
        $wlcmd[] = "-mediaopt adhoc";
    }

    /* Not neccesary to set BSS mode as this is default if adhoc and/or hostap is NOT set */

    /* handle hide ssid option */
    if (isset($wlcfg['hidessid']['enable'])) {
        $wlcmd[] = "hidessid";
    } else {
        $wlcmd[] = "-hidessid";
    }

    /* handle pureg (802.11g) only option */
    if (isset($wlcfg['pureg']['enable'])) {
        $wlcmd[] = "mode 11g pureg";
    } else {
        $wlcmd[] = "-pureg";
    }

    /* handle puren (802.11n) only option */
    if (isset($wlcfg['puren']['enable'])) {
        $wlcmd[] = "puren";
    } else {
        $wlcmd[] = "-puren";
    }

    /* enable apbridge option */
    if (isset($wlcfg['apbridge']['enable'])) {
        $wlcmd[] = "apbridge";
    } else {
        $wlcmd[] = "-apbridge";
    }

    /* handle turbo option */
    if (isset($wlcfg['turbo']['enable'])) {
        $wlcmd[] = "mediaopt turbo";
    } else {
        $wlcmd[] = "-mediaopt turbo";
    }

    /* handle wme option */
    if (isset($wlcfg['wme']['enable'])) {
        $wlcmd[] = "wme";
    } else {
        $wlcmd[] = "-wme";
    }

    /* set up wep if enabled */
    $wepset = "";
    if (isset($wlcfg['wep']['enable']) && is_array($wlcfg['wep']['key'])) {
        switch($wlcfg['wpa']['auth_algs']) {
            case "1":
                $wepset .= "authmode open wepmode on ";
                break;
            case "2":
                $wepset .= "authmode shared wepmode on ";
                break;
            case "3":
                $wepset .= "authmode mixed wepmode on ";
        }
        $i = 1;
        foreach ($wlcfg['wep']['key'] as $wepkey) {
            $wepset .= "wepkey " . escapeshellarg("{$i}:{$wepkey['value']}") . " ";
            if (isset($wepkey['txkey'])) {
                $wlcmd[] = "weptxkey {$i} ";
            }
            $i++;
        }
        $wlcmd[] = $wepset;
    } else {
        $wlcmd[] = "authmode open wepmode off ";
    }

    kill_hostapd($if);
    mwexec(kill_wpasupplicant("{$if}"));

    /* generate wpa_supplicant/hostap config if wpa is enabled */
    switch ($wlcfg['mode']) {
        case 'bss':
            if (isset($wlcfg['wpa']['enable'])) {
                $wpa .= <<<EOD
ctrl_interface=/var/run/wpa_supplicant
ctrl_interface_group=0
ap_scan=1
#fast_reauth=1
network={
ssid="{$wlcfg['ssid']}"
scan_ssid=1
priority=5
key_mgmt={$wlcfg['wpa']['wpa_key_mgmt']}
psk="{$wlcfg['wpa']['passphrase']}"
pairwise={$wlcfg['wpa']['wpa_pairwise']}
group={$wlcfg['wpa']['wpa_pairwise']}
}
EOD;

                @file_put_contents("/var/etc/wpa_supplicant_{$if}.conf", $wpa);
                unset($wpa);
            }
            break;
        case 'hostap':
            if (!empty($wlcfg['wpa']['passphrase'])) {
                $wpa_passphrase = "wpa_passphrase={$wlcfg['wpa']['passphrase']}\n";
            } else {
                $wpa_passphrase = "";
            }
            if (isset($wlcfg['wpa']['enable'])) {
                $wpa .= <<<EOD
interface={$if}
driver=bsd
logger_syslog=-1
logger_syslog_level=0
logger_stdout=-1
logger_stdout_level=0
dump_file=/tmp/hostapd_{$if}.dump
ctrl_interface=/var/run/hostapd
ctrl_interface_group=wheel
#accept_mac_file=/tmp/hostapd_{$if}.accept
#deny_mac_file=/tmp/hostapd_{$if}.deny
#macaddr_acl={$wlcfg['wpa']['macaddr_acl']}
ssid={$wlcfg['ssid']}
debug={$wlcfg['wpa']['debug_mode']}
auth_algs={$wlcfg['wpa']['auth_algs']}
wpa={$wlcfg['wpa']['wpa_mode']}
wpa_key_mgmt={$wlcfg['wpa']['wpa_key_mgmt']}
wpa_pairwise={$wlcfg['wpa']['wpa_pairwise']}
wpa_group_rekey={$wlcfg['wpa']['wpa_group_rekey']}
wpa_gmk_rekey={$wlcfg['wpa']['wpa_gmk_rekey']}
wpa_strict_rekey={$wlcfg['wpa']['wpa_strict_rekey']}
{$wpa_passphrase}

EOD;

                if (isset($wlcfg['wpa']['rsn_preauth'])) {
                    $wpa .= <<<EOD
# Enable the next lines for preauth when roaming. Interface = wired or wireless interface talking to the AP you want to roam from/to
rsn_preauth=1
rsn_preauth_interfaces={$if}

EOD;
                }
                if (is_array($wlcfg['wpa']['ieee8021x']) && isset($wlcfg['wpa']['ieee8021x']['enable'])) {
                    $wpa .= "ieee8021x=1\n";

                    if (!empty($wlcfg['auth_server_addr']) && !empty($wlcfg['auth_server_shared_secret'])) {
                        $auth_server_port = "1812";
                        if (!empty($wlcfg['auth_server_port']) && is_numeric($wlcfg['auth_server_port'])) {
                            $auth_server_port = intval($wlcfg['auth_server_port']);
                        }
                        $wpa .= <<<EOD

auth_server_addr={$wlcfg['auth_server_addr']}
auth_server_port={$auth_server_port}
auth_server_shared_secret={$wlcfg['auth_server_shared_secret']}

EOD;
                        if (!empty($wlcfg['auth_server_addr2']) && !empty($wlcfg['auth_server_shared_secret2'])) {
                            $auth_server_port2 = "1812";
                            if (!empty($wlcfg['auth_server_port2']) && is_numeric($wlcfg['auth_server_port2'])) {
                                $auth_server_port2 = intval($wlcfg['auth_server_port2']);
                            }

                            $wpa .= <<<EOD
auth_server_addr={$wlcfg['auth_server_addr2']}
auth_server_port={$auth_server_port2}
auth_server_shared_secret={$wlcfg['auth_server_shared_secret2']}

EOD;
                        }
                    }
                }

                @file_put_contents("/var/etc/hostapd_{$if}.conf", $wpa);
                unset($wpa);
            }
            break;
    }

    /*
     *    all variables are set, lets start up everything
     */

    $baseif = interface_get_wireless_base($if);
    preg_match("/^(.*?)([0-9]*)$/", $baseif, $baseif_split);
    $wl_sysctl_prefix = 'dev.' . $baseif_split[1] . '.' . $baseif_split[2];

    /* set sysctls for the wireless interface */
    if (!empty($wl_sysctl)) {
        fwrite($fd_set, "# sysctls for {$baseif}\n");
        foreach ($wl_sysctl as $wl_sysctl_line) {
            fwrite($fd_set, "{$sysctl} {$wl_sysctl_prefix}.{$wl_sysctl_line}\n");
        }
    }

    if (isset($wlcfg['wpa']['enable'])) {
        if ($wlcfg['mode'] == "bss") {
            fwrite($fd_set, "{$wpa_supplicant} -B -i {$if} -c /var/etc/wpa_supplicant_{$if}.conf\n");
        }
        if ($wlcfg['mode'] == "hostap") {
            /* add line to script to restore old mac to make hostapd happy */
            if (file_exists("/tmp/{$if}_oldmac")) {
                $if_oldmac = file_get_contents("/tmp/{$if}_oldmac");
                if (is_macaddr($if_oldmac)) {
                    fwrite($fd_set, "{$ifconfig} " . escapeshellarg($if) .
                        " link " . escapeshellarg($if_oldmac) . "\n");
                }
            }

            fwrite($fd_set, "{$hostapd} -B -P /var/run/hostapd_{$if}.pid /var/etc/hostapd_{$if}.conf\n");

            /* add line to script to restore spoofed mac after running hostapd */
            if (file_exists("/tmp/{$if}_oldmac")) {
                if ($wl['spoofmac']) {
                    $if_curmac = $wl['spoofmac'];
                } else {
                    $if_curmac = get_interface_mac($if);
                }
                if (is_macaddr($if_curmac)) {
                    fwrite($fd_set, "{$ifconfig} " . escapeshellarg($if) .
                        " link " . escapeshellarg($if_curmac) . "\n");
                }
            }
        }
    }

    fclose($fd_set);

    /* Making sure regulatory settings have actually changed
   * before applying, because changing them requires bringing
   * down all wireless networks on the interface. */
    exec("{$ifconfig} " . escapeshellarg($if), $output);
    $ifconfig_str = implode($output);
    unset($output);
    $reg_changing = false;

    /* special case for the debug country code */
    if ($wlcfg['regcountry'] == 'DEBUG' && !preg_match("/\sregdomain\s+DEBUG\s/si", $ifconfig_str)) {
        $reg_changing = true;
    } elseif ($wlcfg['regdomain'] && !preg_match("/\sregdomain\s+{$wlcfg['regdomain']}\s/si", $ifconfig_str)) {
        $reg_changing = true;
    } elseif ($wlcfg['regcountry'] && !preg_match("/\scountry\s+{$wlcfg['regcountry']}\s/si", $ifconfig_str)) {
        $reg_changing = true;
    } elseif ($wlcfg['reglocation'] == 'anywhere' && preg_match("/\s(indoor|outdoor)\s/si", $ifconfig_str)) {
        $reg_changing = true;
    } elseif ($wlcfg['reglocation'] && $wlcfg['reglocation'] != 'anywhere' && !preg_match("/\s{$wlcfg['reglocation']}\s/si", $ifconfig_str)) {
        $reg_changing = true;
    }

    if ($reg_changing) {
        /* set regulatory domain */
        if ($wlcfg['regdomain']) {
            $wlregcmd[] = "regdomain " . escapeshellarg($wlcfg['regdomain']);
        }

        /* set country */
        if ($wlcfg['regcountry']) {
            $wlregcmd[] = "country " . escapeshellarg($wlcfg['regcountry']);
        }

        /* set location */
        if ($wlcfg['reglocation']) {
            $wlregcmd[] = escapeshellarg($wlcfg['reglocation']);
        }

        $wlregcmd_args = implode(" ", $wlregcmd);

        /* build a complete list of the wireless clones for this interface */
        $clone_list = array();
        if (does_interface_exist(interface_get_wireless_clone($baseif))) {
            $clone_list[] = interface_get_wireless_clone($baseif);
        }
        if (isset($config['wireless']['clone'])) {
            foreach ($config['wireless']['clone'] as $clone) {
                if ($clone['if'] == $baseif) {
                    $clone_list[] = $clone['cloneif'];
                }
            }
        }

        /* find which clones are up and bring them down */
        $ifup = legacy_interface_listget('up');
        $clones_up = array();
        foreach ($clone_list as $clone_if) {
            if (in_array($clone_if, $ifup)) {
                $clones_up[] = $clone_if;
                mwexec("{$ifconfig} " . escapeshellarg($clone_if) . " down");
            }
        }

        /* apply the regulatory settings */
        mwexec("{$ifconfig} " . escapeshellarg($if) . " {$wlregcmd_args}");

        /* bring the clones back up that were previously up */
        foreach ($clones_up as $clone_if) {
            mwexec("{$ifconfig} " . escapeshellarg($clone_if) . " up");

            /*
             * Rerun the setup script for the interface if it isn't this interface, the interface
             * is in infrastructure mode, and WPA is enabled.
             * This can be removed if wpa_supplicant stops dying when you bring the interface down.
             */
            if ($clone_if != $if) {
                $friendly_if = convert_real_interface_to_friendly_interface_name($clone_if);
                if (!empty($friendly_if)
                    && $config['interfaces'][$friendly_if]['wireless']['mode'] == "bss"
                    && isset($config['interfaces'][$friendly_if]['wireless']['wpa']['enable'])
                ) {
                    mwexec('/bin/sh /tmp/' . escapeshellarg($clone_if) . '_setup.sh');
                }
            }
        }
    }

    /* The mode must be specified in a separate command before ifconfig
     * will allow the mode and channel at the same time in the next. */
    mwexec("/sbin/ifconfig " . escapeshellarg($if) . " mode " . escapeshellarg($standard));

    /* configure wireless */
    $wlcmd_args = implode(" ", $wlcmd);
    mwexec("/sbin/ifconfig " . escapeshellarg($if) . " " . $wlcmd_args, false);
    unset($wlcmd_args, $wlcmd);

    /* configure txpower setting (it has been known to fail so run it separately) */
    if (!empty($wlcfg['txpower'])) {
        mwexecf('/sbin/ifconfig %s txpower %s', array($if, $wlcfg['txpower']));
    }

    sleep(1);
    /* execute hostapd and wpa_supplicant if required in shell */
    mwexec('/bin/sh /tmp/' . escapeshellarg($if) . '_setup.sh');

    return 0;
}

function kill_hostapd($interface)
{
    return killbypid("/var/run/hostapd_{$interface}.pid");
}

function kill_wpasupplicant($interface)
{
    return "/bin/pkill -f \"wpa_supplicant .*{$interface}\\.conf\"\n";
}

function find_dhclient_process($interface)
{
    if ($interface) {
        $pid = `/bin/pgrep -axf "dhclient: {$interface}"`;
    } else {
        $pid = 0;
    }

    return intval($pid);
}

function kill_dhclient_process($interface)
{
    if (empty($interface) || !does_interface_exist($interface)) {
        return;
    }

    killbypid(find_dhclient_process($interface), 'TERM', true);
}

function interface_virtual_create($interface)
{
    global $config;

    if (strstr($interface, '_vlan')) {
        interfaces_vlan_configure($vlan);
    } elseif (substr($interface, 0, 3) == 'gre') {
        interfaces_gre_configure(0, false, $interface);
    } elseif (substr($interface, 0, 3) == 'gif') {
        interfaces_gif_configure(0, false, $interface);
    } elseif (substr($interface, 0, 4) == 'ovpn') {
        /* XXX this looks like a plugin spot... */
        openvpn_configure_interface($interface);
    } elseif (substr($interface, 0, 4) == 'lagg') {
        interfaces_lagg_configure($interface);
    } elseif (substr($interface, 0, 6) == 'bridge') {
        interfaces_bridge_configure(0, $interface);
    }
}


function interface_configure($interface = 'wan', $reloadall = false, $linkupevent = false, $verbose = false)
{
    global $config;

    $wancfg = $config['interfaces'][$interface];

    if (!isset($wancfg['enable'])) {
        return;
    }

    $wandescr = !empty($wancfg['descr']) ? strtoupper($wancfg['descr']) : strtoupper($interface);

    if ($verbose) {
        echo sprintf('Configuring %s interface...', $wandescr);
        flush();
    }

    $realif = get_real_interface($interface);
    $realhwif_array = get_parent_interface($interface);
    // Need code to handle MLPPP if we ever use $realhwif for MLPPP handling
    $realhwif = $realhwif_array[0];

    if (!file_exists("/var/run/booting") && !(substr($realif, 0, 4) == "ovpn")) {
        /* remove all IPv4 and IPv6 addresses */
        $tmpifaces = legacy_getall_interface_addresses($realif);
        foreach ($tmpifaces as $tmpiface) {
            if (is_ipaddrv6($tmpiface) || is_subnetv6($tmpiface)) {
                if (!is_linklocal($tmpiface)) {
                    mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 {$tmpiface} delete");
                }
            } else {
                if (is_subnetv4($tmpiface)) {
                    $tmpip = explode('/', $tmpiface);
                    $tmpip = $tmpip[0];
                } else {
                    $tmpip = $tmpiface;
                }
                legacy_interface_deladdress($realif, $tmpip);
            }
        }

        /* only bring down the interface when both v4 and v6 are set to NONE */
        if (empty($wancfg['ipaddr']) && empty($wancfg['ipaddrv6'])) {
            interface_bring_down($interface);
        }
    }

    $interface_to_check = $realif;
    switch ($wancfg['ipaddr']) {
        case 'pppoe':
        case 'l2tp':
        case 'pptp':
        case 'ppp':
            $interface_to_check = $realhwif;
            break;
    }

    /* Need to check that the interface exists or not in the case where its coming back from disabled state see #3270 */
    if (in_array(substr($realif, 0, 3), array("gre", "gif")) || !does_interface_exist($interface_to_check)) {
        interface_virtual_create($interface_to_check);
    }

    /* Disable Accepting router advertisements unless specifically requested */
    mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 -accept_rtadv");

    /* wireless configuration? */
    if (isset($wancfg['wireless']) && is_array($wancfg['wireless']) && !$linkupevent) {
        interface_wireless_configure($realif, $wancfg, $wancfg['wireless']);
    }

    $mac = get_interface_mac($realhwif);
    /*
     * Don't try to reapply the spoofed MAC if it's already applied.
     * When ifconfig link is used, it cycles the interface down/up, which triggers
     * the interface config again, which attempts to spoof the MAC again,
     * which cycles the link again...
     */
    if (!empty($wancfg['spoofmac']) && ($wancfg['spoofmac'] != $mac)) {
        mwexec("/sbin/ifconfig " . escapeshellarg($realhwif) .
            " link " . escapeshellarg($wancfg['spoofmac']));

        /* All vlans need to spoof their parent mac address, too. */
        if (isset($config['vlans']['vlan'])) {
            foreach ($config['vlans']['vlan'] as $vlan) {
                if ($vlan['if'] == $realhwif) {
                    mwexec("/sbin/ifconfig " . escapeshellarg($vlan['vlanif']) .
                        " link " . escapeshellarg($wancfg['spoofmac']));
                }
            }
        }
    }

    /* media */
    if (!empty($wancfg['media']) || !empty($wancfg['mediaopt'])) {
        $cmd = "/sbin/ifconfig " . escapeshellarg($realhwif);
        if (!empty($wancfg['media'])) {
            $cmd .= " media " . escapeshellarg($wancfg['media']);
        }
        if (!empty($wancfg['mediaopt'])) {
            $cmd .= " mediaopt " . escapeshellarg($wancfg['mediaopt']);
        }
        mwexec($cmd);
    }
    // apply interface hardware settings (tso, lro, ..)
    configure_interface_hardware($realhwif);

    $tunnelif = substr($realif, 0, 3);
    switch ($wancfg['ipaddr']) {
        case 'dhcp':
            interface_dhcp_configure($interface);
            break;
        case 'pppoe':
        case 'l2tp':
        case 'pptp':
        case 'ppp':
            interface_ppps_configure($interface);
            break;
        default:
            /* XXX: Kludge for now related to #3280 */
            if (!in_array($tunnelif, array("gif", "gre", "ovp"))) {
                if (is_ipaddrv4($wancfg['ipaddr']) && $wancfg['subnet'] <> "") {
                    legacy_interface_setaddress($realif, "{$wancfg['ipaddr']}/{$wancfg['subnet']}");
                }
            }
            break;
    }

    if (isset($wancfg['ipaddrv6'])) {
        switch ($wancfg['ipaddrv6']) {
            case 'slaac':
            case 'dhcp6':
                interface_dhcpv6_prepare($interface, $wancfg);
                interface_dhcpv6_configure($interface, $wancfg);
                break;
            case '6rd':
                interface_6rd_configure($interface, $wancfg);
                break;
            case '6to4':
                interface_6to4_configure($interface, $wancfg);
                break;
            case 'track6':
                interface_track6_configure($interface, $wancfg, $reloadall || $linkupevent);
                break;
            default:
                /* XXX: Kludge for now related to #3280 */
                if (!in_array($tunnelif, array("gif", "gre", "ovp"))) {
                    if (is_ipaddrv6($wancfg['ipaddrv6']) && $wancfg['subnetv6'] <> "") {
                        mwexec("/sbin/ifconfig " . escapeshellarg($realif) . " inet6 {$wancfg['ipaddrv6']} prefixlen " . escapeshellarg($wancfg['subnetv6']));
                    }
                }
                break;
        }
    }

    $intf_stats = legacy_interface_stats();
    if (!empty($wancfg['mtu'])) {
        if (stristr($realif, "_vlan")) {
            // max mtu to parent
            $mtu = $wancfg['mtu'] < $intf_stats[$realhwif]['mtu'] ? $wancfg['mtu'] : $intf_stats[$realhwif]['mtu'];
            if ($mtu != $intf_stats[$realif]['mtu']) {
                legacy_interface_mtu($realif, $mtu);
            }
        } elseif ($wancfg['mtu'] != $intf_stats['mtu']) {
            legacy_interface_mtu($realhwif, $wancfg['mtu']);
        }
    }

    // in case this interface has vlan's configured, make sure none of them has an mtu set higher than it's parent
    if (!empty($config['vlans']['vlan'])) {
        $intf_stats = legacy_interface_stats();
        foreach ($config['vlans']['vlan'] as $vlan) {
            if ($realhwif == $vlan['if'] && $intf_stats[$realhwif]['mtu'] < $intf_stats[$vlan['vlanif']]['mtu']) {
                legacy_interface_mtu($vlan['vlanif'], $intf_stats[$realhwif]['mtu']);
            }
        }
    }

    if (does_interface_exist($wancfg['if'])) {
        interfaces_bring_up($wancfg['if']);
    }

    if (!file_exists("/var/run/booting")) {
        link_interface_to_vips($interface, "update");

        unset($gre);
        $gre = link_interface_to_gre($interface);
        if (!empty($gre)) {
            array_walk($gre, 'interface_gre_configure');
        }

        unset($gif);
        $gif = link_interface_to_gif($interface);
        if (!empty($gif)) {
            array_walk($gif, 'interface_gif_configure');
        }

        if ($linkupevent == false || substr($realif, 0, 4) == "ovpn") {
            unset($bridgetmp);
            $bridgetmp = link_interface_to_bridge($interface);
            if (!empty($bridgetmp)) {
                interface_bridge_add_member($bridgetmp, $realif);
            }
        }

        $grouptmp = link_interface_to_group($interface);
        if (!empty($grouptmp)) {
            array_walk($grouptmp, 'interface_group_add_member');
        }

        if ($interface == 'lan') {
            /* make new hosts file */
            system_hosts_generate();
        }

        if ($reloadall == true) {
            system_routing_configure(false, $interface);
            ipsec_configure_do();
            plugins_configure('dns');
            services_dhcpd_configure();
            configd_run("dyndns reload {$interface}");
            configd_run("rfc2136 reload {$interface}");
        }
    }

    interfaces_staticarp_configure($interface);

    if ($verbose) {
        echo "done.\n";
    }

    return 0;
}

function interface_track6_configure($interface = 'lan', $wancfg, $linkupevent = false)
{
    global $config;

    if (!is_array($wancfg)) {
        return;
    }

    if (!isset($wancfg['enable'])) {
        return;
    }

    /* If the interface is not configured via another, exit */
    if (empty($wancfg['track6-interface'])) {
        return;
    }

    /* always configure a link-local of fe80::1:1 on the track6 interfaces */
    $realif = get_real_interface($interface);
    $linklocal = find_interface_ipv6_ll($realif);
    if (!empty($linklocal)) {
        mwexec("/sbin/ifconfig {$realif} inet6 {$linklocal} delete");
    }
    /* XXX: This might break for good on a carp installation using link-local as network ips */
    /* XXX: Probably should remove? */
    mwexec("/sbin/ifconfig {$realif} inet6 fe80::1:1%{$realif}");

    $trackcfg = $config['interfaces'][$wancfg['track6-interface']];
    if (!isset($trackcfg['enable'])) {
        log_error("Interface {$interface} tracking non-existent interface {$wancfg['track6-interface']}");
        return;
    }

    switch($trackcfg['ipaddrv6']) {
        case "6to4":
            interface_track6_6to4_configure($interface, $wancfg);
            break;
        case "6rd":
            interface_track6_6rd_configure($interface, $wancfg);
            break;
        case "dhcp6":
            if ($linkupevent == true) {
                /*
                 * NOTE: Usually come here from rc.linkup calling so just call directly intead of generating event
                 *  Instead of disrupting all other v4 configuration just restart DHCPv6 client for now
                 *
                 * XXX: Probably DHCPv6 client should handle this autmagically itself?
                 */
                $parentrealif = get_real_interface($wancfg['track6-interface']);
                killbypid("/var/run/dhcp6c_{$parentrealif}.pid", 'HUP');
            }
            break;
    }

    if (!file_exists("/var/run/booting") && $linkupevent == false) {
        services_dhcpd_configure("inet6");
    }

    return 0;
}

function interface_track6_6rd_configure($interface = 'lan', $lancfg)
{
    global $config;

    /* If the interface is not configured via another, exit */
    if (empty($lancfg['track6-interface'])) {
        return;
    }

    $wancfg = $config['interfaces'][$lancfg['track6-interface']];
    if (empty($wancfg)) {
        log_error("Interface {$interface} tracking non-existent interface {$lancfg['track6-interface']}");
        return;
    }

    $ip4address = get_interface_ip($lancfg['track6-interface']);
    if (!is_ipaddrv4($ip4address)) { /* XXX: This should not be needed by 6rd || (is_private_ip($ip4address))) { */
        log_error("The interface IPv4 '{$ip4address}' address on interface '{$lancfg['track6-interface']}' is not public, not configuring 6RD tunnel");
        return;
    }
    $hexwanv4 = return_hex_ipv4($ip4address);

    /* create the long prefix notation for math, save the prefix length */
    $rd6prefix = explode("/", $wancfg['prefix-6rd']);
    $rd6prefixlen = $rd6prefix[1];
    $rd6prefix = Net_IPv6::uncompress($rd6prefix[0]);

    /* binary presentation of the prefix for all 128 bits. */
    $rd6lanbin = convert_ipv6_to_128bit($rd6prefix);

    /* just save the left prefix length bits */
    $rd6lanbin = substr($rd6lanbin, 0, $rd6prefixlen);
    /* add the v4 address, offset n bits from the left */
    $rd6lanbin .= substr(sprintf("%032b", hexdec($hexwanv4)), (0 + $wancfg['prefix-6rd-v4plen']), 32);

    /* add the custom prefix id, max 32bits long? (64 bits - (prefixlen + (32 - v4plen)) */
    /* 64 - (37 + (32 - 17)) = 8 == /52 */
    $restbits = 64 - ($rd6prefixlen + (32 - $wancfg['prefix-6rd-v4plen']));
    // echo "64 - (prefixlen {$rd6prefixlen} + v4len (32 - {$wancfg['prefix-6rd-v4plen']})) = {$restbits} \n";
    $rd6lanbin .= substr(sprintf("%032b", str_pad($lancfg['track6-prefix-id'], 32, "0", STR_PAD_LEFT)), (32 - $restbits), 32);
    /* fill the rest out with zeros */
    $rd6lanbin = str_pad($rd6lanbin, 128, "0", STR_PAD_RIGHT);

    /* convert the 128 bits for the lan address back into a valid IPv6 address */
    $rd6lan = convert_128bit_to_ipv6($rd6lanbin) ."1";

    $lanif = get_real_interface($interface);
    $oip = find_interface_ipv6($lanif);
    if (is_ipaddrv6($oip)) {
        mwexec("/sbin/ifconfig {$lanif} inet6 {$oip} delete");
    }
    log_error("rd6 {$interface} with ipv6 address {$rd6lan} based on {$lancfg['track6-interface']} ipv4 {$ip4address}");
    mwexec("/sbin/ifconfig {$lanif} inet6 {$rd6lan} prefixlen 64");

    return 0;
}

function interface_track6_6to4_configure($interface = 'lan', $lancfg)
{
    global $config;

    /* If the interface is not configured via another, exit */
    if (empty($lancfg['track6-interface'])) {
        return;
    }

    $wancfg = $config['interfaces'][$lancfg['track6-interface']];
    if (empty($wancfg)) {
        log_error("Interface {$interface} tracking non-existent interface {$lancfg['track6-interface']}");
        return;
    }

    $ip4address = get_interface_ip($lancfg['track6-interface']);
    if (!is_ipaddrv4($ip4address) || is_private_ip($ip4address)) {
        log_error("The interface IPv4 '{$ip4address}' address on interface '{$lancfg['track6-interface']}' is not public, not configuring 6RD tunnel");
        return;
    }
    $hexwanv4 = return_hex_ipv4($ip4address);

    /* create the long prefix notation for math, save the prefix length */
    $sixto4prefix = "2002::";
    $sixto4prefixlen = 16;
    $sixto4prefix = Net_IPv6::uncompress($sixto4prefix);

    /* binary presentation of the prefix for all 128 bits. */
    $sixto4lanbin = convert_ipv6_to_128bit($sixto4prefix);

    /* just save the left prefix length bits */
    $sixto4lanbin = substr($sixto4lanbin, 0, $sixto4prefixlen);
    /* add the v4 address */
    $sixto4lanbin .= sprintf("%032b", hexdec($hexwanv4));
    /* add the custom prefix id */
    $sixto4lanbin .= sprintf("%016b", $lancfg['track6-prefix-id']);
    /* fill the rest out with zeros */
    $sixto4lanbin = str_pad($sixto4lanbin, 128, "0", STR_PAD_RIGHT);

    /* convert the 128 bits for the lan address back into a valid IPv6 address */
    $sixto4lan = convert_128bit_to_ipv6($sixto4lanbin) ."1";

    $lanif = get_real_interface($interface);
    $oip = find_interface_ipv6($lanif);
    if (is_ipaddrv6($oip)) {
        mwexec("/sbin/ifconfig {$lanif} inet6 {$oip} delete");
    }
    log_error("sixto4 {$interface} with ipv6 address {$sixto4lan} based on {$lancfg['track6-interface']} ipv4 {$ip4address}");
    mwexec("/sbin/ifconfig {$lanif} inet6 {$sixto4lan} prefixlen 64");

    return 0;
}

function interface_6rd_configure($interface = "wan", $wancfg)
{
    global $config;

    /* because this is a tunnel interface we can only function
     *  with a public IPv4 address on the interface */
    if (!is_array($wancfg)) {
        return;
    }

    $wanif = get_real_interface($interface);
    $ip4address = find_interface_ip($wanif);
    if ((!is_ipaddrv4($ip4address)) || (is_private_ip($ip4address))) {
        log_error("The interface IPv4 '{$ip4address}' address on interface '{$wanif}' is not public, not configuring 6RD tunnel");
        return false;
    }
    $hexwanv4 = return_hex_ipv4($ip4address);

    if (!is_numeric($wancfg['prefix-6rd-v4plen'])) {
        $wancfg['prefix-6rd-v4plen'] = 0;
    }

    /* create the long prefix notation for math, save the prefix length */
    $rd6prefix = explode("/", $wancfg['prefix-6rd']);
    $rd6prefixlen = $rd6prefix[1];
    $rd6prefix = Net_IPv6::uncompress($rd6prefix[0]);

    /* binary presentation of the prefix for all 128 bits. */
    $rd6prefixbin = convert_ipv6_to_128bit($rd6prefix);

    /* just save the left prefix length bits */
    $rd6prefixbin = substr($rd6prefixbin, 0, $rd6prefixlen);
    /* if the prefix length is not 32 bits we need to shave bits off from the left of the v4 address. */
    $rd6prefixbin .= substr(sprintf("%032b", hexdec($hexwanv4)), $wancfg['prefix-6rd-v4plen'], 32);
    /* fill out the rest with 0's */
    $rd6prefixbin = str_pad($rd6prefixbin, 128, "0", STR_PAD_RIGHT);

    /* convert the 128 bits for the broker address back into a valid IPv6 address */
    $rd6prefix = convert_128bit_to_ipv6($rd6prefixbin);

    $rd6brgw = "{$rd6prefix}{$wancfg['gateway-6rd']}";

    /* XXX: need to extend to support variable prefix size for v4 */
    $stfiface = "{$interface}_stf";
    if (does_interface_exist($stfiface)) {
        legacy_interface_destroy($stfiface);
    }
    legacy_interface_create('stf', $stfiface);
    legacy_interface_flags($stfiface, 'link2');
    if ($wancfg['prefix-6rd-v4plen'] > 0) {
        $rd6prefixlen += intval($wancfg['prefix-6rd-v4plen']);
    } else {
        $rd6prefixlen += 32;
    }
    mwexec("/sbin/ifconfig {$stfiface} inet6 {$rd6prefix}/{$rd6prefixlen}");
    mwexec("/sbin/ifconfig {$stfiface} stfv4br " . escapeshellarg($wancfg['gateway-6rd']));
    if ($wancfg['prefix-6rd-v4plen'] > 0 && $wancfg['prefix-6rd-v4plen'] < 32) {
        mwexec("/sbin/ifconfig {$stfiface} stfv4net {$ip4address}/{$wancfg['prefix-6rd-v4plen']}");
    }

    /* write out a default router file */
    file_put_contents("/tmp/{$wanif}_routerv6", "{$rd6brgw}\n");
    file_put_contents("/tmp/{$wanif}_defaultgwv6", "{$rd6brgw}\n");

    $ip4gateway = get_interface_gateway($interface);
    if (is_ipaddrv4($ip4gateway)) {
        system_host_route($wancfg['gateway-6rd'], $ip4gateway);
    }

    /* configure dependent interfaces */
    if (!file_exists("/var/run/booting")) {
        link_interface_to_track6($interface, "update");
    }

    return 0;
}

function interface_6to4_configure($interface = 'wan', $wancfg)
{
    global $config;

    /* because this is a tunnel interface we can only function
     *  with a public IPv4 address on the interface */

    if (!is_array($wancfg)) {
        return;
    }

    $wanif = get_real_interface($interface);
    $ip4address = find_interface_ip($wanif);
    if ((!is_ipaddrv4($ip4address)) || (is_private_ip($ip4address))) {
        log_error("The interface IPv4 '{$ip4address}' address on interface '{$wanif}' is not public, not configuring 6RD tunnel");
        return false;
    }

    /* create the long prefix notation for math, save the prefix length */
    $stfprefixlen = 16;
    $stfprefix = Net_IPv6::uncompress("2002::");
    $stfarr = explode(":", $stfprefix);
    $v4prefixlen = "0";

    /* we need the hex form of the interface IPv4 address */
    $ip4arr = explode(".", $ip4address);
    $hexwanv4 = "";
    foreach ($ip4arr as $octet) {
        $hexwanv4 .= sprintf("%02x", $octet);
    }

    /* we need the hex form of the broker IPv4 address */
    $ip4arr = explode(".", "192.88.99.1");
    $hexbrv4 = "";
    foreach ($ip4arr as $octet) {
        $hexbrv4 .= sprintf("%02x", $octet);
    }

    /* binary presentation of the prefix for all 128 bits. */
    $stfprefixbin = "";
    foreach ($stfarr as $element) {
        $stfprefixbin .= sprintf("%016b", hexdec($element));
    }
    /* just save the left prefix length bits */
    $stfprefixstartbin = substr($stfprefixbin, 0, $stfprefixlen);

    /* if the prefix length is not 32 bits we need to shave bits off from the left of the v4 address. */
    $stfbrokerbin = substr(sprintf("%032b", hexdec($hexbrv4)), $v4prefixlen, 32);
    $stfbrokerbin = str_pad($stfprefixstartbin . $stfbrokerbin, 128, "0", STR_PAD_RIGHT);

    /* for the local subnet too. */
    $stflanbin = substr(sprintf("%032b", hexdec($hexwanv4)), $v4prefixlen, 32);
    $stflanbin = str_pad($stfprefixstartbin . $stflanbin, 128, "0", STR_PAD_RIGHT);

    /* convert the 128 bits for the broker address back into a valid IPv6 address */
    $stfbrarr = array();
    $stfbrbinarr = array();
    $stfbrbinarr = str_split($stfbrokerbin, 16);
    foreach ($stfbrbinarr as $bin) {
        $stfbrarr[] = dechex(bindec($bin));
    }
    $stfbrgw = Net_IPv6::compress(implode(":", $stfbrarr));

    /* convert the 128 bits for the broker address back into a valid IPv6 address */
    $stflanarr = array();
    $stflanbinarr = array();
    $stflanbinarr = str_split($stflanbin, 16);
    foreach ($stflanbinarr as $bin) {
        $stflanarr[] = dechex(bindec($bin));
    }
    $stflanpr = Net_IPv6::compress(implode(":", $stflanarr));
    $stflanarr[7] = 1;
    $stflan = Net_IPv6::compress(implode(":", $stflanarr));

    $stfiface = "{$interface}_stf";
    if (does_interface_exist($stfiface)) {
        legacy_interface_destroy($stfiface);
    }
    legacy_interface_create('stf', $stfiface);
    legacy_interface_flags($stfiface, 'link2');
    mwexec("/sbin/ifconfig {$stfiface} inet6 {$stflanpr} prefixlen 16");

    /* write out a default router file */
    file_put_contents("/tmp/{$wanif}_routerv6", "{$stfbrgw}");
    file_put_contents("/tmp/{$wanif}_defaultgwv6", "{$stfbrgw}");

    $ip4gateway = get_interface_gateway($interface);
    if (is_ipaddrv4($ip4gateway)) {
        system_host_route('192.88.99.1', $ip4gateway);
    }

    if (!file_exists("/var/run/booting")) {
        link_interface_to_track6($interface, "update");
    }

    return 0;
}

function interface_dhcpv6_configure($interface = 'wan', $wancfg)
{
    global $config;

    if (!is_array($wancfg)) {
        return;
    }

    /* write DUID if override was set */
    if (!empty($config['system']['ipv6duid'])) {
        $temp = str_replace(':', '', $config['system']['ipv6duid']);
        $duid_binstring = pack('H*', $temp);
        $fd = fopen('/var/db/dhcp6c_duid', 'wb');
        if ($fd) {
            fwrite($fd, $duid_binstring);
            fclose($fd);
        }
    }

    $wanif = get_real_interface($interface, "inet6");

    /* accept router advertisements for this interface */
    set_single_sysctl("net.inet6.ip6.accept_rtadv", "1");
    log_error("Accept router advertisements on interface {$wanif} ");
    mwexec("/sbin/ifconfig {$wanif} inet6 accept_rtadv -ifdisabled");

    /* Enable RFC6204w support for IPv6 Customer Edge (CE) router */
    set_single_sysctl("net.inet6.ip6.rfc6204w3", "1");

    /* always kill rtsold in case of reconfigure */
    killbypid("/var/run/rtsold_{$wanif}.pid", 'TERM', true);

    /* fire up rtsold for IPv6 RAs first */
    mwexecf(
        '/usr/sbin/rtsold -p %s -O %s -R %s %s %s',
        array(
            "/var/run/rtsold_{$wanif}.pid",
            "/var/etc/rtsold_{$wanif}_script.sh",
            '/usr/bin/true', /* XXX missing proper script to refresh resolv.conf */
            empty($wancfg['adv_dhcp6_debug']) ? '-d' : '-D',
            $wanif
        )
    );

    if (isset($wancfg['dhcp6sendsolicit'])) {
        mwexec("/var/etc/rtsold_{$wanif}_script.sh");
    }
}

function interface_dhcpv6_prepare($interface = 'wan', $wancfg)
{
    if (!is_array($wancfg)) {
        return;
    }

    $wanif = get_real_interface($interface, "inet6");

    $dhcp6cconf = "interface {$wanif} {\n";

    /* for SLAAC interfaces we do fire off a dhcp6 client for just our name servers */
    if ($wancfg['ipaddrv6'] == "slaac") {
        $dhcp6cconf .= "  information-only;\n";
        $dhcp6cconf .= "  request domain-name-servers;\n";
        $dhcp6cconf .= "  request domain-name;\n";
        $dhcp6cconf .= "  script \"/var/etc/dhcp6c_{$interface}_script.sh\"; # we'd like some nameservers please\n";
        $dhcp6cconf .= "};\n";
    } else {
        /* skip address request if this is set */
        if (!isset($wancfg['dhcp6prefixonly'])) {
            $dhcp6cconf .= "  send ia-na 0; # request stateful address\n";
        }
        if (is_numeric($wancfg['dhcp6-ia-pd-len'])) {
            $dhcp6cconf .= "  send ia-pd 0; # request prefix delegation\n";
        }

        $dhcp6cconf .= "  request domain-name-servers;\n";
        $dhcp6cconf .= "  request domain-name;\n";
        $dhcp6cconf .= "  script \"/var/etc/dhcp6c_{$interface}_script.sh\"; # we'd like some nameservers please\n";

        $dhcp6cconf .= "};\n";

        if (!isset($wancfg['dhcp6prefixonly'])) {
            $dhcp6cconf .= "id-assoc na 0 { };\n";
        }

        if (is_numeric($wancfg['dhcp6-ia-pd-len'])) {
            /* Setup the prefix delegation */
            $dhcp6cconf .= "id-assoc pd 0 {\n";
            $preflen = 64 - $wancfg['dhcp6-ia-pd-len'];
            if (isset($wancfg['dhcp6-ia-pd-send-hint'])) {
                $dhcp6cconf .= "  prefix ::/{$preflen} infinity;\n";
            }
            $iflist = link_interface_to_track6($interface);
            foreach ($iflist as $friendly => $ifcfg) {
                if (is_numeric($ifcfg['track6-prefix-id'])) {
                    $realif = get_real_interface($friendly);
                    $dhcp6cconf .= "  prefix-interface {$realif} {\n";
                    $dhcp6cconf .= "    sla-id {$ifcfg['track6-prefix-id']};\n";
                    $dhcp6cconf .= "    sla-len {$wancfg['dhcp6-ia-pd-len']};\n";
                    $dhcp6cconf .= "  };\n";
                }
            }
            $dhcp6cconf .= "};\n";
        }
    }

    // DHCP6 Config File Advanced
    if ($wancfg['adv_dhcp6_config_advanced']) {
        $dhcp6cconf = DHCP6_Config_File_Advanced($interface, $wancfg, $wanif);
    }

    // DHCP6 Config File Override
    if (!empty($wancfg['adv_dhcp6_config_file_override'])) {
        $dhcp6cfile = $wancfg['adv_dhcp6_config_file_override_path'];
        if (file_exists($dhcp6cfile)) {
            $dhcp6cconf = file_get_contents($dhcp6cfile);
            $dhcp6cconf = DHCP6_Config_File_Substitutions($wancfg, $wanif, $dhcp6cconf);
        } else {
            log_error("DHCP6 config file override does not exist: '{$dhcp6cfile}'");
        }
    }

    if (!@file_put_contents("/var/etc/dhcp6c_{$interface}.conf", $dhcp6cconf)) {
        log_error("Error: cannot open dhcp6c_{$interface}.conf in interface_dhcpv6_configure() for writing.");
    }

    $dhcp6cscript = "#!/bin/sh\n";
    $dhcp6cscript .= "if [ -n '" . (!empty($wancfg['adv_dhcp6_debug']) ? 'debug' : '') . "' ]; then\n";
    $dhcp6cscript .= "\t/usr/bin/logger -t dhcp6c \"dhcp6c \$REASON on {$wanif}\"\n";
    $dhcp6cscript .= "fi\n";
    $dhcp6cscript .= "case \$REASON in\n";
    $dhcp6cscript .= "REQUEST|" . (!empty($wancfg['dhcp6norelease']) ? 'EXIT' : 'RELEASE') . ")\n";
    $dhcp6cscript .= "\t/usr/bin/logger -t dhcp6c \"dhcp6c \$REASON on {$wanif} - running newipv6\"\n";
    $dhcp6cscript .= "\t/usr/local/opnsense/service/configd_ctl.py interface newipv6 {$wanif}\n";
    $dhcp6cscript .= "\t;;\n";
    $dhcp6cscript .= "*)\n";
    $dhcp6cscript .= "\t;;\n";
    $dhcp6cscript .= "esac\n";

    if (!@file_put_contents("/var/etc/dhcp6c_{$interface}_script.sh", $dhcp6cscript)) {
        log_error("Error: cannot open dhcp6c_{$interface}_script.sh in interface_dhcpv6_configure() for writing.");
        return;
    }

    chmod("/var/etc/dhcp6c_{$interface}_script.sh", 0755);

    $dhcp6ccommand = exec_safe(
        "/usr/local/sbin/dhcp6c %s -c %s -p %s %s",
        array(
            '-' . (empty($wancfg['adv_dhcp6_debug']) ? 'd' : 'D' ) . (!empty($wancfg['dhcp6norelease']) ? 'n' : ''),
            "/var/etc/dhcp6c_{$interface}.conf",
            "/var/run/dhcp6c_{$wanif}.pid",
            "{$wanif}"
        )
    );

    $rtsoldscript = "#!/bin/sh\n";
    $rtsoldscript .= "# this file was auto-generated, do not edit\n";
    $rtsoldscript .= "if [ -n \"\${2}\" ]; then\n";
    $rtsoldscript .= "\techo \${2} > /tmp/{$wanif}_routerv6\n";
    $rtsoldscript .= "\techo \${2} > /tmp/{$wanif}_defaultgwv6\n";
    $rtsoldscript .= "fi\n";
    $rtsoldscript .= "if [ -f /var/run/dhcp6c_{$wanif}.pid ]; then\n";
    $rtsoldscript .= "\t/usr/bin/logger -t dhcpd \"RTSOLD script - Sending SIGHUP to dhcp6c for interface {$interface}({$wanif})\"\n";
    $rtsoldscript .= "\t/bin/pkill -HUP -F /var/run/dhcp6c_{$wanif}.pid\n";
    $rtsoldscript .= "else\n";
    $rtsoldscript .= "\t/usr/bin/logger -t dhcpd \"RTSOLD script - Starting dhcp6 client for interface {$interface}({$wanif})\"\n";
    $rtsoldscript .= "\t$dhcp6ccommand\n";
    $rtsoldscript .= "fi\n";

    if (!@file_put_contents("/var/etc/rtsold_{$wanif}_script.sh", $rtsoldscript)) {
        log_error("Error: cannot open rtsold_{$wanif}_script.sh in interface_dhcpv6_configure() for writing.");
        return;
    }

    chmod("/var/etc/rtsold_{$wanif}_script.sh", 0755);
}

function DHCP6_Config_File_Advanced($interface, $wancfg, $wanif)
{
    $send_options = "";
    if ($wancfg['adv_dhcp6_interface_statement_send_options'] != '') {
        $options = preg_split('/\s*,\s*(?=(?:[^"]*"[^"]*")*[^"]*$)/', $wancfg['adv_dhcp6_interface_statement_send_options']);
        foreach ($options as $option) {
            $send_options .= "  send {$option};\n";
        }
    }

    $request_options = "";
    if ($wancfg['adv_dhcp6_interface_statement_request_options'] != '') {
        $options = preg_split('/\s*,\s*(?=(?:[^"]*"[^"]*")*[^"]*$)/', $wancfg['adv_dhcp6_interface_statement_request_options']);
        foreach ($options as $option) {
            $request_options .= "  request {$option};\n";
        }
    }

    $information_only = "";
    if ($wancfg['adv_dhcp6_interface_statement_information_only_enable'] != '') {
        $information_only = "  information-only;\n";
    }

    $script = "  script \"/var/etc/dhcp6c_{$interface}_script.sh\";\n";
    if ($wancfg['adv_dhcp6_interface_statement_script'] != '') {
        $script = "  script \"{$wancfg['adv_dhcp6_interface_statement_script']}\";\n";
    }

    $interface_statement  = "interface {$wanif} {\n";
    $interface_statement .= $send_options;
    $interface_statement .= $request_options;
    $interface_statement .= $information_only;
    $interface_statement .= $script;
    $interface_statement .= "};\n";

    $id_assoc_statement_address = "";
    if ($wancfg['adv_dhcp6_id_assoc_statement_address_enable'] != '') {
        $id_assoc_statement_address .= "id-assoc na ";
        if (is_numeric($wancfg['adv_dhcp6_id_assoc_statement_address_id'])) {
            $id_assoc_statement_address .= "{$wancfg['adv_dhcp6_id_assoc_statement_address_id']}";
        } else {
            $id_assoc_statement_address .= '0';
        }
        $id_assoc_statement_address .= " {\n";

        if (($wancfg['adv_dhcp6_id_assoc_statement_address'] != '') &&
            (is_numeric($wancfg['adv_dhcp6_id_assoc_statement_address_pltime']) ||
                $wancfg['adv_dhcp6_id_assoc_statement_address_pltime'] == 'infinity')
           ) {
            $id_assoc_statement_address .= "  address";
            $id_assoc_statement_address .= " {$wancfg['adv_dhcp6_id_assoc_statement_address']}";
            $id_assoc_statement_address .= " {$wancfg['adv_dhcp6_id_assoc_statement_address_pltime']}";
            if (is_numeric($wancfg['adv_dhcp6_id_assoc_statement_address_vltime']) ||
              $wancfg['adv_dhcp6_id_assoc_statement_address_vltime'] == 'infinity' ) {
                $id_assoc_statement_address .= " {$wancfg['adv_dhcp6_id_assoc_statement_address_vltime']}";
            }
            $id_assoc_statement_address .= ";\n";
        }

        $id_assoc_statement_address  .= "};\n";
    }

    $id_assoc_statement_prefix = "";
    if ($wancfg['adv_dhcp6_id_assoc_statement_prefix_enable'] != '') {
        $id_assoc_statement_prefix .= "id-assoc pd ";
        if (is_numeric($wancfg['adv_dhcp6_id_assoc_statement_prefix_id'])) {
            $id_assoc_statement_prefix .= "{$wancfg['adv_dhcp6_id_assoc_statement_prefix_id']}";
        } else {
            $id_assoc_statement_prefix .= '0';
        }
        $id_assoc_statement_prefix .= " {\n";

        if (($wancfg['adv_dhcp6_id_assoc_statement_prefix'] != '') &&
            (is_numeric($wancfg['adv_dhcp6_id_assoc_statement_prefix_pltime']) ||
            $wancfg['adv_dhcp6_id_assoc_statement_prefix_pltime'] == 'infinity')
           ) {
             $id_assoc_statement_prefix .= "  prefix";
             $id_assoc_statement_prefix .= " {$wancfg['adv_dhcp6_id_assoc_statement_prefix']}";
            $id_assoc_statement_prefix .= " {$wancfg['adv_dhcp6_id_assoc_statement_prefix_pltime']}";
            if ((is_numeric($wancfg['adv_dhcp6_id_assoc_statement_prefix_vltime'])) ||
                ($wancfg['adv_dhcp6_id_assoc_statement_prefix_vltime'] == 'infinity') ) {
                $id_assoc_statement_prefix .= " {$wancfg['adv_dhcp6_id_assoc_statement_prefix_vltime']}";
            }
            $id_assoc_statement_prefix .= ";\n";
        }

        if (is_numeric($wancfg['adv_dhcp6_prefix_interface_statement_sla_id'])) {
            $id_assoc_statement_prefix .= "  prefix-interface {$wanif} {\n";
            $id_assoc_statement_prefix .= "    sla-id {$wancfg['adv_dhcp6_prefix_interface_statement_sla_id']};\n";
            if (($wancfg['adv_dhcp6_prefix_interface_statement_sla_len'] >= 0) &&
                ($wancfg['adv_dhcp6_prefix_interface_statement_sla_len'] <= 128)
            ) {
                $id_assoc_statement_prefix .= "    sla-len {$wancfg['adv_dhcp6_prefix_interface_statement_sla_len']};\n";
            }
            $id_assoc_statement_prefix .= "  };\n";
        }

        $iflist = link_interface_to_track6($interface);

        foreach ($iflist as $friendly => $ifcfg) {
            if (is_numeric($ifcfg['track6-prefix-id'])) {
                $realif = get_real_interface($friendly);
                $id_assoc_statement_prefix .= "  prefix-interface {$realif} {\n";
                $id_assoc_statement_prefix .= "    sla-id {$ifcfg['track6-prefix-id']};\n";
                if (($wancfg['adv_dhcp6_prefix_interface_statement_sla_len'] >= 0) &&
                    ($wancfg['adv_dhcp6_prefix_interface_statement_sla_len'] <= 128)
                ) {
                    $id_assoc_statement_prefix .= "    sla-len {$wancfg['adv_dhcp6_prefix_interface_statement_sla_len']};\n";
                }
                $id_assoc_statement_prefix .= "  };\n";
            }
        }

        $id_assoc_statement_prefix  .= "};\n";
    }

    $authentication_statement = "";
    if (($wancfg['adv_dhcp6_authentication_statement_authname'] != '') &&
        ($wancfg['adv_dhcp6_authentication_statement_protocol'] == 'delayed') ) {
        $authentication_statement .= "authentication {$wancfg['adv_dhcp6_authentication_statement_authname']} {\n";
        $authentication_statement .= "  protocol {$wancfg['adv_dhcp6_authentication_statement_protocol']};\n";
        if (preg_match("/(hmac(-)?md5)||(HMAC(-)?MD5)/", $wancfg['adv_dhcp6_authentication_statement_algorithm'])) {
            $authentication_statement .= "  algorithm {$wancfg['adv_dhcp6_authentication_statement_algorithm']};\n";
        }
        if ($wancfg['adv_dhcp6_authentication_statement_rdm'] == 'monocounter') {
            $authentication_statement .= "  rdm {$wancfg['adv_dhcp6_authentication_statement_rdm']};\n";
        }
        $authentication_statement .= "};\n";
    }

    $key_info_statement = "";
    if (($wancfg['adv_dhcp6_key_info_statement_keyname'] != '') &&
        ($wancfg['adv_dhcp6_key_info_statement_realm'] != '') &&
        (is_numeric($wancfg['adv_dhcp6_key_info_statement_keyid'])) &&
        ($wancfg['adv_dhcp6_key_info_statement_secret'] != '') ) {
        $key_info_statement .= "keyinfo {$wancfg['adv_dhcp6_key_info_statement_keyname']} {\n";
        $key_info_statement .= "  realm \"{$wancfg['adv_dhcp6_key_info_statement_realm']}\";\n";
        $key_info_statement .= "  keyid {$wancfg['adv_dhcp6_key_info_statement_keyid']};\n";
        $key_info_statement .= "  secret \"{$wancfg['adv_dhcp6_key_info_statement_secret']}\";\n";
        if (preg_match("/((([0-9]{4}-)?[0-9]{2}[0-9]{2} )?[0-9]{2}:[0-9]{2})||(foreever)/", $wancfg['adv_dhcp6_key_info_statement_expire'])) {
            $key_info_statement .= "  expire \"{$wancfg['adv_dhcp6_key_info_statement_expire']}\";\n";
        }
        $key_info_statement .= "};\n";
    }

    $dhcp6cconf  = $interface_statement;
    $dhcp6cconf .= $id_assoc_statement_address;
    $dhcp6cconf .= $id_assoc_statement_prefix;
    $dhcp6cconf .= $authentication_statement;
    $dhcp6cconf .= $key_info_statement;

    $dhcp6cconf = DHCP6_Config_File_Substitutions($wancfg, $wanif, $dhcp6cconf);

    return $dhcp6cconf;
}

function DHCP6_Config_File_Substitutions($wancfg, $wanif, $dhcp6cconf)
{
    return DHCP_Config_File_Substitutions($wancfg, $wanif, $dhcp6cconf);
}

function interface_dhcp_configure($interface = 'wan')
{
    global $config;

    $wancfg = $config['interfaces'][$interface];
    $wanif = $wancfg['if'];
    if (empty($wancfg)) {
        $wancfg = array();
    }

    /* generate dhclient_wan.conf */
    $fd = fopen("/var/etc/dhclient_{$interface}.conf", "w");
    if (!$fd) {
        log_error("Error: cannot open dhclient_{$interface}.conf in interface_dhcp_configure() for writing.");
        return;
    }

    if ($wancfg['dhcphostname']) {
        $dhclientconf_hostname = "send dhcp-client-identifier \"{$wancfg['dhcphostname']}\";\n";
        $dhclientconf_hostname .= "\tsend host-name \"{$wancfg['dhcphostname']}\";\n";
    } else {
        $dhclientconf_hostname = "";
    }

    $wanif = get_real_interface($interface);
    if (empty($wanif)) {
        log_error("Invalid interface '{$interface}' in interface_dhcp_configure()");
        return;
    }

    $dhclientconf = "";
    $dhclientconf .= <<<EOD
interface "{$wanif}" {
timeout 60;
retry 15;
select-timeout 0;
initial-interval 1;
  {$dhclientconf_hostname}
  script "/usr/local/sbin/dhclient-script.ext";
EOD;

    if (is_ipaddrv4($wancfg['dhcprejectfrom'])) {
        $dhclientconf .= <<<EOD

  reject {$wancfg['dhcprejectfrom']};
EOD;
    }
    $dhclientconf .= <<<EOD

}

EOD;

    // DHCP Config File Advanced
    if ($wancfg['adv_dhcp_config_advanced']) {
        $dhclientconf = DHCP_Config_File_Advanced($interface, $wancfg, $wanif);
    }

    if (is_ipaddr($wancfg['alias-address'])) {
        $subnetmask = gen_subnet_mask($wancfg['alias-subnet']);
        $dhclientconf .= <<<EOD
alias {
  interface  "{$wanif}";
  fixed-address {$wancfg['alias-address']};
  option subnet-mask {$subnetmask};
}

EOD;
    }

    // DHCP Config File Override
    if (!empty($wancfg['adv_dhcp_config_file_override'])) {
        $dhclientfile = $wancfg['adv_dhcp_config_file_override_path'];
        if (file_exists($dhclientfile)) {
            $dhclientconf = file_get_contents($dhclientfile);
            $dhclientconf = DHCP_Config_File_Substitutions($wancfg, $wanif, $dhclientconf);
        } else {
            log_error("DHCP config file override does not exist: '{$dhclientfile}'");
        }
    }

    fwrite($fd, $dhclientconf);
    fclose($fd);

    /* bring wan interface up before starting dhclient */
    if ($wanif) {
        interfaces_bring_up($wanif);
    } else {
        log_error("Could not bring up {$wanif} interface in interface_dhcp_configure()");
    }

    /* Make sure dhclient is not running */
    kill_dhclient_process($wanif);

    /* fire up dhclient */
    mwexec("/sbin/dhclient -c /var/etc/dhclient_{$interface}.conf {$wanif} > /tmp/{$wanif}_output 2> /tmp/{$wanif}_error_output");
}

function DHCP_Config_File_Advanced($interface, $wancfg, $wanif)
{
    $hostname = "";
    if ($wancfg['dhcphostname'] != '') {
        $hostname = "\tsend host-name \"{$wancfg['dhcphostname']}\";\n";
    }

    /* DHCP Protocol Timings */
    $protocol_timings = array ('adv_dhcp_pt_timeout' => "timeout", 'adv_dhcp_pt_retry' => "retry", 'adv_dhcp_pt_select_timeout' => "select-timeout", 'adv_dhcp_pt_reboot' => "reboot", 'adv_dhcp_pt_backoff_cutoff' => "backoff-cutoff", 'adv_dhcp_pt_initial_interval' => "initial-interval");
    foreach ($protocol_timings as $Protocol_Timing => $PT_Name) {
        $pt_variable = "{$Protocol_Timing}";
        ${$pt_variable} = "";
        if ($wancfg[$Protocol_Timing] != "") {
            ${$pt_variable} = "{$PT_Name} {$wancfg[$Protocol_Timing]};\n";
        }
    }

    $send_options = "";
    if ($wancfg['adv_dhcp_send_options'] != '') {
        $options = preg_split('/\s*,\s*(?=(?:[^"]*"[^"]*")*[^"]*$)/', $wancfg['adv_dhcp_send_options']);
        foreach ($options as $option) {
            $send_options .= "\tsend " . $option . ";\n";
        }
    }

    $request_options = "";
    if ($wancfg['adv_dhcp_request_options'] != '') {
        $request_options = "\trequest {$wancfg['adv_dhcp_request_options']};\n";
    }

    $required_options = "";
    if ($wancfg['adv_dhcp_required_options'] != '') {
        $required_options = "\trequire {$wancfg['adv_dhcp_required_options']};\n";
    }

    $option_modifiers = "";
    if ($wancfg['adv_dhcp_option_modifiers'] != '') {
        $modifiers = preg_split('/\s*,\s*(?=(?:[^"]*"[^"]*")*[^"]*$)/', $wancfg['adv_dhcp_option_modifiers']);
        foreach ($modifiers as $modifier) {
            $option_modifiers .= "\t" . $modifier . ";\n";
        }
    }

    $dhclientconf  = "interface \"{$wanif}\" {\n";
    $dhclientconf .= "\n";
    $dhclientconf .= "# DHCP Protocol Timing Values\n";
    $dhclientconf .= "{$adv_dhcp_pt_timeout}";
    $dhclientconf .= "{$adv_dhcp_pt_retry}";
    $dhclientconf .= "{$adv_dhcp_pt_select_timeout}";
    $dhclientconf .= "{$adv_dhcp_pt_reboot}";
    $dhclientconf .= "{$adv_dhcp_pt_backoff_cutoff}";
    $dhclientconf .= "{$adv_dhcp_pt_initial_interval}";
    $dhclientconf .= "\n";
    $dhclientconf .= "# DHCP Protocol Options\n";
    $dhclientconf .= "{$hostname}";
    $dhclientconf .= "{$send_options}";
    $dhclientconf .= "{$request_options}";
    $dhclientconf .= "{$required_options}";
    $dhclientconf .= "{$option_modifiers}";
    $dhclientconf .= "\n";
    $dhclientconf .= "\tscript \"/usr/local/sbin/dhclient-script.ext\";\n";
    if (is_ipaddrv4($wancfg['dhcprejectfrom'])) {
        $dhclientconf .= "\treject {$wancfg['dhcprejectfrom']};\n";
    }
    $dhclientconf .= "}\n";

    $dhclientconf = DHCP_Config_File_Substitutions($wancfg, $wanif, $dhclientconf);

    return $dhclientconf;
}

function DHCP_Config_File_Substitutions($wancfg, $wanif, $dhclientconf)
{

    /* Apply Interface Substitutions */
    $dhclientconf = str_replace("{interface}", "{$wanif}", $dhclientconf);

    /* Apply Hostname Substitutions */
    $dhclientconf = str_replace("{hostname}", $wancfg['dhcphostname'], $dhclientconf);

    /* Arrays of MAC Address Types, Cases, Delimiters */
    /* ASCII or HEX, Upper or Lower Case, Various Delimiters (none, space, colon, hyphen, period) */
    $various_mac_types      = array("mac_addr_ascii", "mac_addr_hex");
    $various_mac_cases      = array("U", "L");
    $various_mac_delimiters = array("", " ", ":", "-", ".");

    /* Apply MAC Address Substitutions */
    foreach ($various_mac_types as $various_mac_type) {
        foreach ($various_mac_cases as $various_mac_case) {
            foreach ($various_mac_delimiters as $various_mac_delimiter) {
                $res = stripos($dhclientconf, $various_mac_type . $various_mac_case . $various_mac_delimiter);
                if ($res !== false) {
                    /* Get MAC Address as ASCII String With Colon (:) Celimiters */
                    if ("$various_mac_case" == "U") {
                        $dhcpclientconf_mac = strtoupper(get_interface_mac($wanif));
                    }
                    if ("$various_mac_case" == "L") {
                        $dhcpclientconf_mac = strtolower(get_interface_mac($wanif));
                    }

                    if ("$various_mac_type" == "mac_addr_hex") {
                        /* Convert MAC ascii string to HEX with colon (:) delimiters. */
                        $dhcpclientconf_mac = str_replace(":", "", $dhcpclientconf_mac);
                        $dhcpclientconf_mac_hex = "";
                        $delimiter = "";
                        for ($i = 0; $i < strlen($dhcpclientconf_mac); $i++) {
                            $dhcpclientconf_mac_hex .= $delimiter. bin2hex($dhcpclientconf_mac[$i]);
                            $delimiter = ":";
                        }
                        $dhcpclientconf_mac = $dhcpclientconf_mac_hex;
                    }

                    /* MAC Address Delimiter Substitutions */
                    $dhcpclientconf_mac = str_replace(":", $various_mac_delimiter, $dhcpclientconf_mac);

                    /* Apply MAC Address Substitutions */
                    $dhclientconf = str_replace("{" . $various_mac_type . $various_mac_case . $various_mac_delimiter . "}", $dhcpclientconf_mac, $dhclientconf);
                }
            }
        }
    }

    return $dhclientconf;
}

function interfaces_group_setup()
{
    global $config;

    if (!isset($config['ifgroups']['ifgroupentry'])) {
        return;
    }

    foreach ($config['ifgroups']['ifgroupentry'] as $groupar) {
        interface_group_setup($groupar);
    }
}

function interface_group_setup(&$groupname)
{
    global $config;

    if (!is_array($groupname)) {
        return;
    }

    $members = explode(" ", $groupname['members']);
    foreach ($members as $ifs) {
        $realif = get_real_interface($ifs);
        if ($realif) {
            mwexec("/sbin/ifconfig {$realif} group {$groupname['ifname']}");
        }
    }
}

function is_interface_group($if)
{
    global $config;

    if (isset($config['ifgroups']['ifgroupentry'])) {
        foreach ($config['ifgroups']['ifgroupentry'] as $groupentry) {
            if ($groupentry['ifname'] === $if) {
                return true;
            }
        }
    }

    return false;
}

function interface_group_add_member($interface, $groupname)
{
    $interface = get_real_interface($interface);
    mwexec("/sbin/ifconfig {$interface} group " . escapeshellarg($groupname), true);
}

/* convert fxp0 -> wan, etc. */
function convert_real_interface_to_friendly_interface_name($interface = 'wan')
{
    global $config;

    if (!isset($config['interfaces'])) {
        /* some people do trigger this, I don't know why :) */
        return null;
    }

    // search direct
    foreach ($config['interfaces'] as $if => $ifname) {
        if ($if == $interface || $ifname['if'] == $interface) {
            return $if;
        }
    }

    // search related
    foreach ($config['interfaces'] as $if => $ifname) {
        if (get_real_interface($if) == $interface) {
            return $if;
        }

        $int = get_parent_interface($if, true);
        if (is_array($int)) {
            foreach ($int as $iface) {
                if ($iface == $interface) {
                    return $if;
                }
            }
        }
    }

    if ($interface == 'enc0') {
        return 'IPsec';
    }

    return null;
}

/* attempt to resolve interface to friendly descr */
function convert_friendly_interface_to_friendly_descr($interface)
{
    global $config;
    $ifdesc = null;

    switch ($interface) {
        case "l2tp":
            $ifdesc = "L2TP";
            break;
        case "pptp":
            $ifdesc = "PPTP";
            break;
        case "pppoe":
            $ifdesc = "PPPoE";
            break;
        case "openvpn":
            $ifdesc = "OpenVPN";
            break;
        case "enc0":
        case "ipsec":
        case "IPsec":
            $ifdesc = "IPsec";
            break;
        default:
            if (isset($config['interfaces'][$interface])) {
                if (empty($config['interfaces'][$interface]['descr'])) {
                    $ifdesc = strtoupper($interface);
                } else {
                    $ifdesc = strtoupper($config['interfaces'][$interface]['descr']);
                }
                break;
            } elseif (stristr($interface, "_vip")) {
                if (isset($config['virtualip']['vip'])) {
                    foreach ($config['virtualip']['vip'] as $counter => $vip) {
                        if ($vip['mode'] == "carp") {
                            if ($interface == "{$vip['interface']}_vip{$vip['vhid']}") {
                                return "{$vip['subnet']} - {$vip['descr']}";
                            }
                        }
                    }
                }
            } else {
                /* if list */
                $ifdescrs = get_configured_interface_with_descr(false, true);
                foreach ($ifdescrs as $if => $ifname) {
                    if ($if == $interface || $ifname == $interface) {
                        return $ifname;
                    }
                }
            }
            break;
    }

    return $ifdesc;
}

/*
 *  get_parent_interface($interface):
 *      --returns the (real or virtual) parent interface(s) array for a given interface friendly name (i.e. wan)
 *        or virtual interface (i.e. vlan)
 *        (We need array because MLPPP and bridge interfaces have more than one parent.)
 *      -- returns $interface passed in if $interface parent is not found
 *      -- returns empty array if an invalid interface is passed
 *  (Only handles ppps and vlans now.)
 */
function get_parent_interface($interface, $avoidrecurse = false)
{
    global $config;

    $parents = array();
    //Check that we got a valid interface passed
    $realif = get_real_interface($interface);
    if ($realif == null) {
        return $parents;
    }

    // If we got a real interface, find it's friendly assigned name
    if ($interface == $realif && $avoidrecurse == false) {
        $interface = convert_real_interface_to_friendly_interface_name($interface);
    }

    if (!empty($interface) && isset($config['interfaces'][$interface])) {
        $ifcfg = $config['interfaces'][$interface];
        switch ($ifcfg['ipaddr']) {
            case "ppp":
            case "pppoe":
            case "pptp":
            case "l2tp":
                if (empty($parents)) {
                    if (isset($config['ppps']['ppp'])) {
                        foreach ($config['ppps']['ppp'] as $pppidx => $ppp) {
                            if ($ifcfg['if'] == $ppp['if']) {
                                $ports = explode(',', $ppp['ports']);
                                foreach ($ports as $pid => $parent_if) {
                                    $parents[$pid] = get_real_interface($parent_if);
                                }
                                break;
                            }
                        }
                    }
                }
                break;
            case "dhcp":
            case "static":
            default:
                // Handle _vlans
                if (stristr($realif, "_vlan")) {
                    if (isset($config['vlans']['vlan'])) {
                        foreach ($config['vlans']['vlan'] as $vlanidx => $vlan) {
                            if ($ifcfg['if'] == $vlan['vlanif']) {
                                $parents[0] = $vlan['if'];
                                break;
                            }
                        }
                    }
                }
                break;
        }
    }

    if (empty($parents)) {
        $parents[0] = $realif;
    }

    return $parents;
}


function interface_get_wireless_base($wlif)
{
    if (!stristr($wlif, "_wlan")) {
        return $wlif;
    } else {
        return substr($wlif, 0, stripos($wlif, "_wlan"));
    }
}

function interface_get_wireless_clone($wlif)
{
    if (!stristr($wlif, "_wlan")) {
        return $wlif . "_wlan0";
    } else {
        return $wlif;
    }
}

function get_real_interface($interface = "wan", $family = "all", $flush = true)
{
    global $config;

    $wanif = null;

    switch ($interface) {
        case "openvpn":
        case "ppp":
            $wanif = $interface;
            break;
        case "ipsec":
        case "enc0":
            $wanif = "enc0";
            break;
        default:
            if (empty($config['interfaces'][$interface])) {
                // leftover from legacy code, it's not a very bright idea to use the same function call
                // for both virtual as real interface names. does_interface_exist() is quite expensive.
                if (does_interface_exist($interface, $flush)) {
                    $wanif = $interface;
                }
                break;
            }

            $cfg = &config_read_array('interfaces', $interface);

            if ($family == "inet6") {
                if (isset($cfg['ipaddrv6'])) {
                    $ipaddrv6 = $cfg['ipaddrv6'];
                } else {
                    $ipaddrv6 = "";
                }
                switch ($ipaddrv6) {
                    case "6rd":
                    case "6to4":
                        $wanif = "{$interface}_stf";
                        break;
                    case 'pppoe':
                    case 'ppp':
                    case 'l2tp':
                    case 'pptp':
                        if (isset($cfg['wireless']) || match_wireless_interface($cfg['if'])) {
                            $wanif = interface_get_wireless_clone($cfg['if']);
                        } else {
                            $wanif = $cfg['if'];
                        }
                        break;
                    default:
                        switch ($cfg['ipaddr']) {
                            case 'pppoe':
                            case 'ppp':
                            case 'l2tp':
                            case 'pptp':
                                if (isset($cfg['dhcp6usev4iface'])) {
                                    $wanif = $cfg['if'];
                                } else {
                                    $parents = get_parent_interface($interface);
                                    if (!empty($parents[0])) {
                                        $wanif = $parents[0];
                                    } else {
                                        $wanif = $cfg['if'];
                                    }
                                }
                                break;
                            default:
                                if (isset($cfg['wireless']) || match_wireless_interface($cfg['if'])) {
                                    $wanif = interface_get_wireless_clone($cfg['if']);
                                } else {
                                    $wanif = $cfg['if'];
                                }
                                break;
                        }
                        break;
                }
            } else {
                // Wireless cloned NIC support (FreeBSD 8+)
                // interface name format: $parentnic_wlanparentnic#
                // example: ath0_wlan0
                if (isset($cfg['wireless']) || match_wireless_interface($cfg['if'])) {
                    $wanif = interface_get_wireless_clone($cfg['if']);
                } else {
                    $wanif = $cfg['if'];
                }
            }
            break;
    }

    return $wanif;
}

/* Guess the physical interface by providing a IP address */
function guess_interface_from_ip($ipaddress)
{
    if (!is_ipaddr($ipaddress)) {
        return false;
    }
    if (is_ipaddrv4($ipaddress)) {
        /* create a route table we can search */
        exec("/usr/bin/netstat -rnWf inet", $output, $ret);
        foreach ($output as $line) {
            if (preg_match("/^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+\/[0-9]+[ ]+link[#]/", $line)) {
                $fields = preg_split("/[ ]+/", $line);
                if (ip_in_subnet($ipaddress, $fields[0])) {
                    return $fields[5];
                }
            }
        }
    }
    /* FIXME: This works from cursory testing, regexp might need fine tuning */
    if (is_ipaddrv6($ipaddress)) {
        /* create a route table we can search */
        exec("/usr/bin/netstat -rnWf inet6", $output, $ret);
        foreach ($output as $line) {
            if (preg_match("/[0-9a-f]+[:]+[0-9a-f]+[:]+[\/][0-9]+/", $line)) {
                $fields = preg_split("/[ ]+/", $line);
                if (ip_in_subnet($ipaddress, $fields[0])) {
                    return $fields[5];
                }
            }
        }
    }
    $ret = exec_command("/sbin/route -n get {$ipaddress} | /usr/bin/awk '/interface/ { print \$2; };'");
    if (empty($ret)) {
        return false;
    }
    return $ret;
}

function link_interface_to_track6($int, $action = '')
{
    global $config;

    if (empty($int)) {
        return;
    }

    if (isset($config['interfaces'])) {
        $list = array();
        foreach (legacy_config_get_interfaces(array("virtual" => false)) as $ifname => $ifcfg) {
            if (!isset($ifcfg['enable'])) {
                continue;
            }
            if (!empty($ifcfg['ipaddrv6']) && $ifcfg['track6-interface'] == $int) {
                if ($action == "update") {
                    interface_track6_configure($ifname, $ifcfg);
                } elseif ($action == "") {
                    $list[$ifname] = $ifcfg;
                }
            }
        }
        return $list;
    }
}

function link_interface_to_vips($int, $action = '')
{
    global $config;

    if (isset($config['virtualip']['vip'])) {
        $result = array();
        foreach ($config['virtualip']['vip'] as $vip) {
            if ($int == $vip['interface']) {
                if ($action == 'update') {
                    interfaces_vips_configure($int);
                } else {
                    $result[] = $vip;
                }
            }
        }
        return $result;
    }
}

/****f* interfaces/link_interface_to_bridge
 * NAME
 *   link_interface_to_bridge - Finds out a bridge group for an interface
 * INPUTS
 *   $ip
 * RESULT
 *   bridge[0-99]
 ******/
function link_interface_to_bridge($int)
{
    global $config;

    if (isset($config['bridges']['bridged'])) {
        foreach ($config['bridges']['bridged'] as $bridge) {
            if (in_array($int, explode(',', $bridge['members']))) {
                return "{$bridge['bridgeif']}";
            }
        }
    }
}

function link_interface_to_group($int)
{
    global $config;

    $result = array();

    if (isset($config['ifgroups']['ifgroupentry'])) {
        foreach ($config['ifgroups']['ifgroupentry'] as $group) {
            if (in_array($int, explode(" ", $group['members']))) {
                $result[$group['ifname']] = $int;
            }
        }
    }

    return $result;
}

function link_interface_to_gre($interface)
{
    global $config;

    $result = array();

    if (isset($config['gres']['gre'])) {
        foreach ($config['gres']['gre'] as $gre) {
            if ($gre['if'] == $interface) {
                $result[] = $gre;
            }
        }
    }

    return $result;
}

function link_interface_to_gif($interface)
{
    global $config;

    $result = array();

    if (isset($config['gifs']['gif'])) {
        foreach ($config['gifs']['gif'] as $gif) {
            if ($gif['if'] == $interface) {
                $result[] = $gif;
            }
        }
    }

    return $result;
}

/*
 * find_interface_ip($interface): return the interface ip (first found)
 */
function find_interface_ip($interface)
{
    $interface = trim($interface);
    if (does_interface_exist($interface)) {
        $ifinfo = legacy_get_interface_addresses($interface);
        if (isset($ifinfo['ipaddr'])) {
            return $ifinfo['ipaddr'];
        }
    }
    return null;
}

/*
 * find_interface_ipv6($interface): return the interface ip (first found)
 */
function find_interface_ipv6($interface)
{
    // a bit obscure, why should this be different then find_interface_ip?
    $interface = get_real_interface(trim($interface));
    if (does_interface_exist($interface)) {
        $ifinfo = legacy_get_interface_addresses($interface);
        if (isset($ifinfo['ipaddr6'])) {
            return $ifinfo['ipaddr6'];
        }
    }
    return null;
}

/*
 * find_interface_ipv6_ll($interface): return the interface ipv6 link local (first found)
 */
function find_interface_ipv6_ll($interface)
{
    $interface = trim($interface);
    if (does_interface_exist($interface)) {
        $ifinfo = legacy_getall_interface_addresses($interface);
        foreach ($ifinfo as $line) {
            if (strstr($line, ":")) {
                $parts = explode("/", $line);
                if (is_linklocal($parts[0])) {
                    return $parts[0];
                }
            }
        }
    }
    return null;
}

function find_interface_subnet($interface)
{
    $interface = trim($interface);
    if (does_interface_exist($interface)) {
        $ifinfo = legacy_get_interface_addresses($interface);
        if (isset($ifinfo['subnetbits'])) {
            return $ifinfo['subnetbits'];
        }
    }
    return null;
}

function find_interface_subnet6($interface)
{
    $interface = trim($interface);
    if (does_interface_exist($interface)) {
        $ifinfo = legacy_get_interface_addresses($interface);
        if (isset($ifinfo['subnetbits6'])) {
            return $ifinfo['subnetbits6'];
        }
    }
    return null;
}

function ip_in_interface_alias_subnet($interface, $ipalias)
{
    global $config;

    if (empty($interface) || !is_ipaddr($ipalias)) {
        return false;
    }
    if (isset($config['virtualip']['vip'])) {
        foreach ($config['virtualip']['vip'] as $vip) {
            switch ($vip['mode']) {
                case "ipalias":
                    if ($vip['interface'] <> $interface) {
                        break;
                    }
                    $subnet = is_ipaddrv6($ipalias) ? gen_subnetv6($vip['subnet'], $vip['subnet_bits']) : gen_subnet($vip['subnet'], $vip['subnet_bits']);
                    if (ip_in_subnet($ipalias, $subnet . "/" . $vip['subnet_bits'])) {
                        return true;
                    }
                    break;
            }
        }
    }

    return false;
}

function get_interface_ip($interface = "wan")
{
    $realif = get_failover_interface($interface);
    if (!$realif) {
        if (strstr($interface, "_vip")) {
            return get_configured_carp_interface_list($interface);
        } else {
            return null;
        }
    }

    $curip = find_interface_ip($realif);
    if ($curip && is_ipaddr($curip) && ($curip != "0.0.0.0")) {
        return $curip;
    } else {
        return null;
    }
}

function get_interface_ipv6($interface = "wan", $flush = false)
{
    global $config;

    $realif = get_failover_interface($interface, "inet6");
    if (!$realif) {
        if (strstr($interface, "_vip")) {
            return get_configured_carp_interface_list($interface, "inet6");
        } else {
            return null;
        }
    }

    /*
     * NOTE: On the case when only the prefix is requested,
     * the communication on WAN will be done over link-local.
     */
    if (isset($config['interfaces'][$interface])) {
        switch ($config['interfaces'][$interface]['ipaddr']) {
            case 'pppoe':
            case 'l2tp':
            case 'pptp':
            case 'ppp':
                if ($config['interfaces'][$interface]['ipaddrv6'] == 'dhcp6') {
                    $realif = get_real_interface($interface, 'inet6');
                }
                break;
        }
        if (isset($config['interfaces'][$interface]['dhcp6prefixonly'])) {
            $curip = find_interface_ipv6_ll($realif, $flush);
            if ($curip && is_ipaddrv6($curip) && ($curip != "::")) {
                return $curip;
            }
        }
    }

    $curip = find_interface_ipv6($realif, $flush);
    if ($curip && is_ipaddrv6($curip) && ($curip != "::")) {
        return $curip;
    } else {
        return null;
    }
}

function get_interface_linklocal($interface = "wan")
{
    $realif = get_failover_interface($interface, "inet6");
    if (!$realif) {
        if (strstr($interface, "_vip")) {
            list($interface, $vhid) = explode("_vip", $interface);
            $realif = get_real_interface($interface);
        } else {
            return null;
        }
    }

    $curip = find_interface_ipv6_ll($realif);
    if ($curip && is_ipaddrv6($curip) && ($curip != "::")) {
        return $curip;
    } else {
        return null;
    }
}

function get_interface_subnet($interface = "wan")
{
    $realif = get_real_interface($interface);
    if (!$realif) {
        if (strstr($interface, "_vip")) {
            list($interface, $vhid) = explode("_vip", $interface);
            $realif = get_real_interface($interface);
        } else {
            return null;
        }
    }

    $cursn = find_interface_subnet($realif);
    if (!empty($cursn)) {
        return $cursn;
    }

    return null;
}

function get_interface_subnetv6($interface = "wan")
{
    global $config;

    $realif = get_real_interface($interface, "inet6");
    if (!$realif) {
        if (strstr($interface, "_vip")) {
            list($interface, $vhid) = explode("_vip", $interface);
            $realif = get_real_interface($interface);
        } else {
            return null;
        }
    }

    if (does_interface_exist($realif)) {
        $ifinfo = legacy_get_interface_addresses($realif);
        if (isset($ifinfo['subnetbits6'])) {
            return $ifinfo['subnetbits6'];
        }
    }

    return null;
}

/* return true if interface has a gateway */
function interface_has_gateway($friendly)
{
    global $config;

    if (!empty($config['interfaces'][$friendly])) {
        $ifname = &config_read_array('interfaces', $friendly);
        switch ($ifname['ipaddr']) {
            case "dhcp":
            case "pppoe":
            case "pptp":
            case "l2tp":
            case "ppp":
                return true;
                break;
            default:
                if (substr($ifname['if'], 0, 4) ==  "ovpn") {
                    return true;
                }
                $tunnelif = substr($ifname['if'], 0, 3);
                if ($tunnelif == "gif" || $tunnelif == "gre") {
                    return true;
                }
                if (!empty($ifname['gateway'])) {
                    return true;
                }
                break;
        }
    }

    return false;
}

/* return true if interface has a gateway */
function interface_has_gatewayv6($friendly)
{
    global $config;

    if (!empty($config['interfaces'][$friendly])) {
        $ifname = &config_read_array('interfaces', $friendly);
        if (isset($ifname['ipaddrv6'])) {
            $ipaddrv6 = $ifname['ipaddrv6'];
        } else {
            $ipaddrv6 = "";
        }
        switch ($ipaddrv6) {
            case "slaac":
            case "dhcp6":
            case "6to4":
            case "6rd":
                return true;
                break;
            default:
                if (substr($ifname['if'], 0, 4) ==  "ovpn") {
                    return true;
                }
                $tunnelif = substr($ifname['if'], 0, 3);
                if ($tunnelif == "gif" || $tunnelif == "gre") {
                    return true;
                }
                if (!empty($ifname['gatewayv6'])) {
                    return true;
                }
                break;
        }
    }

    return false;
}

/****f* interfaces/is_interface_wireless
 * NAME
 *   is_interface_wireless - Returns if an interface is wireless
 * RESULT
 *   $tmp       - Returns if an interface is wireless
 ******/
function is_interface_wireless($interface)
{
    global $config;

    $friendly = convert_real_interface_to_friendly_interface_name($interface);
    if (!isset($config['interfaces'][$friendly]['wireless'])) {
        if (match_wireless_interface($interface)) {
            config_read_array('interfaces', $friendly, 'wireless');
            return true;
        }
        return false;
    }

    return true;
}

function get_interface_mac($interface)
{
    $macinfo = legacy_get_interface_addresses($interface);

    return $macinfo["macaddr"];
}

function get_vip_descr($ipaddress)
{
    global $config;

    foreach ($config['virtualip']['vip'] as $vip) {
        if ($vip['subnet'] == $ipaddress) {
            return ($vip['descr']);
        }
    }

    return '';
}

function interfaces_staticarp_configure($if)
{
    global $config;

    $ifcfg = $config['interfaces'][$if];
    if (empty($ifcfg['if']) || !isset($ifcfg['enable'])) {
        return;
    }

    if (isset($config['dhcpd'][$if]['staticarp'])) {
        mwexecf('/sbin/ifconfig %s staticarp', array($ifcfg['if']));
        mwexecf('/usr/sbin/arp -d -i %s  -a > /dev/null 2>&1', array($ifcfg['if']));
        if (isset($config['dhcpd'][$if]['staticmap'])) {
            foreach ($config['dhcpd'][$if]['staticmap'] as $arpent) {
                if (isset($arpent['ipaddr'])) {
                    mwexecf(
                        '/usr/sbin/arp -s %s %s',
                        array($arpent['ipaddr'], $arpent['mac'])
                    );
                }
            }
        }
    } else {
        mwexecf('/sbin/ifconfig %s -staticarp', array($ifcfg['if']));
        mwexecf('/usr/sbin/arp -d -i %s  -a > /dev/null 2>&1', array($ifcfg['if']));
        if (isset($config['dhcpd'][$if]['staticmap'])) {
            foreach ($config['dhcpd'][$if]['staticmap'] as $arpent) {
                if (isset($arpent['arp_table_static_entry'])) {
                    if (isset($arpent['ipaddr'])) {
                        mwexecf(
                            '/usr/sbin/arp -s %s %s',
                            array($arpent['ipaddr'], $arpent['mac'])
                        );
                    }
                }
            }
        }
    }
}

function get_failover_interface($interface, $family = "all")
{
    global $config;

    /* shortcut to get_real_interface if we find it in the config */
    if (isset($config['interfaces'][$interface])) {
        return get_real_interface($interface, $family);
    }

    /* compare against gateway groups */
    $a_groups = return_gateway_groups_array();
    if (isset($a_groups[$interface])) {
        /* we found a gateway group, fetch the interface or vip */
        if ($a_groups[$interface][0]['vip'] <> "") {
            return $a_groups[$interface][0]['vip'];
        } else {
            return $a_groups[$interface][0]['int'];
        }
    }
    /* fall through to get_real_interface */
    /* XXX: Really needed? */
    return get_real_interface($interface, $family);
}

//returns interface information
function get_interfaces_info()
{
    global $config;
    $result = array();
    $all_intf_details = legacy_interfaces_details();
    $ifup = legacy_interface_listget('up');
    $all_intf_stats = legacy_interface_stats();
    foreach (legacy_config_get_interfaces(array("virtual" => false)) as $ifdescr => $ifcnf) {
        $ifinfo = array();
        $ifinfo['if'] = get_real_interface($ifdescr);
        $ifinfo['status'] = (is_array($ifup) && in_array($ifinfo['if'], $ifup)) ? 'up' : 'down';
        if (!empty($all_intf_details[$ifinfo['if']])) {
            if (in_array($all_intf_details[$ifinfo['if']]['status'], array('active', 'running'))) {
                $all_intf_details[$ifinfo['if']]['status'] = $ifinfo['status'];
            }
            $ifinfo = array_merge($ifinfo, $all_intf_details[$ifinfo['if']]);
        }

        if (!empty($ifinfo['ipv6'])) {
            foreach ($ifinfo['ipv6'] as $ipv6addr) {
                if (!empty($ipv6addr['link-local'])) {
                    $ifinfo['linklocal'] = $ipv6addr['ipaddr'];
                } elseif (empty($ifinfo['ipaddrv6'])) {
                    $ifinfo['ipaddrv6'] = $ipv6addr['ipaddr'];
                    $ifinfo['subnetv6'] = $ipv6addr['subnetbits'];
                }
            }
        }
        if (!empty($ifinfo['ipv4'])) {
            $ifinfo['ipaddr'] = $ifinfo['ipv4'][0]['ipaddr'];
            $ifinfo['subnet'] = $ifinfo['ipv4'][0]['subnetbits'];
        }

        $ifinfotmp = $all_intf_stats[$ifinfo['if']];
        $ifinfo['inerrs'] = $ifinfotmp['input errors'];
        $ifinfo['outerrs'] = $ifinfotmp['output errors'];
        $ifinfo['collisions'] = $ifinfotmp['collisions'];

        $ifconfiginfo = "";
        $link_type = $config['interfaces'][$ifdescr]['ipaddr'];
        switch ($link_type) {
            /* DHCP? -> see if dhclient is up */
            case "dhcp":
                /* see if dhclient is up */
                if (find_dhclient_process($ifinfo['if']) != 0) {
                    $ifinfo['dhcplink'] = "up";
                } else {
                    $ifinfo['dhcplink'] = "down";
                }

                break;
            /* PPPoE/PPTP/L2TP interface? -> get status from virtual interface */
            case "pppoe":
            case "pptp":
            case "l2tp":
                if ($ifinfo['status'] == "up") {
                    /* XXX get PPPoE link status for dial on demand */
                    $ifinfo["{$link_type}link"] = "up";
                } else {
                    $ifinfo["{$link_type}link"] = "down";
                }
                break;
            /* PPP interface? -> get uptime for this session and cumulative uptime from the persistent log file in conf */
            case "ppp":
                if ($ifinfo['status'] == "up") {
                    $ifinfo['ppplink'] = "up";
                } else {
                    $ifinfo['ppplink'] = "down";
                }

                if (empty($ifinfo['status'])) {
                    $ifinfo['status'] = "down";
                }

                if (isset($config['ppps']['ppp'])) {
                    foreach ($config['ppps']['ppp'] as $pppid => $ppp) {
                        if ($config['interfaces'][$ifdescr]['if'] == $ppp['if']) {
                            break;
                        }
                    }
                }
                $dev = $ppp['ports'];
                if ($config['interfaces'][$ifdescr]['if'] != $ppp['if'] || empty($dev)) {
                    break;
                }
                if (!file_exists($dev)) {
                    $ifinfo['nodevice'] = 1;
                    $ifinfo['pppinfo'] = $dev . " " . gettext("device not present! Is the modem attached to the system?");
                }

                $usbmodemoutput = array();
                exec("usbconfig", $usbmodemoutput);
                $mondev = "/tmp/3gstats.{$ifdescr}";
                if (file_exists($mondev)) {
                    $cellstats = file($mondev);
                    /* skip header */
                    $a_cellstats = explode(",", $cellstats[1]);
                    if (preg_match("/huawei/i", implode("\n", $usbmodemoutput))) {
                        $ifinfo['cell_rssi'] = huawei_rssi_to_string($a_cellstats[1]);
                        $ifinfo['cell_mode'] = huawei_mode_to_string($a_cellstats[2], $a_cellstats[3]);
                        $ifinfo['cell_simstate'] = huawei_simstate_to_string($a_cellstats[10]);
                        $ifinfo['cell_service'] = huawei_service_to_string(trim($a_cellstats[11]));
                    }
                    if (preg_match("/zte/i", implode("\n", $usbmodemoutput))) {
                        $ifinfo['cell_rssi'] = zte_rssi_to_string($a_cellstats[1]);
                        $ifinfo['cell_mode'] = zte_mode_to_string($a_cellstats[2], $a_cellstats[3]);
                        $ifinfo['cell_simstate'] = zte_simstate_to_string($a_cellstats[10]);
                        $ifinfo['cell_service'] = zte_service_to_string(trim($a_cellstats[11]));
                    }
                    $ifinfo['cell_upstream'] = $a_cellstats[4];
                    $ifinfo['cell_downstream'] = trim($a_cellstats[5]);
                    $ifinfo['cell_sent'] = $a_cellstats[6];
                    $ifinfo['cell_received'] = trim($a_cellstats[7]);
                    $ifinfo['cell_bwupstream'] = $a_cellstats[8];
                    $ifinfo['cell_bwdownstream'] = trim($a_cellstats[9]);
                }
                // Calculate cumulative uptime for PPP link. Useful for connections that have per minute/hour contracts so you don't go over!
                if (isset($ppp['uptime'])) {
                    $ifinfo['ppp_uptime_accumulated'] = "(" . get_ppp_uptime($ifinfo['if']) . ")";
                }
                break;
            default:
                break;
        }

        if (file_exists("/var/run/{$link_type}_{$ifdescr}.pid")) {
            $sec = trim(`/usr/local/sbin/ppp-uptime.sh {$ifinfo['if']}`);
            $ifinfo['ppp_uptime'] = convert_seconds_to_hms($sec);
        }

        if ($ifinfo['status'] == "up") {
            $wifconfiginfo = array();
            if (is_interface_wireless($ifdescr)) {
                exec("/sbin/ifconfig {$ifinfo['if']} list sta", $wifconfiginfo);
                array_shift($wifconfiginfo);
            }
            foreach ($wifconfiginfo as $ici) {
                $elements = preg_split("/[ ]+/i", $ici);
                if ($elements[0] != "") {
                    $ifinfo['bssid'] = $elements[0];
                }
                if ($elements[3] != "") {
                    $ifinfo['rate'] = $elements[3];
                }
                if ($elements[4] != "") {
                    $ifinfo['rssi'] = $elements[4];
                }

            }
            /* lookup the gateway */
            if (interface_has_gateway($ifdescr)) {
                $ifinfo['gateway'] = get_interface_gateway($ifdescr);
                $ifinfo['gatewayv6'] = get_interface_gateway_v6($ifdescr);
            }
        }

        $bridge = "";
        $bridge = link_interface_to_bridge($ifdescr);
        if ($bridge) {
            $bridge_text = `/sbin/ifconfig {$bridge}`;
            if (stristr($bridge_text, "blocking") <> false) {
                $ifinfo['bridge'] = "<b><font color='red'>" . gettext("blocking") . "</font></b> - " . gettext("check for ethernet loops");
                $ifinfo['bridgeint'] = $bridge;
            } elseif (stristr($bridge_text, "learning") <> false) {
                $ifinfo['bridge'] = gettext("learning");
                $ifinfo['bridgeint'] = $bridge;
            } elseif (stristr($bridge_text, "forwarding") <> false) {
                $ifinfo['bridge'] = gettext("forwarding");
                $ifinfo['bridgeint'] = $bridge;
            }
        }
        $result[$ifdescr] = $ifinfo;
    }

    return $result;
}

function convert_seconds_to_hms($sec)
{
    $min=$hrs=0;
    if ($sec != 0) {
        $min = floor($sec/60);
        $sec %= 60;
    }
    if ($min != 0) {
        $hrs = floor($min/60);
        $min %= 60;
    }
    if ($sec < 10) {
        $sec = "0" . $sec;
    }
    if ($min < 10) {
        $min = "0" . $min;
    }
    if ($hrs < 10) {
        $hrs = "0" . $hrs;
    }
    $result = $hrs.":".$min.":".$sec;
    return $result;
}

function huawei_rssi_to_string($rssi)
{
    $dbm = array();
    $i = 0;
    $dbstart = -113;
    while ($i < 32) {
        $dbm[$i] = $dbstart + ($i * 2);
        $i++;
    }
    $percent = round(($rssi / 31) * 100);
    $string = "rssi:{$rssi} level:{$dbm[$rssi]}dBm percent:{$percent}%";
    return $string;
}

function huawei_mode_to_string($mode, $submode)
{
    $modes[0] = "None";
    $modes[1] = "AMPS";
    $modes[2] = "CDMA";
    $modes[3] = "GSM/GPRS";
    $modes[4] = "HDR";
    $modes[5] = "WCDMA";
    $modes[6] = "GPS";

    $submodes[0] = "No Service";
    $submodes[1] = "GSM";
    $submodes[2] = "GPRS";
    $submodes[3] = "EDGE";
    $submodes[4] = "WCDMA";
    $submodes[5] = "HSDPA";
    $submodes[6] = "HSUPA";
    $submodes[7] = "HSDPA+HSUPA";
    $submodes[8] = "TD-SCDMA";
    $submodes[9] = "HSPA+";
    $string = "{$modes[$mode]}, {$submodes[$submode]} Mode";
    return $string;
}

function huawei_service_to_string($state)
{
    $modes[0] = "No";
    $modes[1] = "Restricted";
    $modes[2] = "Valid";
    $modes[3] = "Restricted Regional";
    $modes[4] = "Powersaving";
    $string = "{$modes[$state]} Service";
    return $string;
}

function huawei_simstate_to_string($state)
{
    $modes[0] = "Invalid SIM/locked";
    $modes[1] = "Valid SIM";
    $modes[2] = "Invalid SIM CS";
    $modes[3] = "Invalid SIM PS";
    $modes[4] = "Invalid SIM CS/PS";
    $modes[255] = "Missing SIM";
    $string = "{$modes[$state]} State";
    return $string;
}

function zte_rssi_to_string($rssi)
{
    return huawei_rssi_to_string($rssi);
}

function zte_mode_to_string($mode, $submode)
{
    $modes[0] = "No Service";
    $modes[1] = "Limited Service";
    $modes[2] = "GPRS";
    $modes[3] = "GSM";
    $modes[4] = "UMTS";
    $modes[5] = "EDGE";
    $modes[6] = "HSDPA";

    $submodes[0] = "CS_ONLY";
    $submodes[1] = "PS_ONLY";
    $submodes[2] = "CS_PS";
    $submodes[3] = "CAMPED";
    $string = "{$modes[$mode]}, {$submodes[$submode]} Mode";
    return $string;
}

function zte_service_to_string($state)
{
    $modes[0] = "Initializing";
    $modes[1] = "Network Lock error";
    $modes[2] = "Network Locked";
    $modes[3] = "Unlocked or correct MCC/MNC";
    $string = "{$modes[$state]} Service";
    return $string;
}

function zte_simstate_to_string($state)
{
    $modes[0] = "No action";
    $modes[1] = "Network lock";
    $modes[2] = "(U)SIM card lock";
    $modes[3] = "Network Lock and (U)SIM card Lock";
    $string = "{$modes[$state]} State";
    return $string;
}

/* Compute the total uptime from the ppp uptime log file in the conf directory */
function get_ppp_uptime($port)
{
    if (file_exists("/conf/{$port}.log")) {
        $saved_time = file_get_contents("/conf/{$port}.log");
        $uptime_data = explode("\n", $saved_time);
        $sec=0;
        foreach ($uptime_data as $upt) {
            $sec += substr($upt, 1 + strpos($upt, " "));
        }
        return convert_seconds_to_hms($sec);
    } else {
        $total_time = gettext("No history data found!");
        return $total_time;
    }
}

/****f* legacy/is_ipaddr_configured
 * NAME
 *   is_ipaddr_configured
 * INPUTS
 *   IP Address to check.
 * RESULT
 *   returns true if the IP Address is
 *   configured and present on this device.
*/
function is_ipaddr_configured($ipaddr, $ignore_if = '')
{
    global $config;
    $if = get_real_interface($ignore_if);
    $interface_list_ips = get_configured_ip_addresses();

    if (empty($interface_list_ips[$ipaddr]) || $interface_list_ips[$ipaddr] == $if) {
        return false;
    } else {
        return true;
    }
}

/*
 * get_carp_interface_status($carpinterface): returns the status of a carp ip
 */
function get_carp_interface_status($carpinterface)
{
    $carp_query = '';

    /* XXX: Need to fidn a better way for this! */
    list ($interface, $vhid) = explode("_vip", $carpinterface);
    $interface = get_real_interface($interface);
    exec("/sbin/ifconfig $interface | /usr/bin/grep -v grep | /usr/bin/grep carp: | /usr/bin/grep 'vhid {$vhid}'", $carp_query);
    foreach($carp_query as $int) {
        if (stristr($int, "MASTER")) {
            return gettext("MASTER");
        } elseif (stristr($int, "BACKUP")) {
            return gettext("BACKUP");
        } elseif(stristr($int, "INIT")) {
            return gettext("INIT");
        }
    }
    return;
}
