/*
*******************************************************************************
* Copyright (C) 2015, International Business Machines
* Corporation and others.  All Rights Reserved.
*******************************************************************************
* smallintformatter.h
*
* created on: 2015jan06
* created by: Travis Keep
*/

#ifndef __SMALLINTFORMATTER_H__
#define __SMALLINTFORMATTER_H__

#include "unicode/uobject.h"
#include "unicode/utypes.h"

U_NAMESPACE_BEGIN

class UnicodeString;

/**
 * A representation an acceptable range of digit counts for integers.
 */
class U_I18N_API IntDigitCountRange : public UMemory {
public:
    /**
     * No constraints: 0 up to INT32_MAX
     */
    IntDigitCountRange() : fMin(0), fMax(INT32_MAX) { }
    IntDigitCountRange(int32_t min, int32_t max);
    int32_t pin(int32_t digitCount) const;
    int32_t getMax() const { return fMax; }
    int32_t getMin() const { return fMin; }
private:
    int32_t fMin;
    int32_t fMax;
};


/**
 * A formatter for small, positive integers.
 */
class U_I18N_API SmallIntFormatter : public UMemory {
public:
    /**
     * Estimates the actual digit count needed to format positiveValue
     * using the given range of digit counts.
     * Returns a value that is at least the actual digit count needed.
     *
     * @param positiveValue the value to format
     * @param range the acceptable range of digit counts.
     */
    static int32_t estimateDigitCount(
            int32_t positiveValue, const IntDigitCountRange &range);

    /**
     * Returns TRUE if this class can format positiveValue using
     * the given range of digit counts.
     *
     * @param positiveValue the value to format
     * @param range the acceptable range of digit counts.
     */
    static UBool canFormat(
            int32_t positiveValue, const IntDigitCountRange &range);

    /**
     * Formats positiveValue using the given range of digit counts.
     * Always uses standard digits '0' through '9'. Formatted value is
     * left padded with '0' as necessary to achieve minimum digit count.
     * Does not produce any grouping separators or trailing decimal point.
     * Calling format to format a value with a particular digit count range
     * when canFormat indicates that the same value and digit count range
     * cannot be formatted results in undefined behavior.
     *
     * @param positiveValue the value to format
     * @param range the acceptable range of digit counts.
     */
    static UnicodeString &format(
            int32_t positiveValue,
            const IntDigitCountRange &range,
            UnicodeString &appendTo);

};

U_NAMESPACE_END

#endif  // __SMALLINTFORMATTER_H__
