%feature("docstring") OT::JointDistribution
"Joint distribution.

Parameters
----------
distributions : list of :class:`~openturns.Distribution`
    List of :math:`n` marginals of the distribution. Each marginal must be of
    dimension 1.
core : :class:`~openturns.Distribution`, optional
    A distribution over :math:`[0,1]^\\inputDim`. If not mentioned, the core is set to an
    :class:`~openturns.IndependentCopula` with the same dimension as
    *distributions*.

Notes
-----
A JointDistribution is a :math:`n`-dimensional distribution which can be
written in terms of 1-d distribution functions and a core :math:`K`
which describes the interaction between the variables. If :math:`K` is a
copula, then it represents the dependence structure of the joint
distribution and the given 1-d distributions are its marginal distributions.
Its cumulative distribution function :math:`F` is defined by the 1-d
distributions :math:`F_i` and the core :math:`C` through the relation:

.. math::

    F(x_1, \\cdots, x_\\inputDim) = C(F_1(x_1), \\cdots, F_\\inputDim(x_\\inputDim))

See also
--------
SklarCopula

Examples
--------
>>> import openturns as ot
>>> copula = ot.GumbelCopula(2.0)
>>> marginals = [ot.Uniform(1.0, 2.0), ot.Normal(2.0, 3.0)]
>>> distribution = ot.JointDistribution(marginals, copula)

Draw a sample:

>>> sample = distribution.getSample(5)

Joint distribution based on a non-copula core:

>>> core = ot.Dirichlet([2.0, 1.5, 2.5])
>>> distribution = ot.JointDistribution(marginals, core)

Draw a sample:

>>> sample = distribution.getSample(5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::JointDistribution::getDistributionCollection
"Get the marginals of the distribution.

Returns
-------
distributions : list of :class:`~openturns.Distribution`
    List of the marginals of the distribution."

// ---------------------------------------------------------------------

%feature("docstring") OT::JointDistribution::setDistributionCollection
"Set the marginals of the distribution.

Parameters
----------
distributions : list of :class:`~openturns.Distribution`
    List of the marginals of the distribution."

// ---------------------------------------------------------------------

%feature("docstring") OT::JointDistribution::setCopula
"Set the copula of the distribution.

Parameters
----------
copula : :class:`~openturns.Distribution`
    Copula of the distribution."

// ---------------------------------------------------------------------

%feature("docstring") OT::JointDistribution::getCore
"Get the core of the distribution.

Returns
-------
core : :class:`~openturns.Distribution`
    Core of the distribution. A distribution whose range is
    included in the unit hypercube :math:`[0, 1]^\\inputDim` where
    :math:`\\inputDim` is the distribution dimension."

// ---------------------------------------------------------------------

%feature("docstring") OT::JointDistribution::setCore
"Set the core of the distribution.

Parameters
----------
core : :class:`~openturns.Distribution`
    Core of the distribution. Any distribution whose range is
    included in the unit hypercube :math:`[0, 1]^\\inputDim` where
    :math:`\\inputDim` is the distribution dimension."
