//$ nocpp

/**
 * @file r8bsrc.h
 *
 * @brief Inclusion file for use with the "r8bsrc.dll" library.
 *
 * This is the inclusion file for the "r8bsrc.dll" dynamic link library
 * (the "r8bsrc.lib" library should be included into the project). This DLL
 * is designed to be used on Windows, on a processor with SSE2 support.
 * On non-Windows systems it is preferrable to use the C++ library directly.
 *
 * Before using this DLL library please read the description of the
 * r8b::CDSPResampler class and its functions.
 *
 * Note that the "int" and "enum" types have 32-bit size on both 32-bit and
 * 64-bit systems. Pointer types, including the CR8BResampler type, have
 * 32-bit size on 32-bit system and 64-bit size on 64-bit system.
 *
 * r8brain-free-src Copyright (c) 2013-2022 Aleksey Vaneev
 * See the "LICENSE" file for license.
 */

#ifndef R8BSRC_INCLUDED
#define R8BSRC_INCLUDED

/**
 * Resampler object handle.
 */

typedef void* CR8BResampler;

/**
 * Possible resampler object resolutions.
 */

enum ER8BResamplerRes
{
	r8brr16 = 0, ///< 16-bit precision resampler.
	r8brr16IR = 1, ///< 16-bit precision resampler for impulse responses.
	r8brr24 = 2 ///< 24-bit precision resampler (including 32-bit floating
		///< point).
};

extern "C" {

#ifndef R8BSRC_DECL
	#define R8BSRC_DECL __declspec( dllimport )
#endif // R8BSRC_DECL

/**
 * Function creates a new linear-phase resampler object.
 *
 * @param SrcSampleRate Source signal's sample rate. Both sample rates can
 * be specified as a ratio, e.g. SrcSampleRate = 1.0, DstSampleRate = 2.0.
 * @param DstSampleRate Destination signal's sample rate.
 * @param MaxInLen The maximal planned length of the input buffer (in samples)
 * that will be passed to the resampler. The resampler relies on this value as
 * it allocates intermediate buffers. Input buffers longer than this value
 * should never be supplied to the resampler.
 * @param ReqTransBand Required transition band, in percent of the spectral
 * space of the input signal (or the output signal if downsampling is
 * performed) between filter's -3 dB point and the Nyquist frequency.
 * @param Res Resampler's required resolution.
 * @return Pointer to the resampler object.
 */

R8BSRC_DECL CR8BResampler r8b_create( const double SrcSampleRate,
	const double DstSampleRate, const int MaxInLen, const double ReqTransBand,
	const ER8BResamplerRes Res );

/**
 * Function deletes a resampler object previously created via the r8b_create()
 * function.
 *
 * @param rs Resampler object to delete.
 */

R8BSRC_DECL void r8b_delete( CR8BResampler const rs );

/**
 * Function returns the number of input samples required to produce at least
 * the specified number of output samples, starting at the cleared or
 * after-construction state of resampler object.
 *
 * @param rs Resampler object to query.
 * @param ReqOutSamples The number of output samples required. If a
 * non-positive value was specified, the function returns 0.
 * @return The number of input samples required.
 */

R8BSRC_DECL int r8b_inlen( CR8BResampler const rs, const int ReqOutSamples );

/**
 * Function clears (resets) the state of the resampler object and returns it
 * to the state after construction. All input data accumulated in the
 * internal buffer of the resampler object so far will be discarded.
 *
 * @param rs Resampler object to clear.
 */

R8BSRC_DECL void r8b_clear( CR8BResampler const rs );

/**
 * Function performs sample rate conversion.
 *
 * If the source and destination sample rates are equal, the resampler will do
 * nothing and will simply return the input buffer unchanged.
 *
 * You do not need to allocate an intermediate output buffer for use with this
 * function. If required, the resampler will allocate a suitable intermediate
 * output buffer itself.
 *
 * @param rs Resampler object that performs processing.
 * @param ip0 Input buffer. This buffer may be used as output buffer by this
 * function.
 * @param l The number of samples available in the input buffer.
 * @param[out] op0 This variable receives the pointer to the resampled data.
 * This pointer may point to the address within the "ip0" input buffer, or to
 * *this object's internal buffer. In real-time applications it is suggested
 * to pass this pointer to the next output audio block and consume any data
 * left from the previous output audio block first before calling the
 * r8b_process() function again. The buffer pointed to by the "op0" on return
 * may be owned by the resampler, so it should not be freed by the caller.
 * @return The number of samples available in the "op0" output buffer. If the
 * data from the output buffer "op0" is going to be written to a bigger output
 * buffer, it is suggested to check the returned number of samples so that no
 * overflow of the bigger output buffer happens.
 */

R8BSRC_DECL int r8b_process( CR8BResampler const rs, double* const ip0,
	const int l, double*& op0 );

} // extern "C"

#endif // R8BSRC_INCLUDED
