﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/Distribution.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class UpdateDistributionConfigurationRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API UpdateDistributionConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDistributionConfiguration"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the distribution configuration that you
   * want to update.</p>
   */
  inline const Aws::String& GetDistributionConfigurationArn() const { return m_distributionConfigurationArn; }
  inline bool DistributionConfigurationArnHasBeenSet() const { return m_distributionConfigurationArnHasBeenSet; }
  template <typename DistributionConfigurationArnT = Aws::String>
  void SetDistributionConfigurationArn(DistributionConfigurationArnT&& value) {
    m_distributionConfigurationArnHasBeenSet = true;
    m_distributionConfigurationArn = std::forward<DistributionConfigurationArnT>(value);
  }
  template <typename DistributionConfigurationArnT = Aws::String>
  UpdateDistributionConfigurationRequest& WithDistributionConfigurationArn(DistributionConfigurationArnT&& value) {
    SetDistributionConfigurationArn(std::forward<DistributionConfigurationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the distribution configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateDistributionConfigurationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The distributions of the distribution configuration.</p>
   */
  inline const Aws::Vector<Distribution>& GetDistributions() const { return m_distributions; }
  inline bool DistributionsHasBeenSet() const { return m_distributionsHasBeenSet; }
  template <typename DistributionsT = Aws::Vector<Distribution>>
  void SetDistributions(DistributionsT&& value) {
    m_distributionsHasBeenSet = true;
    m_distributions = std::forward<DistributionsT>(value);
  }
  template <typename DistributionsT = Aws::Vector<Distribution>>
  UpdateDistributionConfigurationRequest& WithDistributions(DistributionsT&& value) {
    SetDistributions(std::forward<DistributionsT>(value));
    return *this;
  }
  template <typename DistributionsT = Distribution>
  UpdateDistributionConfigurationRequest& AddDistributions(DistributionsT&& value) {
    m_distributionsHasBeenSet = true;
    m_distributions.emplace_back(std::forward<DistributionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateDistributionConfigurationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_distributionConfigurationArn;
  bool m_distributionConfigurationArnHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<Distribution> m_distributions;
  bool m_distributionsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
