# data_combo_row.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later


import os
from glob import glob
from typing import Any

from gi.repository import GObject, Gtk

from refine.widgets.combo_row import RefineComboRow
from refine.compositions import RefineCompositions
from refine.settings import RefineSettings

import logging


@Gtk.Template(resource_path="/page/tesk/Refine/widgets/data-combo-row.ui")
class RefineDataComboRow(RefineComboRow):
    """
    An `Adw.ComboRow` implementing the properties needed to
    work with `Gio.Settings`.
    """

    __gtype_name__ = "RefineDataComboRow"

    revert_button = Gtk.Template.Child()
    revealer = Gtk.Template.Child()

    lookup_directory = GObject.Property(type=str, default="")

    aliases_dict: dict[str, str] = {}

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

    @RefineCompositions.key
    def key(self, key: str) -> None:
        try:
            if not self.schema.has_key(key):
                raise TypeError

            if not (string := self.settings.get_string(key)):
                return

            self.set_selected(self.original_list.index(string))

        except (IndexError, TypeError):
            self.is_valid_setting = False
            logging.warning(f"Key “{key}” does not exist. Skipping…")
            return

        self.set_is_default(key)
        self.connect("notify::selected", self.selected_callback)

    @GObject.Property(type=str, default="")
    def data_subdirectory(self) -> str:
        """
        Get or set the data subdirectory. If the subdirectory is set, the
        `Gtk.StringList` is automatically set up for @self. If the
        `@self:lookup-directory` property is set, the list is filtered
        before creating the model.
        """

        return self._data_subdirectory

    @data_subdirectory.setter  # type: ignore [no-redef]
    def data_subdirectory(self, data_subdirectory: str) -> None:
        system_data_dirs_list = []
        for directory in RefineSettings.get_system_data_dirs():
            found_directories = glob(
                f"{os.path.join(directory, data_subdirectory)}{os.sep}*"
            )
            system_data_dirs_list += found_directories

        data_dirs_list = system_data_dirs_list

        data_dirs_list += glob(
            f"{os.path.join(RefineSettings.get_user_data_dir(), data_subdirectory)}{os.sep}/*"
        )

        if bool(self.lookup_directory):
            temp_dirs_list = []
            for directory in data_dirs_list:
                if os.path.isdir(os.path.join(directory, self.lookup_directory)):
                    temp_dirs_list.append(directory)
            data_dirs_list = temp_dirs_list

        data_dirs_list = filter(
            lambda directory: os.path.isdir(directory), data_dirs_list
        )
        data_dirs_list = list(set(map(os.path.basename, data_dirs_list)))

        if self.aliases_dict:
            data_dirs_list = list(
                map(lambda item: self.aliases_dict.get(item, item), data_dirs_list)
            )

        self.original_list = sorted(data_dirs_list, key=str.lower)
        self.transformed_list = self.original_list

        data_dirs_model = Gtk.StringList.new(self.transformed_list)
        self.set_model(data_dirs_model)

        self._data_subdirectory = data_subdirectory

    @GObject.Property(type=str)
    def aliases(self) -> dict[str, str]:
        """
        Get a dictionary of aliases or set a newline-separated string
        that gets converted to a dictionary.
        """

        return self.aliases_dict

    @aliases.setter  # type: ignore [no-redef]
    def aliases(self, aliases: str) -> None:
        aliases_list = aliases.split()

        # Credit: https://stackoverflow.com/a/12739974
        self.aliases_dict = dict(zip(aliases_list[::2], aliases_list[1::2]))
