import { Blueprint } from './blueprint';
export interface Homeserver {
    BaseURL: string;
    FedBaseURL: string;
    ContainerID: string;
    AccessTokens: {
        [userId: string]: string;
    };
    DeviceIDs: {
        [userId: string]: string;
    };
    ApplicationServices: {
        [appserviceId: string]: string;
    };
}
/**
 * Options for a /create request if you have your own custom blueprint you want to run.
 *
 * @see https://github.com/matrix-org/complement/blob/3af1311ef9e2d75ab29f2773719eddb826258e6d/cmd/homerunner/route_create.go#L13
 */
export interface CreateOptionsDynamicBlueprint {
    base_image_uri: string;
    blueprint: Blueprint;
}
/**
 * This allows you to deploy any one of the static blueprints in https://github.com/matrix-org/complement/tree/master/internal/b
 */
export interface CreateOptionsStaticBlueprint {
    blueprint_name: string;
    base_image_uri: string;
}
declare type CreateOptions = CreateOptionsStaticBlueprint | CreateOptionsDynamicBlueprint;
/**
 * Options for a /destroy request
 *
 * @see https://github.com/matrix-org/complement/blob/3af1311ef9e2d75ab29f2773719eddb826258e6d/cmd/homerunner/route_destroy.go#L10
 */
export interface DestroyOptions {
    blueprint_name: string;
}
export interface CreateResponse {
    homeservers: {
        [homeserverId: string]: Homeserver;
    };
    expires: string;
}
export declare class HomerunnerError extends Error {
    constructor(statusCode: number, body: string);
}
declare type Fetch = (input: RequestInfo | string, init?: RequestInit) => Promise<Response>;
/**
 * A client interface for Homerunner.
 * @see https://github.com/matrix-org/complement/tree/main/cmd/homerunner
 */
export declare class Client {
    readonly baseUrl: string;
    private readonly fetch;
    /**
     * @param baseUrl The URL for homerunner's API endpoint. This will default to:
     * - The `HOMERUNNER_URL` env var, if defined.
     * - `http://localhost:${HOMERUNNER_PORT}`, if the HOMERUNNER_PORT env is defined.
     * - `http://localhost:54321`
     * @param fetch Fetch function compatable with the [Fetch API](https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API). This will
     * use either the native fetch provided by Node 18+, or a polyfill.
     */
    constructor(baseUrl?: string, fetch?: Fetch);
    /**
     * Deploy a blueprint.
     * @param nameOrOptions Either a blueprint name that has been previously defined, or a in-line blueprint.
     */
    create(nameOrOptions: string | CreateOptions): Promise<CreateResponse>;
    /**
     * Destroy a blueprint.
     * @param blueprintName The name of the blueprint to destroy.
     */
    destroy(blueprintName: string): Promise<void>;
    /**
     * Check to see if the homerunner service is up and listening for requests.
     */
    health(): Promise<void>;
}
export {};
