/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk;

import java.util.Vector;

import org.gnu.gdk.Color;
import org.gnu.glib.EventMap;
import org.gnu.gtk.event.ColorButtonEvent;
import org.gnu.gtk.event.ColorButtonListener;
import org.gnu.gtk.event.GtkEventType;

/**
 */
public class ColorButton extends Button {

	public ColorButton() {
		if (-1 == handle) {
			handle = gtk_color_button_new();
			initializeEventHandlers();
		}
	}
	
	public ColorButton(Color color) {
		handle = gtk_color_button_new_with_color(color.getHandle());
		initializeEventHandlers();
	}
	
	public void setColor(Color color) {
		gtk_color_button_set_color(handle, color.getHandle());
	}
	
	public Color getColor() {
		return new Color(gtk_color_button_get_color(handle));
	}
	
	public void setAlpha(int alpha) {
		gtk_color_button_set_alpha(handle, alpha);
	}
	
	public int getAlpha() {
		return gtk_color_button_get_alpha(handle);
	}
	
	public void setUseAlpha(boolean useAlpha) {
		gtk_color_button_set_use_alpha(handle, useAlpha);
	}
	
	public boolean getUseAlpha() {
		return gtk_color_button_get_use_alpha(handle);
	}
	
	public void setTitle(String title) {
		gtk_color_button_set_title(handle, title);
	}
	
	public String getTitle() {
		return gtk_color_button_get_title(handle);
	}
	
	/* **************************************
	 * EVENT LISTENERS
	 ****************************************/

	/**
	 * Listeners for handling dialog events
	 */
	private Vector cbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see ColorButtonListener
	 */
	public void addListener(ColorButtonListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(cbListeners, listener);
		if (i == -1) {
			if (null == cbListeners) {
				cbListeners = new Vector();
			}
			cbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(ColorButtonListener)
	 */
	public void removeListener(ColorButtonListener listener) {
		int i = findListener(cbListeners, listener);
		if (i > -1) {
			cbListeners.remove(i);
		}
		if (0 == cbListeners.size()) {
			cbListeners = null;
		}
	}

	protected void fireComboBoxEvent(ColorButtonEvent event) {
		if (null == cbListeners) {
			return;
		}
		int size = cbListeners.size();
		int i = 0;
		while (i < size) {
			ColorButtonListener cbl = (ColorButtonListener)cbListeners.elementAt(i);
			cbl.colorButtonEvent(event);
			i++;
		}
	}

	private void handleSetColor() {
		fireComboBoxEvent(new ColorButtonEvent(this, ColorButtonEvent.Type.COLOR_SET));
	}

	protected void initializeEventHandlers() {
		super.initializeEventHandlers();
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		return evtMap.getListenerClass(signal);
	}

	public GtkEventType getEventType(String signal) {
		return evtMap.getEventType(signal);
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("set_color", "handleSetColor", ColorButtonEvent.Type.COLOR_SET, ColorButtonListener.class);
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_color_button_get_type ();
	native static final protected int gtk_color_button_new();
	native static final protected int gtk_color_button_new_with_color(int color);
	native static final protected void gtk_color_button_set_color(int button, int color);
	native static final protected void gtk_color_button_set_alpha(int button, int alpha);
	native static final protected int gtk_color_button_get_color(int button);
	native static final protected int gtk_color_button_get_alpha(int button);
	native static final protected void gtk_color_button_set_use_alpha(int button, boolean useAlpha);
	native static final protected boolean gtk_color_button_get_use_alpha(int button);
	native static final protected void gtk_color_button_set_title(int button, String title);
	native static final protected String gtk_color_button_get_title(int button);                                                                                
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
