/* gnome-db-entry-text.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-entry-text.h"
#include <libgnomedb/gnome-db-data-handler.h>

/* 
 * Main static functions 
 */
static void gnome_db_entry_text_class_init (GnomeDbEntryTextClass * class);
static void gnome_db_entry_text_init (GnomeDbEntryText * srv);
static void gnome_db_entry_text_dispose (GObject   * object);
static void gnome_db_entry_text_finalize (GObject   * object);

/* virtual functions */
static GtkWidget *create_entry (GnomeDbEntryWrapper *mgwrap);
static void       real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value);
static GdaValue  *real_get_value (GnomeDbEntryWrapper *mgwrap);
static void       connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback);
static gboolean   expand_in_layout (GnomeDbEntryWrapper *mgwrap);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _GnomeDbEntryTextPrivate
{
	GtkTextBuffer *buffer;
	GtkWidget     *view;
};


GType
gnome_db_entry_text_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntryTextClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_entry_text_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbEntryText),
			0,
			(GInstanceInitFunc) gnome_db_entry_text_init
		};
		
		type = g_type_register_static (GNOME_DB_ENTRY_WRAPPER_TYPE, "GnomeDbEntryText", &info, 0);
	}
	return type;
}

static void
gnome_db_entry_text_class_init (GnomeDbEntryTextClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_entry_text_dispose;
	object_class->finalize = gnome_db_entry_text_finalize;

	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->create_entry = create_entry;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_set_value = real_set_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_get_value = real_get_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->connect_signals = connect_signals;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->expand_in_layout = expand_in_layout;
}

static void
gnome_db_entry_text_init (GnomeDbEntryText * gnome_db_entry_text)
{
	gnome_db_entry_text->priv = g_new0 (GnomeDbEntryTextPrivate, 1);
	gnome_db_entry_text->priv->buffer = NULL;
	gnome_db_entry_text->priv->view = NULL;
}

/**
 * gnome_db_entry_text_new
 * @dh: the data handler to be used by the new widget
 * @type: the requested data type (compatible with @dh)
 *
 * Creates a new widget which is mainly a GtkEntry
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_entry_text_new (GnomeDbDataHandler *dh, GdaValueType type)
{
	GObject *obj;
	GnomeDbEntryText *mgtxt;

	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, NULL);
	g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (dh, type), NULL);

	obj = g_object_new (GNOME_DB_ENTRY_TEXT_TYPE, "handler", dh, NULL);
	mgtxt = GNOME_DB_ENTRY_TEXT (obj);
	gnome_db_data_entry_set_value_type (GNOME_DB_DATA_ENTRY (mgtxt), type);

	return GTK_WIDGET (obj);
}


static void
gnome_db_entry_text_dispose (GObject   * object)
{
	GnomeDbEntryText *gnome_db_entry_text;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_TEXT (object));

	gnome_db_entry_text = GNOME_DB_ENTRY_TEXT (object);
	if (gnome_db_entry_text->priv) {

	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_entry_text_finalize (GObject   * object)
{
	GnomeDbEntryText *gnome_db_entry_text;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_TEXT (object));

	gnome_db_entry_text = GNOME_DB_ENTRY_TEXT (object);
	if (gnome_db_entry_text->priv) {

		g_free (gnome_db_entry_text->priv);
		gnome_db_entry_text->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static GtkWidget *
create_entry (GnomeDbEntryWrapper *mgwrap)
{
	GnomeDbEntryText *mgtxt;
	GtkWidget *sw;

	g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TEXT (mgwrap), NULL);
	mgtxt = GNOME_DB_ENTRY_TEXT (mgwrap);
	g_return_val_if_fail (mgtxt->priv, NULL);

	mgtxt->priv->view = gtk_text_view_new ();
	mgtxt->priv->buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (mgtxt->priv->view));
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (sw), mgtxt->priv->view);
	gtk_widget_show (mgtxt->priv->view);
	
	return sw;
}

static void
real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value)
{
	GnomeDbEntryText *mgtxt;

	g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TEXT (mgwrap));
	mgtxt = GNOME_DB_ENTRY_TEXT (mgwrap);
	g_return_if_fail (mgtxt->priv);

	if (value) {
		if (gda_value_is_null (value))
                        gtk_text_buffer_set_text (mgtxt->priv->buffer, "", -1);
		else {
			GnomeDbDataHandler *dh;		
			gchar *str;

			dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
			str = gnome_db_data_handler_get_str_from_value (dh, value);
			if (str) {
				gtk_text_buffer_set_text (mgtxt->priv->buffer, str, -1);
				g_free (str);
			}
		}
	}
	else 
		gtk_text_buffer_set_text (mgtxt->priv->buffer, "", -1);
}

static GdaValue *
real_get_value (GnomeDbEntryWrapper *mgwrap)
{
	GdaValue *value;
	GnomeDbEntryText *mgtxt;
	GnomeDbDataHandler *dh;
	gchar *str;
	GtkTextIter start, end;

	g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TEXT (mgwrap), NULL);
	mgtxt = GNOME_DB_ENTRY_TEXT (mgwrap);
	g_return_val_if_fail (mgtxt->priv, NULL);

	dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
	gtk_text_buffer_get_start_iter (mgtxt->priv->buffer, &start);
	gtk_text_buffer_get_end_iter (mgtxt->priv->buffer, &end);
	str = gtk_text_buffer_get_text (mgtxt->priv->buffer, &start, &end, FALSE);
	value = gnome_db_data_handler_get_value_from_sql (dh, str, 
						    gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgwrap)));
	g_free (str);
	if (!value) {
		/* in case the gnome_db_data_handler_get_value_from_sql() returned an error because
		   the contents of the GtkEntry cannot be interpreted as a GdaValue */
		value = gda_value_new_null ();
	}

	return value;
}

static void
connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback)
{
	GnomeDbEntryText *mgtxt;

	g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TEXT (mgwrap));
	mgtxt = GNOME_DB_ENTRY_TEXT (mgwrap);
	g_return_if_fail (mgtxt->priv);

	g_signal_connect (G_OBJECT (mgtxt->priv->buffer), "changed",
			  callback, mgwrap);
}

static gboolean
expand_in_layout (GnomeDbEntryWrapper *mgwrap)
{
	return TRUE;
}
