/* gnome-db-graph-query.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-graph-query.h"
#include "gnome-db-graph-item.h"
#include <libgnomedb/gnome-db-query.h>
#include <libgnomedb/gnome-db-target.h>

/* 
 * Main static functions 
 */
static void gnome_db_graph_query_class_init (GnomeDbGraphQueryClass *class);
static void gnome_db_graph_query_init (GnomeDbGraphQuery *graph);
static void gnome_db_graph_query_dispose (GObject *object);
static void gnome_db_graph_query_finalize (GObject *object);

static void gnome_db_graph_query_initialize (GnomeDbGraphQuery *graph);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;


struct _GnomeDbGraphQueryPrivate
{
	GnomeDbQuery   *query;
};


GType
gnome_db_graph_query_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGraphQueryClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_graph_query_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGraphQuery),
			0,
			(GInstanceInitFunc) gnome_db_graph_query_init
		};

		type = g_type_register_static (GNOME_DB_GRAPH_TYPE, "GnomeDbGraphQuery", &info, 0);
	}
	return type;
}



static void
gnome_db_graph_query_class_init (GnomeDbGraphQueryClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);
	
	object_class->dispose = gnome_db_graph_query_dispose;
	object_class->finalize = gnome_db_graph_query_finalize;
}

static void
gnome_db_graph_query_init (GnomeDbGraphQuery *graph)
{
	graph->priv = g_new0 (GnomeDbGraphQueryPrivate, 1);
	graph->priv->query = NULL;
}

/**
 * gnome_db_graph_query_new
 * @query: a #GnomeDbQuery object
 *
 * Creates a new #GnomeDbGraphQuery object. This graph object is a specialized #GnomeDbGraph object
 * in the way that it always make sure it "contains" #GnomeDbGraphItem objects for each target
 * in @query.
 *
 * However, when created, the new #GnomeDbGraphItem object will not contain any graph item; 
 * but can be brought in sync with the gnome_db_graph_query_sync_targets() method.
 *
 * Returns: the newly created object
 */
GObject *
gnome_db_graph_query_new (GnomeDbQuery *query)
{
	GObject *obj = NULL;
	GnomeDbGraphQuery *graph;
	guint id;
	GnomeDbDict *dict;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_GRAPH_QUERY_TYPE, "dict", dict, NULL);
	graph = GNOME_DB_GRAPH_QUERY (obj);

	g_object_get (G_OBJECT (dict), "graph_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	gnome_db_dict_declare_graph (dict, GNOME_DB_GRAPH (graph));
	g_object_set (obj, "graph_type", GNOME_DB_GRAPH_QUERY_JOINS, "ref_object", query, NULL);

	/* REM: we don't need to catch @query's nullification because #GnomeDbGraph already does it. */
	graph->priv->query = query;

	gnome_db_graph_query_initialize (graph);

	return obj;
}

static void target_added_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbGraphQuery *graph);
static void target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbGraphQuery *graph);

static void
gnome_db_graph_query_dispose (GObject *object)
{
	GnomeDbGraphQuery *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH_QUERY (object));

	graph = GNOME_DB_GRAPH_QUERY (object);
	if (graph->priv) {
		if (graph->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (graph->priv->query),
							      G_CALLBACK (target_added_cb), graph);
			g_signal_handlers_disconnect_by_func (G_OBJECT (graph->priv->query),
							      G_CALLBACK (target_removed_cb), graph);
			graph->priv->query = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}


static void
gnome_db_graph_query_finalize (GObject   * object)
{
	GnomeDbGraphQuery *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH_QUERY (object));

	graph = GNOME_DB_GRAPH_QUERY (object);
	if (graph->priv) {
		g_free (graph->priv);
		graph->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void
gnome_db_graph_query_initialize (GnomeDbGraphQuery *graph)
{
	/* catch the creation and deletion of any new target */
	g_signal_connect (G_OBJECT (graph->priv->query), "target_added",
			  G_CALLBACK (target_added_cb), graph);
	g_signal_connect (G_OBJECT (graph->priv->query), "target_removed",
			  G_CALLBACK (target_removed_cb), graph);
}

static void
target_added_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbGraphQuery *graph)
{
	GnomeDbGraphItem *gitem;

	/* There is a possibility that the graph item has already been created somewhere else,
	   in which case there is nothing to be done */
	gitem = gnome_db_graph_get_item_from_obj (GNOME_DB_GRAPH (graph), GNOME_DB_BASE (target), FALSE);
	if (gitem)
		return;

	/* FIXME: find a better  positionning layout for the target items */
	gitem = GNOME_DB_GRAPH_ITEM (gnome_db_graph_item_new (gnome_db_base_get_dict (GNOME_DB_BASE (graph->priv->query)), GNOME_DB_BASE (target)));
	gnome_db_graph_item_set_position (gitem, 50., 50.);
	gnome_db_graph_add_item (GNOME_DB_GRAPH (graph), gitem);
	g_object_unref (G_OBJECT (gitem));
}

static void
target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbGraphQuery *graph)
{
	GnomeDbGraphItem *gitem;

	gitem = gnome_db_graph_get_item_from_obj (GNOME_DB_GRAPH (graph), GNOME_DB_BASE (target), FALSE);
	if (gitem)
		gnome_db_graph_del_item (GNOME_DB_GRAPH (graph), gitem);
}

/**
 * gnome_db_graph_query_sync_targets
 * @graph: a #GnomeDbGraphQuery object
 *
 * Synchronises the graph items with the targets of the query @graph represents
 */
void
gnome_db_graph_query_sync_targets (GnomeDbGraphQuery *graph)
{
	GSList *targets, *list;

	g_return_if_fail (graph && IS_GNOME_DB_GRAPH_QUERY (graph));
	g_return_if_fail (graph->priv);

	/* for each target in graph->priv->query, we create a #GnomeDbGraphItem, if necessary */
	targets = gnome_db_query_get_targets (graph->priv->query);
	list = targets;
	while (list) {
		target_added_cb (graph->priv->query, GNOME_DB_TARGET (list->data), graph);
		list = g_slist_next (list);
	}
	g_slist_free (targets);
}
