/* gnome-db-join.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-join.h"
#include "gnome-db-database.h"
#include "gnome-db-query.h"
#include "gnome-db-table.h"
#include "gnome-db-constraint.h"
#include "gnome-db-entity.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-field.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-target.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "marshal.h"
#include "gnome-db-condition.h"
#include <string.h>

#include "gnome-db-query-parsing.h"

/* 
 * Main static functions 
 */
static void gnome_db_join_class_init (GnomeDbJoinClass * class);
static void gnome_db_join_init (GnomeDbJoin * srv);
static void gnome_db_join_dispose (GObject   * object);
static void gnome_db_join_finalize (GObject   * object);

static void gnome_db_join_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_join_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_join_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_join_get_xml_id       (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_join_save_to_xml      (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_join_load_from_xml    (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Referer interface */
static void        gnome_db_join_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_join_activate            (GnomeDbReferer *iface);
static void        gnome_db_join_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_join_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_join_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_join_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* When the Query or any of the refering GnomeDbTarget is nullified */
static void        nullified_object_cb (GObject *obj, GnomeDbJoin *join);
static void        target_ref_lost_cb (GnomeDbRefBase *ref, GnomeDbJoin *join);
static void        target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbJoin *join);


#ifdef debug
static void        gnome_db_join_dump (GnomeDbJoin *join, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TYPE_CHANGED,
	CONDITION_CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_join_signals[LAST_SIGNAL] = { 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbJoinPrivate
{
	GnomeDbJoinType   join_type;
	GnomeDbQuery     *query;
	GnomeDbRefBase   *target1;
	GnomeDbRefBase   *target2;
	GnomeDbCondition *cond;
};



/* module error */
GQuark gnome_db_join_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_join_error");
	return quark;
}


GType
gnome_db_join_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbJoinClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_join_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbJoin),
			0,
			(GInstanceInitFunc) gnome_db_join_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_join_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_join_referer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbJoin", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void 
gnome_db_join_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_join_get_xml_id;
	iface->save_to_xml = gnome_db_join_save_to_xml;
	iface->load_from_xml = gnome_db_join_load_from_xml;
}

static void
gnome_db_join_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = gnome_db_join_activate;
	iface->deactivate = gnome_db_join_deactivate;
	iface->is_active = gnome_db_join_is_active;
	iface->get_ref_objects = gnome_db_join_get_ref_objects;
	iface->replace_refs = gnome_db_join_replace_refs;
}


static void
gnome_db_join_class_init (GnomeDbJoinClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_join_signals[TYPE_CHANGED] =
		g_signal_new ("type_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbJoinClass, type_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_join_signals[CONDITION_CHANGED] =
		g_signal_new ("condition_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbJoinClass, condition_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->type_changed = NULL;
	class->condition_changed = NULL;

	object_class->dispose = gnome_db_join_dispose;
	object_class->finalize = gnome_db_join_finalize;

	/* Properties */
	object_class->set_property = gnome_db_join_set_property;
	object_class->get_property = gnome_db_join_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_join_dump;
#endif

}

static void
gnome_db_join_init (GnomeDbJoin * gnome_db_join)
{
	gnome_db_join->priv = g_new0 (GnomeDbJoinPrivate, 1);
	gnome_db_join->priv->join_type = GNOME_DB_JOIN_TYPE_INNER;
	gnome_db_join->priv->query = NULL;
	gnome_db_join->priv->target1 = NULL;
	gnome_db_join->priv->target2 = NULL;
	gnome_db_join->priv->cond = NULL;
}

/**
 * gnome_db_join_new_with_targets
 * @query: a #GnomeDbQuery object in which the join will occur
 * @target_1: the 1st #GnomeDbTarget object participating in the join
 * @target_2: the 2nd #GnomeDbTarget object participating in the join
 *
 * Creates a new GnomeDbJoin object. Note: the #GnomeDbTarget ranks (1st and 2nd) does not matter, but
 * is necessary since the join may not be symetrical (LEFT or RIGHT join). Also, the #GnomeDbJoin object
 * may decide to swap the two if necessary.
 *
 * Returns: the new object
 */
GObject*
gnome_db_join_new_with_targets (GnomeDbQuery *query, GnomeDbTarget *target_1, GnomeDbTarget *target_2)
{
	GObject   *obj;
	GnomeDbJoin *gnome_db_join;
	GnomeDbDict *dict;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target_1 && IS_GNOME_DB_TARGET (target_1), NULL);
	g_return_val_if_fail (target_2 && IS_GNOME_DB_TARGET (target_2), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target_1) == query, NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target_2) == query, NULL);
	g_return_val_if_fail (target_1 != target_2, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_JOIN_TYPE, "dict", dict, NULL);
	gnome_db_join = GNOME_DB_JOIN (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_join), 0);

	gnome_db_join->priv->query = query;
	gnome_db_join->priv->target1 = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_join->priv->target1, GNOME_DB_BASE (target_1));

	gnome_db_join->priv->target2 = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_join->priv->target2, GNOME_DB_BASE (target_2));
	
	gnome_db_base_connect_nullify (query, G_CALLBACK (nullified_object_cb), gnome_db_join);

	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_join);

	/* if the references to the any target is lost then we want to nullify the join */
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target1), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target2), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);

	return obj;
}

/**
 * gnome_db_join_new_with_xml_ids
 * @query: a #GnomeDbQuery object in which the join will occur
 * @target_1_xml_id: the 1st #GnomeDbTarget object's XML id participating in the join
 * @target_2_xml_id: the 2nd #GnomeDbTarget object's XML id participating in the join
 *
 * Creates a new GnomeDbJoin object. Note: the #GnomeDbTarget ranks (1st and 2nd) does not matter, but
 * is necessary since the join may not be symetrical (LEFT or RIGHT join). Also, the #GnomeDbJoin object
 * may decide to swap the two if necessary.
 *
 * Returns: the new object
 */
GObject *
gnome_db_join_new_with_xml_ids (GnomeDbQuery *query, const gchar *target_1_xml_id, const gchar *target_2_xml_id)
{
	GObject   *obj;
	GnomeDbJoin *gnome_db_join;
	GnomeDbDict *dict;
	gchar *qid, *ptr, *tok;
	gchar *tid;
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target_1_xml_id && *target_1_xml_id, NULL);
	g_return_val_if_fail (target_2_xml_id && *target_2_xml_id, NULL);
	g_return_val_if_fail (strcmp (target_1_xml_id, target_2_xml_id), NULL);

	/* check that the XML Ids start with the query's XML Id */
	qid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (query));
	tid = g_strdup (target_1_xml_id);
	ptr = strtok_r (tid, ":", &tok);
	g_return_val_if_fail (!strcmp (ptr, qid), NULL);
	g_free (tid);
	tid = g_strdup (target_2_xml_id);
	ptr = strtok_r (tid, ":", &tok);
	g_return_val_if_fail (!strcmp (ptr, qid), NULL);
	g_free (tid);
	g_free (qid);
	

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_JOIN_TYPE, "dict", dict, NULL);
	gnome_db_join = GNOME_DB_JOIN (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_join), 0);

	gnome_db_join->priv->query = query;
	gnome_db_join->priv->target1 = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_join->priv->target1, GNOME_DB_TARGET_TYPE, REFERENCE_BY_XML_ID, target_1_xml_id);

	gnome_db_join->priv->target2 = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_join->priv->target2, GNOME_DB_TARGET_TYPE, REFERENCE_BY_XML_ID, target_2_xml_id);
	
	gnome_db_base_connect_nullify (query, G_CALLBACK (nullified_object_cb), gnome_db_join);

	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_join);

	/* if the references to the any target is lost then we want to nullify the join */
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target1), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target2), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);

	return obj;
}

/**
 * gnome_db_join_new_copy
 * @orig: a #GnomeDbJoin to make a copy of
 * @replacements: a hash table to store replacements, or %NULL
 * 
 * Copy constructor
 *
 * Returns: a the new copy of @orig
 */
GObject *
gnome_db_join_new_copy (GnomeDbJoin *orig, GHashTable *replacements)
{
	GObject   *obj;
	GnomeDbJoin *gnome_db_join;
	GnomeDbDict *dict;

	g_return_val_if_fail (orig && IS_GNOME_DB_JOIN (orig), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	obj = g_object_new (GNOME_DB_JOIN_TYPE, "dict", dict, NULL);
	gnome_db_join = GNOME_DB_JOIN (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_join), 0);
	if (replacements)
		g_hash_table_insert (replacements, orig, gnome_db_join);

	gnome_db_join->priv->query = orig->priv->query;
	gnome_db_base_connect_nullify (orig->priv->query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_join);

	g_signal_connect (G_OBJECT (orig->priv->query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_join);

	gnome_db_join->priv->target1 = GNOME_DB_REF_BASE (gnome_db_ref_base_new_copy (orig->priv->target1));
	gnome_db_join->priv->target2 = GNOME_DB_REF_BASE (gnome_db_ref_base_new_copy (orig->priv->target2));
	gnome_db_join->priv->join_type = orig->priv->join_type;

	/* if the references to the any target is lost then we want to nullify the join */
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target1), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);
	g_signal_connect (G_OBJECT (gnome_db_join->priv->target2), "ref_lost",
			  G_CALLBACK (target_ref_lost_cb), gnome_db_join);
	
	/* join condition */
	if (orig->priv->cond) {
		GnomeDbCondition *cond = GNOME_DB_CONDITION (gnome_db_condition_new_copy (orig->priv->cond, replacements));
		gnome_db_join_set_condition (gnome_db_join, cond);
		g_object_unref (G_OBJECT (cond));
		if (replacements)
			g_hash_table_insert (replacements, orig->priv->cond, cond);
	}

	return obj;	
}

static void
nullified_object_cb (GObject *obj, GnomeDbJoin *join)
{
	gnome_db_base_nullify (GNOME_DB_BASE (join));
}

static void
target_ref_lost_cb (GnomeDbRefBase *ref, GnomeDbJoin *join)
{
	gnome_db_base_nullify (GNOME_DB_BASE (join));
}

static void
target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbJoin *join)
{
	GnomeDbBase *t;
	gboolean to_nullify = FALSE;

	t = gnome_db_ref_base_get_ref_object (join->priv->target1);
	if (t == (GnomeDbBase *) target)
		to_nullify = TRUE;
	if (! to_nullify) {
		t = gnome_db_ref_base_get_ref_object (join->priv->target2);
		if (t == (GnomeDbBase *) target)
			to_nullify = TRUE;
	}

	if (to_nullify)
		gnome_db_base_nullify (GNOME_DB_BASE (join));
}


static void
cond_changed_cb (GnomeDbCondition *cond, GnomeDbJoin *join)
{
#ifdef debug_signal
	g_print (">> 'CONDITION_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (join), "condition_changed");
#ifdef debug_signal
	g_print ("<< 'CONDITION_CHANGED' from %s\n", __FUNCTION__);
#endif
	gnome_db_base_changed (GNOME_DB_BASE (join));
}

static void
nullified_cond_cb (GnomeDbCondition *cond, GnomeDbJoin *join) 
{
	g_assert (cond == join->priv->cond);
	g_signal_handlers_disconnect_by_func (G_OBJECT (join->priv->cond),
					      G_CALLBACK (nullified_cond_cb), join);
	g_signal_handlers_disconnect_by_func (G_OBJECT (join->priv->cond),
					      G_CALLBACK (cond_changed_cb), join);
	g_object_set (G_OBJECT (cond), "join", NULL, NULL);
	g_object_unref (join->priv->cond);
	join->priv->cond = NULL;
}

static void
gnome_db_join_dispose (GObject *object)
{
	GnomeDbJoin *gnome_db_join;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_JOIN (object));

	gnome_db_join = GNOME_DB_JOIN (object);
	if (gnome_db_join->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		
		if (gnome_db_join->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_join->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_join);
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_join->priv->query),
							      G_CALLBACK (target_removed_cb), gnome_db_join);
			gnome_db_join->priv->query = NULL;
		}
		if (gnome_db_join->priv->target1) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_join->priv->target1),
							      G_CALLBACK (target_ref_lost_cb), gnome_db_join);
			g_object_unref (G_OBJECT (gnome_db_join->priv->target1));
			gnome_db_join->priv->target1 = NULL;
		}
		if (gnome_db_join->priv->target2) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_join->priv->target2),
							      G_CALLBACK (target_ref_lost_cb), gnome_db_join);
			g_object_unref (G_OBJECT (gnome_db_join->priv->target2));
			gnome_db_join->priv->target2 = NULL;
		}
		if (gnome_db_join->priv->cond)
			nullified_cond_cb (gnome_db_join->priv->cond, gnome_db_join);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_join_finalize (GObject   * object)
{
	GnomeDbJoin *gnome_db_join;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_JOIN (object));

	gnome_db_join = GNOME_DB_JOIN (object);
	if (gnome_db_join->priv) {
		g_free (gnome_db_join->priv);
		gnome_db_join->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_join_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbJoin *gnome_db_join;

	gnome_db_join = GNOME_DB_JOIN (object);
	if (gnome_db_join->priv) {
		switch (param_id) {
		case PROP:
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_join_get_property (GObject              *object,
			  guint                 param_id,
			  GValue               *value,
			  GParamSpec           *pspec)
{
	GnomeDbJoin *gnome_db_join;
	gnome_db_join = GNOME_DB_JOIN (object);
	
	if (gnome_db_join->priv) {
		switch (param_id) {
		case PROP:
			break;
		}	
	}
}

/**
 * gnome_db_join_set_join_type
 * @join: a #GnomeDbJoin object
 * @type: the new type of join
 *
 * Sets the type of @join
 */
void
gnome_db_join_set_join_type (GnomeDbJoin *join, GnomeDbJoinType type)
{
	g_return_if_fail (join && IS_GNOME_DB_JOIN (join));
	g_return_if_fail (join->priv);
	
	if (join->priv->join_type != type) {
		join->priv->join_type = type;
#ifdef debug_signal
		g_print (">> 'TYPE_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit_by_name (G_OBJECT (join), "type_changed");
#ifdef debug_signal
		g_print ("<< 'TYPE_CHANGED' from %s\n", __FUNCTION__);
#endif
		gnome_db_base_changed (GNOME_DB_BASE (join));
	}
}

/**
 * gnome_db_join_get_join_type
 * @join: a #GnomeDbJoin object
 *
 * Get the type of a join
 *
 * Returns: the type of @join
 */
GnomeDbJoinType
gnome_db_join_get_join_type (GnomeDbJoin *join)
{
	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), GNOME_DB_JOIN_TYPE_CROSS);
	g_return_val_if_fail (join->priv, GNOME_DB_JOIN_TYPE_CROSS);
	
	return join->priv->join_type;
}


/**
 * gnome_db_join_get_query
 * @join: a #GnomeDbJoin object
 * 
 * Get the #GnomeDbQuery to which @join is attached to
 *
 * Returns: the #GnomeDbQuery
 */
GnomeDbQuery *
gnome_db_join_get_query (GnomeDbJoin *join)
{
	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), NULL);
	g_return_val_if_fail (join->priv, NULL);
	
	return join->priv->query;
}

/**
 * gnome_db_join_get_target_1
 * @join: a #GnomeDbJoin object
 *
 * Get the 1st #GnomeDbTarget participating in the join
 *
 * Returns: the #GnomeDbTarget
 */
GnomeDbTarget *
gnome_db_join_get_target_1 (GnomeDbJoin *join)
{
	GnomeDbBase *base;

	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), NULL);
	g_return_val_if_fail (join->priv, NULL);
	
	base = gnome_db_ref_base_get_ref_object (join->priv->target1);
	if (base)
		return GNOME_DB_TARGET (base);
	else
		return NULL;
}

/**
 * gnome_db_join_get_target_2
 * @join: a #GnomeDbJoin object
 *
 * Get the 2nd #GnomeDbTarget participating in the join
 *
 * Returns: the #GnomeDbTarget
 */
GnomeDbTarget *
gnome_db_join_get_target_2 (GnomeDbJoin *join)
{
	GnomeDbBase *base;

	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), NULL);
	g_return_val_if_fail (join->priv, NULL);
	
	base = gnome_db_ref_base_get_ref_object (join->priv->target2);
	if (base)
		return GNOME_DB_TARGET (base);
	else
		return NULL;
}


/**
 * gnome_db_join_swap_targets
 * @join: a #GnomeDbJoin object
 *
 * Changes the relative roles of the two #GnomeDbTarget objects. It does not
 * change the join condition itself, and is usefull only for the internals
 * of the #GnomeDbQuery object
 */
void
gnome_db_join_swap_targets (GnomeDbJoin *join)
{
	GnomeDbRefBase *ref;
	g_return_if_fail (join && IS_GNOME_DB_JOIN (join));
	g_return_if_fail (join->priv);

	ref = join->priv->target1;
	join->priv->target1 = join->priv->target2;
	join->priv->target2 = ref;

	switch (join->priv->join_type) {
	case GNOME_DB_JOIN_TYPE_LEFT_OUTER:
		join->priv->join_type = GNOME_DB_JOIN_TYPE_RIGHT_OUTER;
		break;
	case GNOME_DB_JOIN_TYPE_RIGHT_OUTER:
		join->priv->join_type = GNOME_DB_JOIN_TYPE_LEFT_OUTER;
		break;
	default:
		break;
	}
}

/**
 * gnome_db_join_set_condition_from_fkcons
 * @join: a #GnomeDbJoin object
 *
 * Creates a #GnomeDbCondition for @join using the foreign key constraints
 * present in the database if the two targets @join joins are database tables
 * (#GnomeDbTable objects).
 *
 * If there is more than one FK constraint between the database tables, then
 * no join is created, and the call returns FALSE.
 *
 * Returns: TRUE if suitable foreign keys were found and a join condition
 * has been created
 */
gboolean
gnome_db_join_set_condition_from_fkcons (GnomeDbJoin *join)
{
	GnomeDbCondition *jcond = NULL;
	GSList *fklist, *list;
	GnomeDbDict *dict;
	GnomeDbTable *tab1, *tab2;
	GnomeDbTarget *target1, *target2;
	GnomeDbConstraint *cons = NULL;
	gboolean tab1_is_pkey = FALSE;

	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), FALSE);
	g_return_val_if_fail (join->priv, FALSE);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (join));

	/* checks */
	target1 = gnome_db_join_get_target_1 (join);
	tab1 = (GnomeDbTable *) gnome_db_target_get_represented_entity (target1);
	if (!tab1 || !IS_GNOME_DB_TABLE (tab1))
		return FALSE;

	target2 = gnome_db_join_get_target_2 (join);
	tab2 = (GnomeDbTable *) gnome_db_target_get_represented_entity (target2);
	if (!tab2 || !IS_GNOME_DB_TABLE (tab2))
		return FALSE;

	list = gnome_db_database_get_tables_fk_constraints (gnome_db_dict_get_database (dict), tab1, tab2, FALSE);
	if (g_slist_length (list) != 1) {
		g_slist_free (list);
		return FALSE;
	}
	else {
		cons = GNOME_DB_CONSTRAINT (list->data);
		g_slist_free (list);
		if (gnome_db_constraint_get_table (cons) == tab1)
			tab1_is_pkey = TRUE;
	}

	/* remove any pre-existing join condition */
	if (join->priv->cond)
		nullified_cond_cb (join->priv->cond, join);
	
	/* condition creation */
	fklist = gnome_db_constraint_fkey_get_fields (cons);
	if (g_slist_length (fklist) > 1)
		jcond = GNOME_DB_CONDITION (gnome_db_condition_new (join->priv->query, GNOME_DB_CONDITION_NODE_AND));
	list = fklist;
	while (list) {
		GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
		GnomeDbCondition *cond;
		GnomeDbQfield *qfield;
		GnomeDbTableField *dbfield;

		if (!jcond) {
			jcond = GNOME_DB_CONDITION (gnome_db_condition_new (join->priv->query, GNOME_DB_CONDITION_LEAF_EQUAL));
			cond = jcond;
		}
		else {
			cond = GNOME_DB_CONDITION (gnome_db_condition_new (join->priv->query, GNOME_DB_CONDITION_LEAF_EQUAL));
			g_assert (gnome_db_condition_node_add_child (jcond, cond, NULL));
			g_object_unref (cond);
		}
				
		/* left operator */
		if (tab1_is_pkey) 
			dbfield = pair->fkey;
		else
			dbfield = pair->ref_pkey;
		qfield = gnome_db_query_get_field_by_ref_field (join->priv->query,
							  target1, GNOME_DB_FIELD (dbfield), GNOME_DB_FIELD_ANY);
		if (!qfield) {
			qfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (join->priv->query, target1,
									  GNOME_DB_FIELD (dbfield)));
			gnome_db_qfield_set_visible (qfield, FALSE);
			gnome_db_entity_add_field (GNOME_DB_ENTITY (join->priv->query), GNOME_DB_FIELD (qfield));
			g_object_unref (qfield);
		}
		gnome_db_condition_leaf_set_operator (cond, GNOME_DB_CONDITION_OP_LEFT, qfield);

		/* right operator */
		if (! tab1_is_pkey) 
			dbfield = pair->fkey;
		else
			dbfield = pair->ref_pkey;
		qfield = gnome_db_query_get_field_by_ref_field (join->priv->query,
							  target2, GNOME_DB_FIELD (dbfield), GNOME_DB_FIELD_ANY);
		if (!qfield) {
			qfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (join->priv->query, target2,
									  GNOME_DB_FIELD (dbfield)));
			gnome_db_qfield_set_visible (qfield, FALSE);
			gnome_db_entity_add_field (GNOME_DB_ENTITY (join->priv->query), GNOME_DB_FIELD (qfield));
			g_object_unref (qfield);
		}
		gnome_db_condition_leaf_set_operator (cond, GNOME_DB_CONDITION_OP_RIGHT, qfield);
		
		g_free (pair);
		list = g_slist_next (list);
	}
	g_slist_free (fklist);

	gnome_db_join_set_condition (join, jcond);
	g_object_unref (jcond);

	return TRUE;
}

/**
 * gnome_db_join_set_condition_from_sql
 * @join: a #GnomeDbJoin object
 * @cond: a SQL expression
 * @error: place to store the error, or %NULL
 *
 * Parses @cond and if it represents a valid SQL expression to be @join's
 * condition, then set it to be @join's condition.
 *
 * Returns: a TRUE on success
 */
gboolean
gnome_db_join_set_condition_from_sql (GnomeDbJoin *join, const gchar *cond, GError **error)
{
	GnomeDbCondition *newcond;
	gboolean has_error = FALSE;
	GSList *targets = NULL;

	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), FALSE);
	g_return_val_if_fail (join->priv, FALSE);
	g_return_val_if_fail (cond && *cond, FALSE);

	newcond = (GnomeDbCondition *) gnome_db_condition_new_from_sql (join->priv->query, cond, &targets, error);
	if (newcond) {
		/* test nb of targets */
		if (g_slist_length (targets) != 2) {
			g_set_error (error,
				     GNOME_DB_JOIN_ERROR,
				     GNOME_DB_JOIN_SQL_ANALYSE_ERROR,
				     _("Join condition must be between two entities"));
			has_error = TRUE;
		}
		else {
			GnomeDbTarget *t1, *t2;
			GnomeDbTarget *nt1, *nt2;
			nt1 = (GnomeDbTarget *) targets->data;
			nt2 = (GnomeDbTarget *) targets->next->data;
			t1 = gnome_db_join_get_target_1 (join);
			t2 = gnome_db_join_get_target_2 (join);
			
			if (((t1 == nt1) && (t2 == nt2)) ||
			    ((t1 == nt2) && (t2 == nt1))) {
				gnome_db_join_set_condition (join, newcond);
			}
			else {
				g_set_error (error, GNOME_DB_JOIN_ERROR, GNOME_DB_JOIN_SQL_ANALYSE_ERROR,
					     _("Condition '%s' does not involve the same "
					       "entities as the join"), cond);
				has_error = TRUE;
			}
		}
		
		if (targets)
			g_slist_free (targets);
		g_object_unref (G_OBJECT (newcond));
	}
	else 
		has_error = TRUE;

	return !has_error;
}

/**
 * gnome_db_join_set_condition
 * @join: a #GnomeDbJoin object
 * @cond: a  #GnomeDbCondition object, or %NULL to remove the join's condition
 * 
 * Sets @cond to be @join's condition. This is possible only if @cond uses
 * query fields which are either of type GnomeDbQfField and reference one of the two
 * targets which @join uses, or any other query field type.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_join_set_condition (GnomeDbJoin *join, GnomeDbCondition *cond)
{
	GnomeDbTarget *t1, *t2;
	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), FALSE);
	g_return_val_if_fail (join->priv, FALSE);

	/* test if the condition is OK */
	if (! gnome_db_condition_represents_join (cond, &t1, &t2, NULL)) 
		return FALSE;

	if (! (((gnome_db_ref_base_get_ref_object (join->priv->target1) == (GnomeDbBase*) t1) && 
		(gnome_db_ref_base_get_ref_object (join->priv->target2) == (GnomeDbBase*) t2)) ||
	       ((gnome_db_ref_base_get_ref_object (join->priv->target1) == (GnomeDbBase*) t2) && 
		(gnome_db_ref_base_get_ref_object (join->priv->target2) == (GnomeDbBase*) t1))))
		return FALSE;

	/* actual work */
	if (join->priv->cond && (join->priv->cond != cond)) 
		nullified_cond_cb (join->priv->cond, join);
	
	if (cond) {
		g_object_ref (G_OBJECT (cond));
		gnome_db_base_connect_nullify (cond, G_CALLBACK (nullified_cond_cb), join);
		g_signal_connect (G_OBJECT (cond), "changed",
				  G_CALLBACK (cond_changed_cb), join);
		join->priv->cond = cond;
		g_object_set (G_OBJECT (cond), "join", join, NULL);
#ifdef debug_signal
		g_print (">> 'CONDITION_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit_by_name (G_OBJECT (join), "condition_changed");
#ifdef debug_signal
		g_print ("<< 'CONDITION_CHANGED' from %s\n", __FUNCTION__);
#endif
		gnome_db_base_changed (GNOME_DB_BASE (join));
	}

	return TRUE;
}

/**
 * gnome_db_join_get_condition
 * @join: a #GnomeDbJoin object
 *
 * Get the join's associated condition
 *
 * Returns: the #GnomeDbCondition object
 */
GnomeDbCondition *
gnome_db_join_get_condition (GnomeDbJoin *join)
{
	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), NULL);
	g_return_val_if_fail (join->priv, NULL);
	
	return join->priv->cond;
}


/**
 * gnome_db_join_render_type
 * @join: a #GnomeDbJoin object
 *
 * Get the SQL version of the join type ("INNER JOIN", "LEFT JOIN", etc)
 *
 * Returns: the type as a const string
 */
const gchar *
gnome_db_join_render_type (GnomeDbJoin *join)
{
	g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), NULL);
	g_return_val_if_fail (join->priv, NULL);

	switch (join->priv->join_type) {
	case GNOME_DB_JOIN_TYPE_INNER:
		return "INNER JOIN";
	case GNOME_DB_JOIN_TYPE_LEFT_OUTER:
		return "LEFT JOIN";
	case GNOME_DB_JOIN_TYPE_RIGHT_OUTER:
		return "RIGHT JOIN";
	case GNOME_DB_JOIN_TYPE_FULL_OUTER:
		return "FULL JOIN";
        case GNOME_DB_JOIN_TYPE_CROSS:
		return "CROSS JOIN";
	default:
		g_assert_not_reached ();
		return NULL;
	}
}

#ifdef debug
static void
gnome_db_join_dump (GnomeDbJoin *join, guint offset)
{
	gchar *str;
        guint i;
	
	g_return_if_fail (join && IS_GNOME_DB_JOIN (join));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (join->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbJoin" D_COL_NOR " %p ", str, join);
		if (gnome_db_join_is_active (GNOME_DB_REFERER (join)))
			g_print ("Active, Targets: %p -> %p ", gnome_db_ref_base_get_ref_object (join->priv->target1),
				 gnome_db_ref_base_get_ref_object (join->priv->target2));
		else
			g_print (D_COL_ERR "Non active" D_COL_NOR ", ");
		g_print ("requested targets: %s & %s", gnome_db_ref_base_get_ref_name (join->priv->target1, NULL, NULL),
			 gnome_db_ref_base_get_ref_name (join->priv->target2, NULL, NULL));
		if (join->priv->cond) {
			g_print (", Condition:\n");
			gnome_db_base_dump (GNOME_DB_BASE (join->priv->cond), offset+5);
		}
		else
			g_print ("\n");
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, join);
}
#endif



/* 
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_join_activate (GnomeDbReferer *iface)
{
	gboolean retval;
	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, FALSE);

	retval = gnome_db_ref_base_activate (GNOME_DB_JOIN (iface)->priv->target1);
	retval = retval && gnome_db_ref_base_activate (GNOME_DB_JOIN (iface)->priv->target2) && retval;
	if (GNOME_DB_JOIN (iface)->priv->cond)
		retval = retval && gnome_db_referer_activate (GNOME_DB_REFERER (GNOME_DB_JOIN (iface)->priv->cond));

	return retval;
}

static void
gnome_db_join_deactivate (GnomeDbReferer *iface)
{
	g_return_if_fail (iface && IS_GNOME_DB_JOIN (iface));
	g_return_if_fail (GNOME_DB_JOIN (iface)->priv);

	gnome_db_ref_base_deactivate (GNOME_DB_JOIN (iface)->priv->target1);
	gnome_db_ref_base_deactivate (GNOME_DB_JOIN (iface)->priv->target2);
	if (GNOME_DB_JOIN (iface)->priv->cond)
		gnome_db_referer_deactivate (GNOME_DB_REFERER (GNOME_DB_JOIN (iface)->priv->cond));
}

static gboolean
gnome_db_join_is_active (GnomeDbReferer *iface)
{
	gboolean retval;

	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, FALSE);

	retval = gnome_db_ref_base_is_active (GNOME_DB_JOIN (iface)->priv->target1);
	retval = retval && gnome_db_ref_base_is_active (GNOME_DB_JOIN (iface)->priv->target2);
	if (GNOME_DB_JOIN (iface)->priv->cond) 
		retval = retval && gnome_db_referer_is_active (GNOME_DB_REFERER (GNOME_DB_JOIN (iface)->priv->cond));
	
	return retval;
}

static GSList *
gnome_db_join_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
	GnomeDbBase *base;
	GnomeDbJoin *join;
	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), NULL);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, NULL);

	join = GNOME_DB_JOIN (iface);
	base = gnome_db_ref_base_get_ref_object (join->priv->target1);
	if (base)
		list = g_slist_append (list, base);
	base = gnome_db_ref_base_get_ref_object (join->priv->target2);
	if (base)
		list = g_slist_append (list, base);
	if (join->priv->cond) {
		GSList *list2 = gnome_db_condition_get_ref_objects_all (join->priv->cond);
		if (list2)
			list = g_slist_concat (list, list2);
	}

	return list;
}

static void
gnome_db_join_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbJoin *join;
	g_return_if_fail (iface && IS_GNOME_DB_JOIN (iface));
	g_return_if_fail (GNOME_DB_JOIN (iface)->priv);

	join = GNOME_DB_JOIN (iface);
	if (join->priv->query) {
		GnomeDbQuery *query = g_hash_table_lookup (replacements, join->priv->query);
		if (query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (join->priv->query),
							      G_CALLBACK (nullified_object_cb), join);
			g_signal_handlers_disconnect_by_func (G_OBJECT (join->priv->query),
							      G_CALLBACK (target_removed_cb), join);
			join->priv->query = query;
			gnome_db_base_connect_nullify (query, 
						 G_CALLBACK (nullified_object_cb), join);
			g_signal_connect (G_OBJECT (query), "target_removed",
					  G_CALLBACK (target_removed_cb), join);
		}
	}

	gnome_db_ref_base_replace_ref_object (join->priv->target1, replacements);
	gnome_db_ref_base_replace_ref_object (join->priv->target2, replacements);
	if (join->priv->cond) 
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (join->priv->cond), replacements);
}

/* 
 * GnomeDbXmlStorage interface implementation
 */

static const gchar *convert_join_type_to_str (GnomeDbJoinType type);
static GnomeDbJoinType   convert_str_to_join_type (const gchar *str);

static gchar *
gnome_db_join_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), NULL);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, NULL);

	return NULL;
}

static xmlNodePtr
gnome_db_join_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbJoin *join;
	gchar *str;
	const gchar *type;

	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), NULL);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, NULL);

	join = GNOME_DB_JOIN (iface);

	node = xmlNewNode (NULL, "GNOME_DB_JOIN");
	
	/* targets */
	if (join->priv->target1) {
		str = NULL;
		if (gnome_db_ref_base_is_active (join->priv->target1)) {
			GnomeDbBase *base = gnome_db_ref_base_get_ref_object (join->priv->target1);
			g_assert (base);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		}
		else 
			str = g_strdup (gnome_db_ref_base_get_ref_name (join->priv->target1, NULL, NULL));
		
		if (str) {
			xmlSetProp (node, "target1", str);
			g_free (str);
		}
	}

	if (join->priv->target2) {
		str = NULL;
		if (gnome_db_ref_base_is_active (join->priv->target2)) {
			GnomeDbBase *base = gnome_db_ref_base_get_ref_object (join->priv->target2);
			g_assert (base);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		}
		else 
			str = g_strdup (gnome_db_ref_base_get_ref_name (join->priv->target2, NULL, NULL));
		
		if (str) {
			xmlSetProp (node, "target2", str);
			g_free (str);
		}
	}

	/* join type */
	type = convert_join_type_to_str (join->priv->join_type);
	xmlSetProp (node, "join_type", type);

	/* join condition */
	if (join->priv->cond) {
		xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (join->priv->cond), error);
		if (sub)
                        xmlAddChild (node, sub);
                else {
                        xmlFreeNode (node);
                        return NULL;
                }
	}

	return node;
}


static gboolean
gnome_db_join_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbJoin *join;
	gchar *prop;
	gboolean t1 = FALSE, t2 = FALSE;
	xmlNodePtr children;

	g_return_val_if_fail (iface && IS_GNOME_DB_JOIN (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_JOIN (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	join = GNOME_DB_JOIN (iface);
	if (strcmp (node->name, "GNOME_DB_JOIN")) {
		g_set_error (error,
			     GNOME_DB_JOIN_ERROR,
			     GNOME_DB_JOIN_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_JOIN>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "target1");
	if (prop) {
		if (join->priv->target1) {
			gnome_db_ref_base_set_ref_name (join->priv->target1, GNOME_DB_TARGET_TYPE, 
						  REFERENCE_BY_XML_ID, prop);
			t1 = TRUE;
		}
		g_free (prop);
	}
	prop = xmlGetProp (node, "target2");
	if (prop) {
		if (join->priv->target2) {
			gnome_db_ref_base_set_ref_name (join->priv->target2, GNOME_DB_TARGET_TYPE, 
						  REFERENCE_BY_XML_ID, prop);
			t2 = TRUE;
		}
		g_free (prop);
	}
	prop = xmlGetProp (node, "join_type");
	if (prop) {
		join->priv->join_type = convert_str_to_join_type (prop);
		g_free (prop);
	}

	/* children nodes */
	children = node->children;
	while (children) {
		if (!strcmp (children->name, "GNOME_DB_COND")) {
			GnomeDbCondition *cond;

			cond = GNOME_DB_CONDITION (gnome_db_condition_new (join->priv->query, GNOME_DB_CONDITION_NODE_AND));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (cond), children, error)) {
				gnome_db_join_set_condition (join, cond);
				g_object_unref (G_OBJECT (cond));
			}
			else
				return FALSE;
                }

		children = children->next;
	}
	if (t1 && t2)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_JOIN_ERROR,
			     GNOME_DB_JOIN_XML_LOAD_ERROR,
			     _("Problem loading <GNOME_DB_JOIN>"));
		return FALSE;
	}
}

static const gchar *
convert_join_type_to_str (GnomeDbJoinType type)
{
	switch (type) {
	default:
	case GNOME_DB_JOIN_TYPE_INNER:
		return "INNER";
	case GNOME_DB_JOIN_TYPE_LEFT_OUTER:
		return "LEFT";
	case GNOME_DB_JOIN_TYPE_RIGHT_OUTER:
		return "RIGHT";
	case GNOME_DB_JOIN_TYPE_FULL_OUTER:
		return "FULL";
        case GNOME_DB_JOIN_TYPE_CROSS:
		return "CROSS";
	}
}

static GnomeDbJoinType
convert_str_to_join_type (const gchar *str)
{
	switch (*str) {
	case 'I':
	default:
		return GNOME_DB_JOIN_TYPE_INNER;
	case 'L':
		return GNOME_DB_JOIN_TYPE_LEFT_OUTER;
	case 'R':
		return GNOME_DB_JOIN_TYPE_RIGHT_OUTER;
	case 'F':
		return GNOME_DB_JOIN_TYPE_FULL_OUTER;
	case 'C':
		return GNOME_DB_JOIN_TYPE_CROSS;
	}
}
