#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/libgnomedb.h>
#include <gtk/gtk.h>
#define BUILD_WITH_GNOME

typedef struct {
	/* General section */
	GnomeDbDict   *dict;
	gchar         *dsn;
	GtkWidget     *mainwin;

	/* Data types page */
	gint           dt_mode;
	GtkTextBuffer *dt_textbuffer;
	GtkWidget     *dt_selector;

	/* Database page */
	GtkTextBuffer *db_textbuffer;
	GtkWidget     *db_selector;
	GtkWidget     *db_view_contents;
} BrowserConfig;

static gboolean delete_event (GtkWidget *widget, GdkEvent  *event, gpointer   data )
{
    return FALSE;
}

static void destroy (GtkWidget *widget, gpointer   data )
{
    gtk_main_quit ();
}

static GtkWidget *build_menu (BrowserConfig *config, GtkWidget *mainwin);
static GtkWidget *build_data_types_page (BrowserConfig *config);
static GtkWidget *build_tables_views_page (BrowserConfig *config);
static void       load_file (BrowserConfig *config);
int 
main (int argc, char **argv)
{
	GtkWidget *mainwin, *vbox, *menu, *nb, *label, *page;
	GnomeDbDict *dictionary;
	BrowserConfig *config;
	GdkPixbuf *icon;
	
	/* Initialize i18n support */
	gtk_set_locale ();
	
	/* Initialize the widget set */
	gtk_init (&argc, &argv);

	/* Browser config */
	config = g_new0 (BrowserConfig, 1);
	dictionary = GNOME_DB_DICT (gnome_db_dict_new ());
	config->dict = dictionary;
	g_object_set (G_OBJECT (gnome_db_dict_get_server (dictionary)), "with_functions", TRUE, NULL);

	/* Application init */
	if (argc > 1) {
		/* use the DSN source given as argument */
		GnomeDbServer *srv;
		
		srv = gnome_db_dict_get_server (config->dict);
		if (! gnome_db_server_set_datasource (srv, argv[1])) {
			g_print ("Datasource '%s' does not exist.\n", argv[1]);
			exit (1);
		}
		config->dsn = g_strdup (argv[1]);
	}
	else {
		g_print ("Usage: %s <Data source name>\n\n", argv[0]);
		exit (1);
	}
	load_file (config);
	
	/* Create the main window */
	mainwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width (GTK_CONTAINER (mainwin), 0);
	g_signal_connect (G_OBJECT (mainwin), "delete_event",
			  G_CALLBACK (delete_event), NULL);
	g_signal_connect (G_OBJECT (mainwin), "destroy",
			  G_CALLBACK (destroy), NULL);
	config->mainwin = mainwin;

	gtk_window_set_title (GTK_WINDOW (mainwin), _("Database browser"));
	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (mainwin), icon);
		g_object_unref (icon);
	}

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (mainwin), vbox);
	gtk_widget_show (vbox);

	/* menu */
	menu = build_menu (config, mainwin);
	gtk_widget_show (menu);
	gtk_box_pack_start (GTK_BOX (vbox), menu, FALSE, FALSE, 0);

	/* main area */
	nb = gtk_notebook_new ();
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);

	/* tables and views page */
	page = build_tables_views_page (config);
	label = gtk_label_new (_("Tables and Views"));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), page, label);

	/* data types page */
	page = build_data_types_page (config);
	label = gtk_label_new (_("Data types"));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), page, label);

	/* Show the application window */
	gtk_widget_set_size_request (mainwin, 700, 600);
	gtk_widget_show_all (mainwin);
	
	gtk_main ();
	g_object_unref (G_OBJECT (dictionary));

	return 0;
}

static gboolean open_connection (BrowserConfig *config, gboolean force_update);
static void update_dbms_data_cb (GtkWidget *wid, BrowserConfig *config);
static void
load_file (BrowserConfig *config)
{
	GError *error = NULL;
	GnomeDbServer *server = gnome_db_dict_get_server (config->dict);

	gnome_db_server_reset (gnome_db_dict_get_server (config->dict));
	/* actual file loading */
	if (!gnome_db_dict_load_xml (config->dict, &error)) {
		if (error && (error->code == GNOME_DB_DICT_LOAD_FILE_NOT_EXIST_ERROR)) {
			GtkWidget *msg;
			gchar *str;

			str = g_strdup_printf ("<b>%s</b>\n\n%s",
					       _("Do you want to build the database dictionary?"),
					       _("The database dictionary must be built before the database can be browsed. "
						 "The process only takes a few minutes.\n"
						 "\n"
						 "The dictionary will need to be rebuilt only if the database structure "
                                                 "changes."));
			
			msg = gtk_message_dialog_new_with_markup (GTK_WINDOW (config->mainwin), 
								  GTK_DIALOG_DESTROY_WITH_PARENT,
								  GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, 
								  str);
			g_free (str);
			if (gtk_dialog_run (GTK_DIALOG (msg)) == GTK_RESPONSE_YES) 
				open_connection (config, TRUE);
			else {
				g_print ("Can't build the dictionary...\n");
				exit (1);
			}
			gtk_widget_destroy (msg);
			g_error_free (error);
			error = NULL;
		}
		else {
			g_print ("Error loading dictionary file:\n%s\n",
				 error ? error->message : _("unreported error"));
			exit (1);
		}
	}
	else 
		open_connection (config, FALSE);
}

static gboolean
open_connection (BrowserConfig *config, gboolean force_update)
{
	GtkWidget *dialog, *login;
	gchar *title;
	gboolean conn_opened = FALSE;
	GError *error = NULL;
	GnomeDbServer *server = gnome_db_dict_get_server (config->dict);
	
	title = g_strdup_printf (_("Login for %s"), config->dsn);
	dialog = gnome_db_login_dialog_new (title);
	g_free (title);
	login = gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dialog));
	gnome_db_login_set_dsn (GNOME_DB_LOGIN (login), config->dsn);
	gnome_db_login_set_show_dsn_selector (GNOME_DB_LOGIN (login), FALSE);
	
	while (!conn_opened && gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
		gnome_db_server_set_user_name (server, 
					       gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (dialog)));
		gnome_db_server_set_user_password (server, 
						   gnome_db_login_dialog_get_password (GNOME_DB_LOGIN_DIALOG (dialog)));
		
		conn_opened = gnome_db_server_open_connect (server, &error);
		if (!conn_opened) {
			gnome_db_show_error (error->message);
			g_error_free (error);
			error = NULL;
		}
		else 
			if (force_update)
				update_dbms_data_cb (NULL, config);
	}
	gtk_widget_destroy (dialog);
	if (!conn_opened) {
		g_print ("Can't open connection...\n");
		exit (1);
	}
}


/*
 * Menu
 */
GtkWidget *
build_menu (BrowserConfig *config, GtkWidget *mainwin)
{
	GtkWidget *menubar1, *menuitem1, *menuitem1_menu, *entry;
	GtkAccelGroup *accel_group;

	accel_group = gtk_accel_group_new ();

	menubar1 = gtk_menu_bar_new ();
	gtk_widget_show (menubar1);

	/* File menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_File"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);
	
	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (destroy), config);

	/* Database menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_Database"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);

	entry = gtk_menu_item_new_with_mnemonic (_("Synchronise metadata with DBMS"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (update_dbms_data_cb), config);


	gtk_window_add_accel_group (GTK_WINDOW (mainwin), accel_group);

	return menubar1;
}

static void stop_dbms_update_cb (GtkWidget *dlg, gint response, BrowserConfig *config);
static void
update_dbms_data_cb (GtkWidget *wid, BrowserConfig *config)
{
	GtkWidget *dlg, *updview;
	GnomeDbDatabase *db;
	GnomeDbServer *srv;
	GError *error = NULL;
	gboolean stopped = FALSE;

	db = gnome_db_dict_get_database (config->dict);
	srv = gnome_db_dict_get_server (config->dict);
	g_assert (gnome_db_server_conn_is_opened (srv));

	dlg = gtk_dialog_new_with_buttons (_("Metadata synchronisation"),
					   GTK_WINDOW (config->mainwin), 
					   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (stop_dbms_update_cb), config);
	updview = gnome_db_dbms_update_viewer_new (config->dict);
	gtk_widget_show (updview);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dlg)->vbox), updview, TRUE, TRUE, 0);
	gtk_widget_show (dlg);
	/* data types, functions and aggregates */
	if (!gnome_db_server_update_dbms_data (srv, &error)) {
		if (error->code != GNOME_DB_SERVER_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Server metadata:\n%s\n"), error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		else
			stopped = TRUE;
		g_error_free (error);
		error = NULL;	
	}
	/* database */
	if (!stopped && !gnome_db_database_update_dbms_data (db, &error)) {
		if (error->code != GNOME_DB_DATABASE_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Database metadata:\n%s\n"), error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		else
			stopped = TRUE;	
		g_error_free (error);
		error = NULL;	
	}

	gtk_widget_destroy (dlg);
	if (!stopped) {
		if (!gnome_db_dict_save_xml (config->dict, &error)) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						      _("Error saving dictionary file:\n%s\n"),
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
			g_error_free (error);
			error = NULL;
		}
	}
}

static void
stop_dbms_update_cb (GtkWidget *dlg, gint response, BrowserConfig *config)
{
	gnome_db_server_stop_update_dbms_data (gnome_db_dict_get_server (config->dict));
}


/*
 * Data types page
 */
static void dt_selection_changed_cb (GnomeDbSelector *mgsel, GObject *dt, BrowserConfig *config);
static void dt_mode_changed_cb (GtkWidget *wid, BrowserConfig *config);
static GtkWidget *
build_data_types_page (BrowserConfig *config)
{
	GtkWidget *wid, *bbox, *label, *paned, *vb;
	GtkWidget *sw, *textview;
	GtkWidget *rd1, *rd2, *rd3;
	GtkTextBuffer *buffer;
	gchar *str;

	paned = gtk_hpaned_new ();
	gtk_container_set_border_width (GTK_CONTAINER (paned), 5);

	/* left part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add1 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Select a data type:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);

	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);
	wid = gnome_db_selector_new (config->dict, NULL, GNOME_DB_SELECTOR_DATA_TYPES, 
			       GNOME_DB_SELECTOR_COLUMN_OWNER | GNOME_DB_SELECTOR_COLUMN_COMMENTS);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (dt_selection_changed_cb), config);
	config->dt_selector = wid;
	gtk_widget_set_size_request (wid, 400, 300);

	/* right part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add2 (GTK_PANED (paned), vb);
	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Select a filter option:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);

	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);
	bbox = gtk_vbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), bbox, FALSE, FALSE, 0);

	rd1 = gtk_radio_button_new_with_label (NULL, _("Functions returning this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd1, FALSE, FALSE, 0);
	g_object_set_data (G_OBJECT (rd1), "mode", GINT_TO_POINTER (0));
	g_signal_connect (G_OBJECT (rd1), "toggled",
			  G_CALLBACK (dt_mode_changed_cb), config);

	rd2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rd1),
							   _("Functions using this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd2, FALSE, FALSE, 0);
	g_object_set_data (G_OBJECT (rd2), "mode", GINT_TO_POINTER (1));
	g_signal_connect (G_OBJECT (rd2), "toggled",
			  G_CALLBACK (dt_mode_changed_cb), config);

	rd3 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rd1),
							   _("Aggregates using this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd3, FALSE, FALSE, 0);
	g_object_set_data (G_OBJECT (rd3), "mode", GINT_TO_POINTER (2));
	g_signal_connect (G_OBJECT (rd3), "toggled",
			  G_CALLBACK (dt_mode_changed_cb), config);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Result of filter:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);

	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vb), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	textview = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), textview);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (textview), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (textview), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (textview), FALSE);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));
	config->dt_textbuffer = buffer;
	gtk_text_buffer_set_text (buffer, _("Select a data type..."), -1);

	gtk_text_buffer_create_tag (buffer, "funcname",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "red", NULL);

	gtk_text_buffer_create_tag (buffer, "descr",
				    "style", PANGO_STYLE_ITALIC, NULL);

	return paned;
}

static void
dt_mode_changed_cb (GtkWidget *wid, BrowserConfig *config)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wid))) {
		config->dt_mode = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "mode"));
		dt_selection_changed_cb (GNOME_DB_SELECTOR (config->dt_selector), NULL, config);
	}
}

static gchar *
function_get_args (GnomeDbServerFunction *func)
{
	GString *string;
	const GSList *args;
	gchar *retval;
	gboolean firstarg = TRUE;

	string = g_string_new ("");
	args = gnome_db_server_function_get_arg_types (func);
	g_string_append (string, " (");
	while (args) {
		if (firstarg)
			firstarg = FALSE;
		else
			g_string_append (string, ", ");
		if (args->data) 
			g_string_append (string,
					 gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (args->data)));
		else
			g_string_append (string, "*");
				 
		args = g_slist_next (args);
	}
	g_string_append (string, ")");
	retval = string->str;
	g_string_free (string, FALSE);

	return retval;
}


static void
dt_selection_changed_cb (GnomeDbSelector *mgsel, GObject *dt, BrowserConfig *config)
{
	GtkTextIter start, end;

	/* empty the text buffer */
	gtk_text_buffer_get_start_iter (config->dt_textbuffer, &start);
	gtk_text_buffer_get_end_iter (config->dt_textbuffer, &end);
	gtk_text_buffer_delete (config->dt_textbuffer, &start, &end);

	if (!dt)
		dt = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->dt_selector));

	if (!dt)
		gtk_text_buffer_set_text (config->dt_textbuffer, _("Select a data type..."), -1);
	else {
		GSList *list, *funcs, *args, *aggs;
		gboolean getfunc;
		GnomeDbServerDataType *ldt;
		
		funcs = gnome_db_server_get_functions (gnome_db_dict_get_server (config->dict));
		aggs = gnome_db_server_get_aggregates (gnome_db_dict_get_server (config->dict));
		list = g_slist_concat (funcs, aggs);
		funcs = list;
		while (list) {
			getfunc = FALSE;
			switch (config->dt_mode) {
			case 0:
				if (IS_GNOME_DB_SERVER_FUNCTION (list->data)) {
					ldt = gnome_db_server_function_get_ret_type (GNOME_DB_SERVER_FUNCTION (list->data));
					getfunc = (G_OBJECT (ldt) == dt) ? TRUE : FALSE;
				}
				break;
			case 1:
				if (IS_GNOME_DB_SERVER_FUNCTION (list->data)) {
					args = gnome_db_server_function_get_arg_types (GNOME_DB_SERVER_FUNCTION (list->data));
					while (args && !getfunc) {
						if (args->data == (gpointer) dt)
							getfunc = TRUE;
						args = g_slist_next (args);
					}
				}
				break;
			case 2:
				if (IS_GNOME_DB_SERVER_AGGREGATE (list->data)) {
					ldt = gnome_db_server_aggregate_get_arg_type (GNOME_DB_SERVER_AGGREGATE (list->data));
					getfunc = !ldt || (G_OBJECT (ldt) == dt) ? TRUE : FALSE;
				}
				break;
			}

			if (getfunc) {
				gchar *str;

				str = gnome_db_base_get_name (GNOME_DB_BASE (list->data));
				gtk_text_buffer_get_end_iter (config->dt_textbuffer, &end);
				gtk_text_buffer_insert_with_tags_by_name (config->dt_textbuffer, 
									  &end, str, -1, 
									  "funcname", NULL);

				if (IS_GNOME_DB_SERVER_FUNCTION (list->data))
					str = function_get_args (GNOME_DB_SERVER_FUNCTION (list->data));
				else {
					if ((ldt = gnome_db_server_aggregate_get_arg_type (GNOME_DB_SERVER_AGGREGATE (list->data)))) {
						str = g_strdup_printf (" (%s)", gnome_db_base_get_name (GNOME_DB_BASE (ldt)));
					}
					else
						str = g_strdup (" (*)");
				}
				gtk_text_buffer_get_end_iter (config->dt_textbuffer, &end);
				gtk_text_buffer_insert (config->dt_textbuffer, &end, str, -1);
				g_free (str);

				str = gnome_db_base_get_description (GNOME_DB_BASE (list->data));
				if (str && *str) {
					gchar *str2 = g_strdup_printf (" -- %s\n", str);
					gtk_text_buffer_get_end_iter (config->dt_textbuffer, &end);
					gtk_text_buffer_insert_with_tags_by_name (config->dt_textbuffer, &end, str2, -1, 
										  "descr", NULL);
					g_free (str2);
				}
				else {
					gtk_text_buffer_get_end_iter (config->dt_textbuffer, &end);
					gtk_text_buffer_insert (config->dt_textbuffer, &end, "\n", -1);
				}
			}
			list = g_slist_next (list);
		}
		g_slist_free (funcs);
	}
}












/**
 * build_tables_views_page
 *
 * Build the Tables and Views page
 */
static void tables_selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, BrowserConfig *config);
static void table_view_contents (GtkWidget *button, BrowserConfig *config);
static GtkWidget *
build_tables_views_page (BrowserConfig *config)
{
	GtkWidget *wid, *label, *paned, *vb;
	GtkWidget *sw, *textview;
	GtkTextBuffer *buffer;
	gchar *str;

	paned = gtk_hpaned_new ();
	gtk_container_set_border_width (GTK_CONTAINER (paned), 5);

	/* left part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add1 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Select a table or field:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);

	wid = gnome_db_selector_new (config->dict, NULL, GNOME_DB_SELECTOR_TABLES | GNOME_DB_SELECTOR_FIELDS, 
			       GNOME_DB_SELECTOR_COLUMN_OWNER | GNOME_DB_SELECTOR_COLUMN_TYPE | GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH |
			       GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL | GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT);

	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (tables_selection_changed_cb), config);
	gtk_widget_set_size_request (wid, 400, 300);
	config->db_selector = wid;

	/* right part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add2 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Table's operations:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);

	wid = gtk_button_new_with_label (_("Edit contents"));
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, TRUE, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (table_view_contents), config);
	gtk_widget_set_sensitive (wid, FALSE);
	config->db_view_contents = wid;

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Table and field's properties:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vb), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	textview = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), textview);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (textview), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (textview), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (textview), FALSE);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));
	config->db_textbuffer = buffer;
	gtk_text_buffer_set_text (buffer, _("Select a table or a table's field..."), -1);

	gtk_text_buffer_create_tag (buffer, "header",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "red", NULL);

	gtk_text_buffer_create_tag (buffer, "section",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "blue", NULL);
	
	return paned;
}

static void
tables_selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, BrowserConfig *config)
{
	GtkTextIter start, end, current;

	/* empty the text buffer */
	gtk_text_buffer_get_start_iter (config->db_textbuffer, &start);
	gtk_text_buffer_get_end_iter (config->db_textbuffer, &end);
	gtk_text_buffer_delete (config->db_textbuffer, &start, &end);

	if (!obj)
		gtk_text_buffer_set_text (config->db_textbuffer, _("Select a table or a table's field..."), -1);
	else {
		gtk_text_buffer_get_start_iter (config->db_textbuffer, &current);
		if (IS_GNOME_DB_TABLE (obj)) {
			gchar *str;
			GnomeDbTable *table = GNOME_DB_TABLE (obj);
			GSList *constraints, *list;

			if (gnome_db_table_is_view (table))
				str = _("View:");
			else
				str = _("Table:");

			gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer, 
								  &current, str, -1, 
								  "section", NULL);
			gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer, 
								  &current, " ", -1, 
								  "section", NULL);
			gtk_text_buffer_insert (config->db_textbuffer, &current, gnome_db_base_get_name (GNOME_DB_BASE (obj)), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n\n", -1);
			
			/* constraints list */
			constraints = gnome_db_table_get_constraints (table);

			/* PKey */
			list = constraints;
			while (list) {
				gboolean first = TRUE;
				if (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == 
				    CONSTRAINT_PRIMARY_KEY) {
					GnomeDbConstraint *cstr = GNOME_DB_CONSTRAINT (list->data);
					GSList *fields, *list2;
					gboolean header = FALSE;

					fields = gnome_db_constraint_pkey_get_fields (cstr);
					list2 = fields;
					while (list2) {
						if (!header) {
							header = TRUE;
							gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer,
												  &current, 
												  _("Primary key"), -1,
												  "section", NULL);
							gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
						}

						if (first) 
							first = FALSE;
						else
							gtk_text_buffer_insert (config->db_textbuffer, &current, ", ", -1);

						gtk_text_buffer_insert (config->db_textbuffer, &current, 
									gnome_db_base_get_name (GNOME_DB_BASE (list2->data)), -1);
						list2 = g_slist_next (list2);
					}
					g_slist_free (fields);
					gtk_text_buffer_insert (config->db_textbuffer, &current, "\n\n", -1);
				}
				list = g_slist_next (list);
			}

			/* FKey */
			list = constraints;
			while (list) {
				if (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == 
				    CONSTRAINT_FOREIGN_KEY) {
					GnomeDbConstraint *cstr = GNOME_DB_CONSTRAINT (list->data);
					GSList *fields, *list2;
					gboolean header = FALSE;

					fields = gnome_db_constraint_fkey_get_fields (cstr);
					list2 = fields;
					while (list2) {
						GnomeDbEntity *ent;
						if (!header) {
							header = TRUE;
							gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer,
												  &current, 
												  _("Foreign key"), -1,
												  "section", NULL);
							gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
						}
						
						str = gnome_db_base_get_name (GNOME_DB_BASE (GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->fkey));
						gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
						gtk_text_buffer_insert (config->db_textbuffer, &current, " --> ", -1);
						ent = gnome_db_field_get_entity (GNOME_DB_FIELD (GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
						str = gnome_db_base_get_name (GNOME_DB_BASE (ent));
						gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
						gtk_text_buffer_insert (config->db_textbuffer, &current, ".", -1);
						str = gnome_db_base_get_name (GNOME_DB_BASE (GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
						gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
						gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
						
						list2 = g_slist_next (list2);
					}
					g_slist_free (fields);
					gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
				}
				list = g_slist_next (list);
			}

			/* Unique */
			list = constraints;
			while (list) {
				if (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == 
				    CONSTRAINT_UNIQUE) {
					GnomeDbConstraint *cstr = GNOME_DB_CONSTRAINT (list->data);
					GSList *fields, *list2;
					gboolean header = FALSE;

					fields = gnome_db_constraint_unique_get_fields (cstr);
					list2 = fields;
					while (list2) {
						if (!header) {
							header = TRUE;
							gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer,
												  &current, 
												  _("UNIQUE constraint"), -1,
												  "section", NULL);
							gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
						}
						else
							gtk_text_buffer_insert (config->db_textbuffer, &current, ", ", -1);

						gtk_text_buffer_insert (config->db_textbuffer, &current, 
									gnome_db_base_get_name (GNOME_DB_BASE (list2->data)), -1);
						
						list2 = g_slist_next (list2);
					}
					g_slist_free (fields);
					gtk_text_buffer_insert (config->db_textbuffer, &current, "\n\n", -1);
				}
				list = g_slist_next (list);
			}

			/* check constraint: FIXME */

			g_slist_free (constraints);
			
		}

		if (IS_GNOME_DB_TABLE_FIELD (obj)) {
			gchar *str;
			GnomeDbServerDataType *type;
			GnomeDbTableField *field = GNOME_DB_TABLE_FIELD (obj);

			gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer, 
								  &current, _("Field:"), -1, 
								  "section", NULL);
			gtk_text_buffer_insert_with_tags_by_name (config->db_textbuffer, 
								  &current, " ", -1, 
								  "section", NULL);
			gtk_text_buffer_insert (config->db_textbuffer, &current, gnome_db_base_get_name (GNOME_DB_BASE (obj)), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n\n", -1);

			type = gnome_db_field_get_data_type (GNOME_DB_FIELD (field));
			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Data type:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, gnome_db_base_get_name (GNOME_DB_BASE (type)), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Description:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, gnome_db_base_get_description (GNOME_DB_BASE (obj)), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Length:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			if (gnome_db_table_field_get_length (field) != -1) {
				str = g_strdup_printf ("%d", gnome_db_table_field_get_length (field));
				gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
				g_free (str);
			}
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Scale:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			if (gnome_db_table_field_get_scale (field) != -1) {
				str = g_strdup_printf ("%d", gnome_db_table_field_get_scale (field));
				gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
				g_free (str);
			}
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			
			gtk_text_buffer_insert (config->db_textbuffer, &current, _("NULL allowed:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			str = gnome_db_table_field_is_null_allowed (field) ? _("Yes") : _("No");
			gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Primary key:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			if (gnome_db_table_field_is_pkey_part (field)) 
				str = gnome_db_table_field_is_pkey_alone (field) ? _("Primary key") : 
					_("Part of primary key");
			else
				str = _("No");
			gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);

			gtk_text_buffer_insert (config->db_textbuffer, &current, _("Foreign key:"), -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, " ", -1);
			if (gnome_db_table_field_is_fkey_part (field)) 
				str = gnome_db_table_field_is_fkey_alone (field) ? _("Foreign key") : 
					_("Part of foreign key");
			else
				str = _("No");
			gtk_text_buffer_insert (config->db_textbuffer, &current, str, -1);
			gtk_text_buffer_insert (config->db_textbuffer, &current, "\n", -1);
		}
	}


	gtk_widget_set_sensitive (config->db_view_contents, obj && IS_GNOME_DB_TABLE (obj));
}


static void dialog_exec_response_cb (GtkDialog *dlg, gint response, GObject *obj);
static void
table_view_contents (GtkWidget *button, BrowserConfig *config)
{
	GnomeDbQuery *query;
	GnomeDbTable *table;
	GnomeDbTarget *target;
	GnomeDbQfield *field;
	GObject *sel;
	GtkWidget *grid, *dlg;
	gchar *str;

	sel = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->db_selector));
	if (!sel || !IS_GNOME_DB_TABLE (sel))
		return;

	table = GNOME_DB_TABLE (sel);
	dlg = g_object_get_data (G_OBJECT (table), "contents");

	if (!dlg) {
		/* Query preparation: "SELECT * FROM table" */
		query = GNOME_DB_QUERY (gnome_db_query_new (config->dict));
		gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_SELECT);
		
		target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (table)));
		gnome_db_query_add_target (query, target, NULL);
		g_object_unref (G_OBJECT (target));
		
		field = GNOME_DB_QFIELD (gnome_db_qf_all_new_with_target (query, target));;
		gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (field));
		g_object_unref (G_OBJECT (field));
		
		
		/* Grid */
		if (gnome_db_table_is_view (table)) {
			str = g_strdup_printf (_("Contents of view '%s'"), gnome_db_base_get_name (GNOME_DB_BASE (table)));
			grid = gnome_db_grid_new_with_select_query (query, NULL);
		}
		else {
			str = g_strdup_printf (_("Contents of table '%s'"), gnome_db_base_get_name (GNOME_DB_BASE (table)));
			grid = gnome_db_grid_new_with_select_query (query, target);
		}
		g_object_set (G_OBJECT (grid), "title_string", str, NULL);
		dlg = gtk_dialog_new_with_buttons (str, GTK_WINDOW (config->mainwin), 0,
						   GTK_STOCK_CLOSE,
						   GTK_RESPONSE_ACCEPT, NULL);
		g_free (str);
		gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dlg)->vbox), grid);
		gtk_widget_set_size_request (grid, 400, 250);
		
		gtk_widget_show_all (dlg);
		/* gnome_db_data_widget_run (GNOME_DB_DATA_WIDGET (grid),  */
/* 					  GNOME_DB_ACTION_ASK_CONFIRM_DELETE | GNOME_DB_ACTION_ASK_CONFIRM_UPDATE |  */
/* 					  GNOME_DB_ACTION_ASK_CONFIRM_INSERT); */
		g_object_unref (G_OBJECT (query));
		g_object_set_data (G_OBJECT (table), "contents", dlg);
		g_signal_connect (G_OBJECT (dlg), "response",
				  G_CALLBACK (dialog_exec_response_cb), table);
	}
	else
		gtk_window_present (GTK_WINDOW (dlg));
}

static void
dialog_exec_response_cb (GtkDialog *dlg, gint response, GObject *obj)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
	g_object_set_data (G_OBJECT (obj), "contents", NULL);
}
