/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <bonobo/bonobo-i18n.h>
#include <bonobo/bonobo-ui-main.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtkstock.h>
#include <glade/glade-init.h>
#include <libgda/libgda.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomedb/libgnomedb.h>

static GtkStockItem builtin_icons [] = {
    { GNOME_DB_STOCK_DATABASE, N_("Database"), 0, 0, GETTEXT_PACKAGE },
    { GNOME_DB_STOCK_DESIGNER, N_("Designer"), 0, 0, GETTEXT_PACKAGE },
    { GNOME_DB_STOCK_SQL, N_("SQL"), 0, 0, GETTEXT_PACKAGE },
    { GNOME_DB_STOCK_TABLES, N_("Tables"), 0, 0, GETTEXT_PACKAGE },
};

/*
 * Private functions
 */

static void
add_sized (GtkIconFactory *factory,
	   const gchar *pixfile,
	   GtkIconSize size,
	   const gchar *stock_id)
{
	GtkIconSet *set;
	GtkIconSource *source;
	GdkPixbuf *pixbuf;
	GError *err = NULL;

	pixbuf = gdk_pixbuf_new_from_file (pixfile, &err);
	if (err) {
		gda_log_error (_("Error: %s"), err->message);
		g_error_free (err);
		return;
	}

	source = gtk_icon_source_new ();
	gtk_icon_source_set_pixbuf (source, pixbuf);
	gtk_icon_source_set_size (source, size);

	set = gtk_icon_set_new ();
	gtk_icon_set_add_source (set, source);

	gtk_icon_factory_add (factory, stock_id, set);

	g_object_unref (G_OBJECT (pixbuf));
	gtk_icon_source_free (source);
	gtk_icon_set_unref (set);
}

static void
get_stock_icons (GtkIconFactory *factory)
{
	add_sized (factory, LIBGNOMEDB_ICONSDIR "/gnome-db-database_24x24.png",
		   GTK_ICON_SIZE_BUTTON, GNOME_DB_STOCK_DATABASE);
	add_sized (factory,  LIBGNOMEDB_ICONSDIR "/gnome-db-designer_24x24.png",
		   GTK_ICON_SIZE_BUTTON, GNOME_DB_STOCK_DESIGNER);
	add_sized (factory,  LIBGNOMEDB_ICONSDIR "/gnome-db-sql_24x24.png",
		   GTK_ICON_SIZE_BUTTON, GNOME_DB_STOCK_SQL);
	add_sized (factory,  LIBGNOMEDB_ICONSDIR "/gnome-db-tables_24x24.png",
		   GTK_ICON_SIZE_BUTTON, GNOME_DB_STOCK_TABLES);
}

/*
 * Public functions
 */

void
gnome_db_init (const gchar *app_id, const gchar *version, gint nargs, gchar *args[])
{
	GtkIconFactory *factory;
	static gboolean initialized = FALSE;

	if (initialized) {
		gda_log_error (_("Attempt to initialize an already initialized client"));
		return;
	}

	gda_init (app_id, version, nargs, args);
	gnome_program_init (app_id, version, LIBGNOMEUI_MODULE, nargs, args, NULL);

	if (!bonobo_ui_is_initialized ())
		bonobo_ui_init (app_id, version, &nargs, args);

	glade_init ();

	/* install our stock icons */
	factory = gtk_icon_factory_new ();
	get_stock_icons (factory);
	gtk_icon_factory_add_default (factory);

	gtk_stock_add_static (builtin_icons, G_N_ELEMENTS (builtin_icons));
}

typedef struct {
	GdaInitFunc init_func;
	gpointer user_data;
} InitCbData;

static gboolean
idle_cb (gpointer user_data)
{
	InitCbData *cb_data = (InitCbData *) user_data;

	g_return_val_if_fail (cb_data != NULL, FALSE);

	if (cb_data->init_func)
		cb_data->init_func (cb_data->user_data);

	g_free (cb_data);

	return FALSE;
}

/**
 * gnome_db_main_run
 * @init_func: function to be called when everything has been initialized
 * @user_data: data to be passed to the init function
 *
 * Runs the GNOME-DB main loop, which is nothing more than the Bonobo UI main
 * loop, but with internally added stuff specific for applications using
 * libgda/libgnomedb.
 *
 * You can specify a function to be called after everything has been correctly
 * initialized (that is, for initializing your own stuff).
 */
void
gnome_db_main_run (GdaInitFunc init_func, gpointer user_data)
{
	if (init_func) {
		InitCbData *cb_data;

		cb_data = g_new (InitCbData, 1);
		cb_data->init_func = init_func;
		cb_data->user_data = user_data;

		g_idle_add ((GSourceFunc) idle_cb, cb_data);
	}

	bonobo_ui_main ();
}

/**
 * gnome_db_main_quit
 */
void
gnome_db_main_quit (void)
{
	gtk_main_quit ();
}
