/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Carlos Perell Marn <carlos@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libgnomedb/gnome-db-browser.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-intl.h>
#include <libgnomedb/gnome-db-util.h>
#include <glib-object.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkhbbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkhpaned.h>
#include <gtk/gtkhseparator.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmenushell.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkstatusbar.h>
#include <gtk/gtktable.h>
#include <gtk/gtktextbuffer.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkvpaned.h>

/**
 * gnome_db_new_browser_widget
 */
GtkWidget *
gnome_db_new_browser_widget (void)
{
	GtkWidget *browser;

	browser = gnome_db_browser_new ();
	gtk_widget_show (browser);

	return browser;
}

/**
 * gnome_db_new_button_widget
 */
GtkWidget *
gnome_db_new_button_widget (const gchar *label)
{
	GtkWidget *button;

	if (label != NULL)
		button = gtk_button_new_with_label (label);
	else
		button = gtk_button_new ();
	gtk_widget_show (button);

	return button;
}

/**
 * gnome_db_new_button_widget_from_stock
 */
GtkWidget *
gnome_db_new_button_widget_from_stock (const gchar *stock_id)
{
	GtkWidget *button;

	button = gtk_button_new_from_stock (stock_id);
	gtk_widget_show (button);

	return button;
}

/**
 * gnome_db_new_check_button_widget
 */
GtkWidget *
gnome_db_new_check_button_widget (const gchar *label)
{
	GtkWidget *check;

	if (label != NULL)
		check = gtk_check_button_new_with_label (label);
	else
		check = gtk_check_button_new ();
	gtk_widget_show (check);

	return check;
}

/**
 * gnome_db_new_combo_widget
 */
GtkWidget *
gnome_db_new_combo_widget (void)
{
	GtkWidget *combo;

	combo = gtk_combo_new ();
	gtk_widget_show (combo);

	return combo;
}

/**
 * gnome_db_new_entry_widget
 */
GtkWidget *
gnome_db_new_entry_widget (gint max_length, gboolean editable)
{
	GtkWidget *entry;

	entry = gtk_entry_new ();
	if (max_length > 0)
		gtk_entry_set_max_length (GTK_ENTRY (entry), max_length);
	gtk_editable_set_editable (GTK_EDITABLE (entry), editable);
	gtk_widget_show (entry);

	return entry;
}

/**
 * gnome_db_new_frame_widget
 */
GtkWidget *
gnome_db_new_frame_widget (const gchar *title)
{
	GtkWidget *frame;

	frame = gtk_frame_new (title);
	gtk_widget_show (frame);

	return frame;
}

/**
 * gnome_db_new_grid_widget
 */
GtkWidget *
gnome_db_new_grid_widget (GdaDataModel *model)
{
	GtkWidget *grid;

	if (GDA_IS_DATA_MODEL (model))
		grid = gnome_db_grid_new_with_model (model);
	else
		grid = gnome_db_grid_new ();
	gtk_widget_show (grid);

	return grid;
}

/**
 * gnome_db_new_hbox_widget
 */
GtkWidget *
gnome_db_new_hbox_widget (gboolean homogenous, gint spacing)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (homogenous, spacing);
	gtk_widget_show (hbox);

	return hbox;
}

/**
 * gnome_db_new_hbutton_box_widget
 */
GtkWidget *
gnome_db_new_hbutton_box_widget (void)
{
	GtkWidget *hbbox;

	hbbox = gtk_hbutton_box_new ();
	gtk_widget_show (hbbox);

	return hbbox;
}

/**
 * gnome_db_new_hpaned_widget
 */
GtkWidget *
gnome_db_new_hpaned_widget (void)
{
	GtkWidget *hpaned;

	hpaned = gtk_hpaned_new ();
	gtk_widget_show (hpaned);

	return hpaned;
}

/**
 * gnome_db_new_hseparator_widget
 */
GtkWidget *
gnome_db_new_hseparator_widget (void)
{
	GtkWidget *hsep;

	hsep = gtk_hseparator_new ();
	gtk_widget_show (hsep);

	return hsep;
}

/**
 * gnome_db_new_label_widget
 */
GtkWidget *
gnome_db_new_label_widget (const gchar *text)
{
	GtkWidget *label;

	label = gtk_label_new (text);
	gtk_label_set_selectable (GTK_LABEL (label), TRUE);
	gtk_widget_show (label);

	return label;
}

/**
 * gnome_db_new_notebook_widget
 */
GtkWidget *
gnome_db_new_notebook_widget (void)
{
	GtkWidget *notebook;

	notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
	gtk_notebook_popup_enable (GTK_NOTEBOOK (notebook));
	gtk_widget_show (notebook);

	return notebook;
}

/**
 * gnome_db_new_option_menu_widget
 */
GtkWidget *
gnome_db_new_option_menu_widget (void)
{
	GtkWidget *option_menu;

	option_menu = gtk_option_menu_new ();
	gtk_widget_show (option_menu);

	return option_menu;
}

/**
 * gnome_db_new_scrolled_window_widget
 */
GtkWidget *
gnome_db_new_scrolled_window_widget (void)
{
	GtkWidget *scroll;

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_IN);
	gtk_widget_show (scroll);

	return scroll;
}

/**
 * gnome_db_new_status_bar_widget
 */
GtkWidget *
gnome_db_new_status_bar_widget (void)
{
        GtkWidget* status;

        status = gtk_statusbar_new ();
        gtk_widget_show (status);

        return status;
}

/**
 * gnome_db_new_table_widget
 */
GtkWidget *
gnome_db_new_table_widget (gint rows, gint cols, gboolean homogenous)
{
	GtkWidget *table;

	table = gtk_table_new (rows, cols, homogenous);
	gtk_widget_show (table);

	return table;
}

/**
 * gnome_db_new_text_widget
 */
GtkWidget *
gnome_db_new_text_widget (const gchar *contents)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	if (contents != NULL) {
		buffer = gtk_text_buffer_new (NULL);
		gtk_text_buffer_set_text (buffer, contents, strlen (contents));

		text = gtk_text_view_new_with_buffer (buffer);
	}
	else
		text = gtk_text_view_new ();

	gtk_widget_show (text);

	return text;
}

/**
 * gnome_db_new_toolbar_widget
 */
GtkWidget *
gnome_db_new_toolbar_widget (GtkOrientation orientation, GtkToolbarStyle style)
{
	GtkWidget *toolbar;

	toolbar = gtk_toolbar_new ();
	gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar), orientation);
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), style);
	gtk_widget_show (toolbar);

	return toolbar;
}

/**
 * gnome_db_new_tree_view_widget
 */
GtkWidget *
gnome_db_new_tree_view_widget (GtkTreeModel *model)
{
	GtkWidget *tree_view;

	tree_view = model ?
		gtk_tree_view_new_with_model (model) :
		gtk_tree_view_new ();
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_widget_show (tree_view);

	return tree_view;
}

/**
 * gnome_db_new_vbox_widget
 */
GtkWidget *
gnome_db_new_vbox_widget (gboolean homogenous, gint spacing)
{
	GtkWidget *vbox;

	vbox = gtk_vbox_new (homogenous, spacing);
	gtk_widget_show (vbox);

	return vbox;
}

/**
 * gnome_db_new_vpaned_widget
 */
GtkWidget *
gnome_db_new_vpaned_widget (void)
{
	GtkWidget *vpaned;

	vpaned = gtk_vpaned_new ();
	gtk_widget_show (vpaned);

	return vpaned;
}

/**
 * gnome_db_set_widget_bg_color
 */
void
gnome_db_set_widget_bg_color (GtkWidget *widget, const gchar *color)
{
	GdkColor gdk_color;

	g_return_if_fail (GTK_IS_WIDGET (widget));

	gdk_color_parse (color, &gdk_color);
	gtk_widget_modify_bg (widget, GTK_STATE_NORMAL, &gdk_color);
}

/**
 * gnome_db_set_widget_fg_color
 */
void
gnome_db_set_widget_fg_color (GtkWidget *widget, const gchar *color)
{
	GdkColor gdk_color;

	g_return_if_fail (GTK_IS_WIDGET (widget));

	gdk_color_parse (color, &gdk_color);
	gtk_widget_modify_fg (widget, GTK_STATE_NORMAL, &gdk_color);
}

/**
 * gnome_db_option_menu_add_item
 */
void
gnome_db_option_menu_add_item (GtkOptionMenu *option_menu, const gchar *label)
{
	GtkWidget *menu;
	GtkWidget *menu_item;

	g_return_if_fail (GTK_IS_OPTION_MENU (option_menu));
	g_return_if_fail (label != NULL);

	menu = gtk_option_menu_get_menu (option_menu);
	if (!GTK_IS_MENU (menu)) {
		menu = gtk_menu_new ();
		gtk_option_menu_set_menu (option_menu, menu);
		gtk_widget_show (menu);
	}

	menu_item = gtk_menu_item_new_with_label (label);
	gtk_widget_show (menu_item);
	g_object_set_data_full (G_OBJECT (menu_item), "GNOME:Database:MenuItemLabel",
				g_strdup (label), (GDestroyNotify) g_free);

	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);
}

/**
 * gnome_db_option_menu_get_selection
 */
const gchar *
gnome_db_option_menu_get_selection (GtkOptionMenu *option_menu)
{
	GtkWidget *menu_item;

	g_return_val_if_fail (GTK_IS_OPTION_MENU (option_menu), NULL);

	menu_item = option_menu->menu_item;
	if (GTK_IS_MENU_ITEM (menu_item)) {
		return (const gchar *) g_object_get_data (G_OBJECT (menu_item),
							  "GNOME:Database:MenuItemLabel");
	}

	return NULL;
}

/**
 * gnome_db_text_clear
 * @text: a #GtkTextView widget
 *
 * Clear the contents of the given text view widget.
 */
void
gnome_db_text_clear (GtkTextView *text)
{
	GtkTextIter start;
	GtkTextIter end;
	gint char_count;
	GtkTextBuffer *buffer;

	g_return_if_fail (GTK_IS_TEXT_VIEW (text));

	buffer = gtk_text_view_get_buffer (text);
	char_count = gnome_db_text_get_char_count (text);

	gtk_text_buffer_get_iter_at_offset (buffer, &start, 0);
	gtk_text_buffer_get_iter_at_offset (buffer, &end, char_count);

	gtk_text_buffer_delete (buffer, &start, &end);
}

/**
 * gnome_db_text_copy_clipboard
 */
void
gnome_db_text_copy_clipboard (GtkTextView *text)
{
	g_return_if_fail (GTK_IS_TEXT_VIEW (text));
	gtk_text_buffer_copy_clipboard (gtk_text_view_get_buffer (text),
					gtk_clipboard_get (GDK_SELECTION_CLIPBOARD));
}

/**
 * gnome_db_text_cut_clipboard
 */
void
gnome_db_text_cut_clipboard (GtkTextView *text)
{
	g_return_if_fail (GTK_IS_TEXT_VIEW (text));
	gtk_text_buffer_cut_clipboard (gtk_text_view_get_buffer (text),
				       gtk_clipboard_get (GDK_SELECTION_CLIPBOARD),
				       gtk_text_view_get_editable (text));
}

/**
 * gnome_db_text_get_char_count
 */
gint
gnome_db_text_get_char_count (GtkTextView *text)
{
	g_return_val_if_fail (GTK_IS_TEXT_VIEW (text), -1);
	return gtk_text_buffer_get_char_count (gtk_text_view_get_buffer (text));
}

/**
 * gnome_db_text_get_line_count
 */
gint
gnome_db_text_get_line_count (GtkTextView *text)
{
	g_return_val_if_fail (GTK_IS_TEXT_VIEW (text), -1);
	return gtk_text_buffer_get_line_count (gtk_text_view_get_buffer (text));
}

/**
 * gnome_db_text_get_text
 */
gchar *
gnome_db_text_get_text (GtkTextView *text)
{
	GtkTextBuffer *buffer;
	GtkTextIter start;
	GtkTextIter end;
	gint char_count;

	g_return_val_if_fail (GTK_IS_TEXT_VIEW (text), NULL);

	buffer = gtk_text_view_get_buffer (text);
	char_count = gnome_db_text_get_char_count (text);

	gtk_text_buffer_get_iter_at_offset (buffer, &start, 0);
	gtk_text_buffer_get_iter_at_offset (buffer, &end, char_count);

	return gtk_text_buffer_get_text (gtk_text_view_get_buffer (text),
					 &start, &end, FALSE);
}

/**
 * gnome_db_text_insert_at_cursor
 */
void
gnome_db_text_insert_at_cursor (GtkTextView *text, const gchar *contents, gint len)
{
	g_return_if_fail (GTK_IS_TEXT_VIEW (text));
	g_return_if_fail (contents != NULL);

	gtk_text_buffer_insert_at_cursor (gtk_text_view_get_buffer (text), contents, len);
}

/**
 * gnome_db_text_paste_clipboard
 */
void
gnome_db_text_paste_clipboard (GtkTextView *text)
{
	g_return_if_fail (GTK_IS_TEXT_VIEW (text));
	gtk_text_buffer_paste_clipboard (gtk_text_view_get_buffer (text),
					 gtk_clipboard_get (GDK_SELECTION_CLIPBOARD),
					 NULL,
					 gtk_text_view_get_editable (text));
}

/**
 * gnome_db_text_set_text
 */
void
gnome_db_text_set_text (GtkTextView *text, const gchar *contents, gint len)
{
	g_return_if_fail (GTK_IS_TEXT_VIEW (text));
	gtk_text_buffer_set_text (gtk_text_view_get_buffer (text), contents, len);
}

/**
 * gnome_db_show_error
 */
void
gnome_db_show_error (const gchar *format, ...)
{
	va_list args;
        gchar sz[2048];
	GtkWidget *dialog;

	/* build the message string */
        va_start (args, format);
        vsprintf (sz, format, args);
        va_end (args);

	/* create the error message dialog */
	dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE, sz);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

#ifdef ENABLE_NLS

#include <libintl.h>

/*
 * This function has been stolen from glib/glib/gutils.c
 */
G_CONST_RETURN gchar *
_gnome_db_gettext (const gchar *str)
{
	static gboolean _gnome_db_gettext_initialized = FALSE;

	if (!_gnome_db_gettext_initialized) {
		bindtextdomain(GETTEXT_PACKAGE, LIBGNOMEDB_LOCALEDIR);
		_gnome_db_gettext_initialized = TRUE;
	}

	return dgettext (GETTEXT_PACKAGE, str);
}

#endif /* ENABLE_NLS */
