/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <config.h>
#include "add-printer.h"

#include <cups/cups.h>
#include <cups/language.h>
#include <cups/http.h>
#include <cups/ipp.h>

#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktreeview.h>
#include <glade/glade.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-druid.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-icon-theme.h>

#include <libgnomecups/gnome-cups-ui-init.h>
#include <libgnomecups/gnome-cups-request.h>
#include <libgnomecups/gnome-cups-util.h>

#include "druid-helper.h"
#include "snmpinter.h"
#include "util.h"

#define ENTRY_TEXT(xml,name) (gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, name))))

/* Keep in sync with the network_type_option menu */
typedef enum {
	NETWORK_TYPE_LPD,
	NETWORK_TYPE_SMB,
	NETWORK_TYPE_IPP,
	NETWORK_TYPE_HP
} NetworkType;

enum {
	LOCAL_DETECTED_LABEL,
	LOCAL_DETECTED_PRINTER,
	LAST_LOCAL_DETECTED
};

typedef struct {
	char *label;
	char *uri;
	char *location;
	char *vendor_and_model;
} LocalPrinter;

/*
 * PPDs are taken from cups and filtered into the following data structures:
 *
 * vendors (GHashTable *)
 *    "vendor1" -> ...
 *    "vendor2" -> ...
 *    "vendor3" -> models (GHashTable *)
 *                   "model 1" -> ...
 *                   "model 2" -> ...
 *                   "model 3" -> ppds (GList *)
 *                                  PPD *
 *                                  PPD *
 *                                  PPD *
 *
 */

/* Identifies a PPD retrieved from the cups server */
typedef struct
{
	char *name;
	char *vendor_and_model;
	char *vendor;
	char *model;
	char *driver;
	char *language;
} PPD;


static void
gnome_cups_error_dialog (GtkWidget *w,
			 const char *prefix,
			 GError *error)
{
	GtkWidget *dialog;
	char *msg;

	msg = error ? error->message : _("Unknown Error");

	dialog = gtk_message_dialog_new (GTK_WINDOW (w),
					 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_OK,
					 "%s: %s", prefix, error->message);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
local_printer_free (LocalPrinter *printer)
{
	g_free (printer->label);
	g_free (printer->location);
	g_free (printer->uri);
	g_free (printer->vendor_and_model);
	g_free (printer);
}

static void
local_printer_list_free (GList *printers)
{
	GList *l;
	for (l = printers; l != NULL; l = l->next) {
		local_printer_free (l->data);
	}
	g_list_free (printers);
}

static char *
combine_vendor_and_model (const char *vendor,
			  const char *model)
{
	char *prefix;
	char *ret;

	if (strstr (model, vendor) == model) {
		return g_strdup (model);
	} else {
		return g_strdup_printf ("%s %s", vendor, model);
	}
	g_free (prefix);

	return ret;
}

static char *
parse_network_detect (char *line)
{
	char *vendor_and_model;
	char *vendor = NULL;
	char *model = NULL;
	char **fields;
	char **p;
	char *sep;

	sep = strchr (line, '\n');
	if (sep) {
		*sep = '\0';
	}

	line = g_strstrip (line);
	fields = g_strsplit (line, ";", -1);

	for (p = fields; *p != NULL; p++) {
		char **pieces;
		char *name;
		char *value;

		pieces = g_strsplit (*p, "=", -1);
		name = pieces[0];
		value = pieces[1];
		if (!name || !value) {
			g_strfreev (pieces);
			continue;
		}

		if (!strcmp (name, "vendor")) {
			vendor = g_strdup (value);
		} else if (!strcmp (name, "model")) {
			model = g_strdup (value);
		}

		g_strfreev (pieces);
	}
	g_strfreev (fields);

	if (!vendor || !model) {
		g_free (vendor);
		g_free (model);
		return NULL;
	}

	vendor_and_model = combine_vendor_and_model (vendor, model);

	g_free (vendor);
	g_free (model);

	return vendor_and_model;
}
static GtkWidget *
option_menu_get_selected_item (GladeXML *xml, const char *name)
{
	int index;
	GtkWidget *option_menu;
	GtkWidget *menu;
	GtkWidget *item;

	option_menu = glade_xml_get_widget (xml, name);
	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (option_menu));
	if (!menu) {
		return NULL;
	}
	index = gtk_option_menu_get_history (GTK_OPTION_MENU (option_menu));
	item = g_list_nth_data (GTK_MENU_SHELL (menu)->children, index);

	return item;
}

/* Vendor/Model Page Helpers */

static int
str_case_compare (const char *v1,
		  const char *v2)
{
	char *a;
	char *b;
	int ret;

	a = g_utf8_casefold (v1, -1);
	b = g_utf8_casefold (v2, -1);

	ret = g_utf8_collate (a, b);

	g_free (a);
	g_free (b);

	return ret;
}

static GList *
model_list_for_vendor (GHashTable *vendors,
		       const char *vendor)
{
	GHashTable *models;

	models = g_hash_table_lookup (vendors, vendor);

	return g_list_sort (gnome_cups_hash_table_keys (models),
			    (GCompareFunc)str_case_compare);
}

static GList *
vendor_list (GHashTable *vendors)
{
	return g_list_sort (gnome_cups_hash_table_keys (vendors),
			    (GCompareFunc)str_case_compare);
}


static char *
remove_vendor (const char *vendor,
	       const char *model)
{
        char *prefix;
        char *ret;

	prefix = g_strconcat (vendor, " ", NULL);
	if (strstr (model, prefix) == model) {
		ret = g_strstrip (g_strdup (model + strlen (prefix)));
        } else {
		ret = g_strdup (model);
        }
        g_free (prefix);

        return ret;
}

/* Return number of characters that match, or -1 if they are identical */
static int
num_match (const char *a, const char *b)
{
	char *a1;
	char *b1;
	int i;

	a1 = g_utf8_casefold (a, -1);
	b1 = g_utf8_casefold (b, -1);

	for (i = 0; a1[i] != '\0' && b1[i] != '\0'; i++) {
		if (a1[i] != b1[i]) {
			break;
		}
	}
	if (a1[i] == '\0' && b1[i] == '\0') {
		i = -1;
	}

	g_free (a1);
	g_free (b1);

	return i;
}

static PPD *
get_detected_ppd (GladeXML *xml, const char *vendor_and_model)
{
	GHashTable *vendor_hash;
	GHashTable *model_hash;
	GList *vendors;
	GList *models;
	GList *drivers;
	GList *l;
	char *vendor;
	char *model;
	char *ppd_model;
	int highest_match;

	if (!vendor_and_model) {
		return NULL;
	}

	vendor_hash = g_object_get_data (G_OBJECT (xml), "vendors");

	vendors = vendor_list (vendor_hash);

	vendor = NULL;
	model = NULL;

	/* Split the vendor and the model out of the vendor_and_model
	 * string */
	for (l = vendors; l != NULL; l = l->next) {
		model = remove_vendor (l->data, vendor_and_model);

		if (strcmp (model, vendor_and_model)) {
			/* Succeeded in removing the vendor */
			vendor = g_strdup (l->data);
			break;
		} else {
			g_free (model);
			model = NULL;
		}
	}

	g_list_free (vendors);

	if (!vendor) {
		return NULL;
	}

	/* Now we have a valid make and model.  Find the closest
	 * ppd */
	models = model_list_for_vendor (vendor_hash, vendor);

	ppd_model = NULL;
	highest_match = 0;
	for (l = models; l != NULL; l = l->next) {
		int match;

		g_assert (l->data);

		match = num_match (model, l->data);

		/* exact match */
		if (match == -1) {
			ppd_model = l->data;
			break;
		}

		if (match > highest_match) {
			ppd_model = l->data;
			highest_match = match;
		}
	}

	if (!ppd_model) {
		return NULL;
	}

	g_list_free (models);

	model_hash = g_hash_table_lookup (vendor_hash, vendor);
	g_return_val_if_fail (model_hash != NULL, NULL);
	drivers = g_hash_table_lookup (model_hash, ppd_model);

	g_free (vendor);
	g_free (model);

	if (!drivers) {
		return NULL;
	}

	return drivers->data;
}

static void
set_selected_uri (GladeXML *xml, const char *uri)
{
	g_object_set_data_full (G_OBJECT (xml), "selected_uri",
				g_strdup (uri), g_free);
}

static char *
get_selected_uri (GladeXML *xml)
{
	char *str;

	str = g_object_get_data (G_OBJECT (xml), "selected_uri");
	g_return_val_if_fail (str != NULL, NULL);

	return g_strdup (str);
}

static void
set_selected_location (GladeXML *xml, const char *location)
{
	g_object_set_data_full (G_OBJECT (xml), "selected_location",
				g_strdup (location), g_free);
}

static char *
get_selected_location (GladeXML *xml)
{
	char *str;

	str = g_object_get_data (G_OBJECT (xml), "selected_location");
	g_return_val_if_fail (str != NULL, NULL);

	return g_strdup (str);
}

static void
select_item (GtkWidget *option_menu, const char *name)
{
	GtkWidget *menu;
	GList *l;
	int i;

	if (!name) {
		return;
	}

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (option_menu));

	i = 0;
	for (l = GTK_MENU_SHELL (menu)->children; l != NULL; l = l->next) {
		int compare;
		GtkWidget *item = l->data;
		char *item_name = g_object_get_data (G_OBJECT (item), "name");
		compare = str_case_compare (item_name, name);
		if (compare >= 0) {
			gtk_option_menu_set_history
				(GTK_OPTION_MENU (option_menu), i);
			break;
		}
		i++;
	}
}

static void
select_vendor (GladeXML *xml, const char *vendor_name)
{
	select_item (glade_xml_get_widget (xml, "vendor_option_menu"),
		     vendor_name);
}

static void
scroll_to_iter (GtkTreeView *view, GtkTreeModel *model, GtkTreeIter *iter)
{
	GtkTreePath *path;

	path = gtk_tree_model_get_path (model, iter);

	gtk_tree_view_scroll_to_cell (view,
				      path,
				      NULL,
				      FALSE,
				      0.0, 0.0);
	gtk_tree_path_free (path);
}

static void
select_model (GladeXML *xml, const char *model_name)
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean success;

	view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "model_treeview"));
	model = gtk_tree_view_get_model (view);
	selection = gtk_tree_view_get_selection (view);

	success = gtk_tree_model_get_iter_first (model, &iter);

	gtk_tree_selection_select_iter (selection, &iter);
	scroll_to_iter (view, model, &iter);

	while (success) {
		int compare;
		char *item_name;

		gtk_tree_model_get (model, &iter, 0, &item_name, -1);
		compare = str_case_compare (item_name, model_name);
		if (compare >= 0) {
			gtk_tree_selection_select_iter (selection, &iter);
			scroll_to_iter (view, model, &iter);
			break;
		}
		gtk_tree_model_iter_next (model, &iter);
	}


}

static void
select_detected_ppd (GladeXML *xml, const char *vendor_and_model)
{
	PPD *ppd;

	ppd = get_detected_ppd (xml, vendor_and_model);
	if (ppd) {
		select_vendor (xml, ppd->vendor);
		select_model (xml, ppd->model);
		g_object_set_data (G_OBJECT (xml), "detected_ppd", ppd);
	} else {
		g_object_set_data (G_OBJECT (xml), "detected_ppd", NULL);
	}
}

/* Local or Remote  page */

static void
local_or_remote_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = FALSE;
	*next = TRUE;
}

static GtkWidget *
local_or_remote_next (GladeXML *xml)
{
	GtkWidget *local_radio;

	local_radio = glade_xml_get_widget (xml, "local_radio");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (local_radio))) {
		return glade_xml_get_widget (xml, "local_location_page");
	} else {
		return glade_xml_get_widget (xml, "network_location_page");
	}
}

/* Local Location Page */

static char *
get_local_manual_uri (GladeXML *xml)
{
	GtkWidget *item;

	item = option_menu_get_selected_item (xml, "local_port_option_menu");
	if (item) {
		LocalPrinter *desc;
		desc = g_object_get_data (G_OBJECT (item), "printer-description");
		return g_strdup (desc->uri);
	}
	return NULL;
}

static char *
get_local_manual_location (GladeXML *xml)
{
	GtkWidget *item;

	item = option_menu_get_selected_item (xml, "local_port_option_menu");
	if (item) {
		LocalPrinter *desc;
		desc = g_object_get_data (G_OBJECT (item), "printer-description");
		return g_strdup (desc->location);
	}
	return NULL;
}


static void
setup_port_option_menu (GladeXML *xml)
{
	GtkWidget *option_menu;
	GtkWidget *menu;
	GList *devices;

	option_menu = glade_xml_get_widget (xml, "local_port_option_menu");
	menu = gtk_menu_new ();

	devices = g_object_get_data (G_OBJECT (xml), "local-devices");

	for (; devices != NULL; devices = devices->next) {
		LocalPrinter *desc = devices->data;
		GtkWidget *item;
		char *label;

		if (desc->vendor_and_model) {
			label = g_strdup_printf (_("%s (%s)"),
						 desc->label,
						 desc->vendor_and_model);
		} else {
			label = g_strdup (desc->label);
		}

		item = gtk_menu_item_new_with_label (label);
		g_object_set_data (G_OBJECT (item),
				   "printer-description", desc);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
}

static void
update_local_location_sensitivities (GtkWidget *widget,
				     GladeXML *xml)
{
	GtkWidget *view;
	GtkWidget *option_menu;

	view = glade_xml_get_widget (xml, "local_detected_view");
	option_menu = glade_xml_get_widget (xml, "local_port_option_menu");

	if (toggle_button_is_active (xml, "local_use_detected_radio")) {
		gtk_widget_set_sensitive (view, TRUE);
		gtk_widget_set_sensitive (option_menu, FALSE);
	} else {
		gtk_widget_set_sensitive (view, FALSE);
		gtk_widget_set_sensitive (option_menu, TRUE);
	}
}

static GList *
get_local_devices (void)
{
	GList *ret;
	ipp_t *request;
	ipp_t *response;

	ret = NULL;

	request = gnome_cups_request_new_for_printer (CUPS_GET_DEVICES, NULL);
	response = gnome_cups_request_execute (request, "/", NULL);

	if (response) {
		ipp_attribute_t *attr;
		LocalPrinter *desc;
		char *device_class = NULL;

		desc = g_new0 (LocalPrinter, 1);
		for (attr = response->attrs; attr != NULL; attr = attr->next) {
			if (!attr->name) {
				if (device_class
				    && strcmp (device_class, "network")
				    && desc->label
				    && desc->uri) {
					ret = g_list_prepend (ret, desc);
				} else {
					desc = g_new0 (LocalPrinter, 1);
				}
				desc = g_new0 (LocalPrinter, 1);

				g_free (device_class);
				device_class = NULL;

				continue;
			}

			if (!strcmp (attr->name, "device-class")) {
				g_free (device_class);
				device_class = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-info")) {
				g_free (desc->label);
				desc->label = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-uri")) {
				g_free (desc->uri);
				desc->uri = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-make-and-model") && strcmp (attr->values[0].string.text, "Unknown")) {
				g_free (desc->vendor_and_model);
				desc->vendor_and_model = g_strdup (attr->values[0].string.text);
			}
		}

		if (device_class
		    && strcmp (device_class, "network")
		    && desc->label
		    && desc->uri) {
			ret = g_list_prepend (ret, desc);
		} else {
			local_printer_free (desc);
		}

		ret = g_list_reverse (ret);

		g_free (device_class);

		ippDelete (response);
	}

	return ret;
}

static void
local_location_setup (GladeXML *xml)
{
	GtkTreeView *tree_view;
	GtkListStore *list_store;
	GtkTreeSelection *selection;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkWidget *widget;
	int num_detected;
	GList *devices;
	GList *l;


	devices = get_local_devices ();
	g_object_set_data_full (G_OBJECT (xml), "local-devices",
				devices,
				(GDestroyNotify)local_printer_list_free);

	tree_view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Printer"),
							   renderer,
							   "markup",
							   LOCAL_DETECTED_LABEL,
							   NULL);
	gtk_tree_view_append_column (tree_view, column);

	list_store = gtk_list_store_new (LAST_LOCAL_DETECTED,
					 G_TYPE_STRING, G_TYPE_POINTER);
	gtk_tree_view_set_model (tree_view, GTK_TREE_MODEL (list_store));

	selection = gtk_tree_view_get_selection (tree_view);
	num_detected = 0;
	for (l = devices; l != NULL; l = l->next) {
		LocalPrinter *desc = l->data;
		if (desc->vendor_and_model
		    /* weed out two that are always returned
		     * regardless of what's there */
		    && strcmp (desc->vendor_and_model, "EPSON")
		    && strcmp (desc->vendor_and_model, "CANON")) {
			GtkTreeIter iter;

			gtk_list_store_append (list_store, &iter);
			gtk_list_store_set (list_store, &iter,
					    0, desc->vendor_and_model,
					    1, desc,
					    -1);
			if (!num_detected) {
				gtk_tree_selection_select_iter (selection, &iter);
			}

			num_detected++;
		}


	}

	if (!num_detected) {
		GtkTreeIter iter;
		gtk_list_store_append (list_store, &iter);
		gtk_list_store_set (list_store, &iter,
				    0, _("<i>No printers detected</i>"),
				    1, NULL,
				    -1);

		gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
	}

	setup_port_option_menu (xml);

	widget = glade_xml_get_widget (xml, "local_use_detected_radio");
	g_signal_connect (widget, "toggled",
			  G_CALLBACK (update_local_location_sensitivities),
			  xml);
	widget = glade_xml_get_widget (xml, "local_specify_port_radio");
	g_signal_connect (widget, "toggled",
			  G_CALLBACK (update_local_location_sensitivities),
			  xml);

	druid_watch_sensitivity_widget (xml, "local_use_detected_radio");
	druid_watch_sensitivity_widget (xml, "local_specify_port_radio");
}

static void
local_location_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = TRUE;

	*next = toggle_button_is_active (xml, "local_specify_port_radio") || tree_view_has_selection (xml, "local_detected_view");
}

static GtkWidget *
local_location_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "local_or_remote_page");
}

static GtkWidget *
local_location_next (GladeXML *xml)
{
	GtkTreeView *tree_view;
	GtkListStore *list_store;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean use_detected;

	tree_view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));
	selection = gtk_tree_view_get_selection (tree_view);
	list_store = GTK_LIST_STORE (gtk_tree_view_get_model (tree_view));

	use_detected = !toggle_button_is_active (xml, "local_specify_port_radio");

	select_detected_ppd (xml, NULL);

	if (use_detected) {
		if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
			char *location;
			LocalPrinter *desc;
			gtk_tree_model_get (GTK_TREE_MODEL (list_store),
					    &iter,
					    1, &desc, -1);
			location = g_strdup_printf (_("Local printer on %s"),
						       desc->location);
			select_detected_ppd (xml, desc->vendor_and_model);
			set_selected_uri (xml, desc->uri);
			set_selected_location (xml, location);
			g_free (location);
		} else {
			g_warning ("no selected printer");
			return NULL;
		}
	} else {
		char *uri;
		char *location;

		uri = get_local_manual_uri (xml);
		if (!uri) {
			g_warning ("no selected port");
			return NULL;
		}

		location = get_local_manual_location (xml);

		set_selected_uri (xml, uri);
		set_selected_location (xml, uri);
		g_free (uri);
		g_free (location);
	}

	return glade_xml_get_widget (xml, "details_page");
}


/* Network Location Page */

static NetworkType
get_manual_network_type (GladeXML *xml)
{
	int index;

	index = gtk_option_menu_get_history (GTK_OPTION_MENU (glade_xml_get_widget (xml, "network_type_option")));

	return (NetworkType)index;
}

static char *
get_manual_lpd_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host;
	char *queue;

	host = g_strdup (ENTRY_TEXT (xml, "lpd_host_entry"));
	queue = g_strdup (ENTRY_TEXT (xml, "lpd_queue_entry"));

	host = g_strstrip (host);
	queue = g_strstrip (queue);

	if (host[0]) {
		ret = g_strdup_printf ("lpd://%s/%s", host, queue);
	}

	g_free (host);
	g_free (queue);

	return ret;
}

static char *
get_manual_smb_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host;
	char *printer;
	char *username;
	char *password;

	host = g_strdup (ENTRY_TEXT (xml, "smb_host_entry"));
	printer = g_strdup (ENTRY_TEXT (xml, "smb_printer_entry"));
	username = g_strdup (ENTRY_TEXT (xml, "smb_username_entry"));
	password = g_strdup (ENTRY_TEXT (xml, "smb_password_entry"));

	host = g_strstrip (host);
	printer = g_strstrip (printer);
	username = g_strstrip (username);

	if (host[0] && printer[0]) {
		if (username[0]) {
			ret = g_strdup_printf ("smb://%s:%s@%s/%s",
					       username,
					       password,
					       host,
					       printer);
		} else {
			ret = g_strdup_printf ("smb://%s/%s",
					       host,
					       printer);
		}
	}

	g_free (host);
	g_free (printer);
	g_free (username);
	g_free (password);

	return ret;
}

static char *
get_manual_ipp_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *uri;

	uri = g_strdup (ENTRY_TEXT (xml, "ipp_uri_entry"));

	uri = g_strstrip (uri);

	if (uri[0]) {
		ret = g_strdup (uri);
	}

	g_free (uri);

	return ret;
}

static char *
get_manual_hp_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host;
	char *port;

	host = g_strdup (ENTRY_TEXT (xml, "hp_host_entry"));
	port = g_strdup (ENTRY_TEXT (xml, "hp_port_entry"));

	host = g_strstrip (host);
	port = g_strstrip (port);

	if (!port[0]) {
		g_free (port);
		port = g_strdup ("9100");
	}

	if (host[0]) {
		ret = g_strdup_printf ("socket://%s:%s", host, port);
	}

	g_free (host);
	g_free (port);

	return ret;
}

/* Read the manually selected preferences from the dialog */
static char *
get_manual_network_uri (GladeXML *xml)
{
	char *ret;

	ret = NULL;

	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_LPD :
		ret = get_manual_lpd_uri (xml);
		break;
	case NETWORK_TYPE_SMB :
		ret = get_manual_smb_uri (xml);
		break;
	case NETWORK_TYPE_IPP :
		ret = get_manual_ipp_uri (xml);
		break;
	case NETWORK_TYPE_HP :
		ret = get_manual_hp_uri (xml);
		break;
	default :
		g_warning ("unsupported type\n");
	}

	return ret;
}

static char *
get_manual_network_location (GladeXML *xml)
{
	char *ret;

	ret = NULL;

	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_LPD :
	{
		char *host;
		char *queue;

		host = g_strdup (ENTRY_TEXT (xml, "lpd_host_entry"));
		queue = g_strdup (ENTRY_TEXT (xml, "lpd_queue_entry"));

		host = g_strstrip (host);
		queue = g_strstrip (queue);

		if (host[0]) {
			if (queue[0]) {
				/* queue, host */
				ret = g_strdup_printf (_("UNIX printer %s on %s"), queue, host);
			} else {
				/* host */
				ret = g_strdup_printf (_("UNIX printer on %s"), host);
			}
		}
		break;
	}
	case NETWORK_TYPE_SMB :
	{
		char *host;
		char *printer;

		host = g_strdup (ENTRY_TEXT (xml, "smb_host_entry"));
		printer = g_strdup (ENTRY_TEXT (xml, "smb_printer_entry"));

		host = g_strstrip (host);
		printer = g_strstrip (printer);

		if (host[0] && printer[0]) {
			/* printer, host */
			ret = g_strdup_printf (_("Windows Printer %s on %s"), printer, host);
		}
		break;
	}
	case NETWORK_TYPE_IPP :
	{
		char *uri;

		uri = g_strdup (ENTRY_TEXT (xml, "ipp_uri_entry"));

		uri = g_strstrip (uri);

		if (uri[0]) {
			/* uri */
			ret = g_strdup_printf (_("IPP Printer at %s"), uri);
		}
		break;
	}
	default :
		g_warning ("unsupported type\n");
	}

	return ret;
}

static char *
lpd_get_vendor_and_model (const char *host)
{
	char *ret;
	char *detected;
	int retval;

	detected = g_strdup (get_snmp_printers ((char*)host, &retval));

	if (detected) {
		ret = parse_network_detect (detected);
	} else {
		ret = NULL;
	}
	g_free (detected);

	return ret;
}

static char *
get_manual_network_vendor_and_model (GladeXML *xml)
{
	char *ret;
	char *host;
	host = NULL;

	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_LPD :
		host = g_strdup (ENTRY_TEXT (xml, "lpd_host_entry"));
		host = g_strstrip (host);
		ret = lpd_get_vendor_and_model (host);
		g_free (host);
		return ret;
	case NETWORK_TYPE_SMB :
	case NETWORK_TYPE_IPP :
		return NULL;
	default :
		g_warning ("unsupported type\n");
		return NULL;
	}
}

static void
network_type_changed_cb (GtkOptionMenu *option_menu,
			 GladeXML *xml)
{
	int index;
	GtkWidget *notebook;

	index = gtk_option_menu_get_history (option_menu);
	notebook = glade_xml_get_widget (xml, "network_info_notebook");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook), index);
}

static void
network_location_setup (GladeXML *xml)
{
	GtkWidget *option_menu;

	druid_watch_sensitivity_widget (xml, "network_type_option");
	druid_watch_sensitivity_widget (xml, "lpd_host_entry");
	druid_watch_sensitivity_widget (xml, "smb_username_entry");
	druid_watch_sensitivity_widget (xml, "smb_password_entry");
	druid_watch_sensitivity_widget (xml, "smb_printer_entry");
	druid_watch_sensitivity_widget (xml, "smb_host_entry");
	druid_watch_sensitivity_widget (xml, "ipp_uri_entry");
	druid_watch_sensitivity_widget (xml, "hp_host_entry");
	druid_watch_sensitivity_widget (xml, "hp_port_entry");

	option_menu = glade_xml_get_widget (xml, "network_type_option");
	g_signal_connect (option_menu, "changed",
			  G_CALLBACK (network_type_changed_cb), xml);
}

static void
network_location_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	char *uri;

	*back = TRUE;

	uri = get_manual_network_uri (xml);

	*next = (uri != NULL);

	g_free (uri);
}

static GtkWidget *
network_location_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "local_or_remote_page");
}

static GtkWidget *
network_location_next (GladeXML *xml)
{
	char *uri;

	uri = get_manual_network_uri (xml);
	if (!uri) {
		return NULL;
	} else {
		char *vendor_and_model;
		char *location;

		location = get_manual_network_location (xml);
		set_selected_uri (xml, uri);
		set_selected_location (xml, location);
		g_free (uri);
		g_free (location);

		vendor_and_model = get_manual_network_vendor_and_model (xml);

		if (vendor_and_model) {
			select_detected_ppd (xml, vendor_and_model);
			g_free (vendor_and_model);
		} else {
			select_detected_ppd (xml, NULL);
		}
		return glade_xml_get_widget (xml, "details_page");
	}
}

static ipp_t *
ppd_request (void)
{
	ipp_t *response;
	ipp_t *request;
	char *attributes[] = { "ppd-name",
			       "ppd-make",
			       "ppd-make-and-model",
			       "ppd-natural-language" };

	request = gnome_cups_request_new_for_printer (CUPS_GET_PPDS, NULL);
	gnome_cups_request_add_requested_attributes (request,
						     IPP_TAG_PRINTER,
						     G_N_ELEMENTS (attributes),
						     attributes);

	response = gnome_cups_request_execute (request, "/", NULL);

	return response;
}

static void
destroy_ppd_id (PPD *ppd)
{
	g_free (ppd->name);
	g_free (ppd->vendor);
	g_free (ppd->vendor_and_model);
	g_free (ppd->model);
	g_free (ppd->driver);
	g_free (ppd->language);
	g_free (ppd);
}

static void
destroy_ppd_id_list (GList *list)
{
	GList *l;

	for (l = list; l != NULL; l = l->next) {
		destroy_ppd_id (l->data);
	}
	g_list_free (list);
}

static void
parse_ppd_vendor_and_model (const char *vendor,
			    const char *vendor_and_model,
			    char **model,
			    char **driver)
{
        /* FIXME: There isn't really a garauntee that breaking on the
	 * comma is ok */
	char *without_vendor;
	char *comma;

	without_vendor = remove_vendor (vendor, vendor_and_model);

	comma = strstr (without_vendor, ", ");
	if (comma) {
		*model = g_strndup (without_vendor, comma - without_vendor);
		*driver = g_strdup (comma + 2);
	} else {
		*model = g_strdup (without_vendor);
		*driver = g_strdup (_("CUPS"));
	}

	*model = g_strstrip (*model);
	*driver = g_strstrip (*driver);

	g_free (without_vendor);
}

static GList *
get_ppds (void)
{
	ipp_attribute_t *attr;
	ipp_t *response;
	PPD *ppd = NULL;
	GList *ppd_ids;

	response = ppd_request ();

	ppd_ids = NULL;
	ppd = g_new0 (PPD, 1);

	if (response) {
		for (attr = response->attrs; attr != NULL; attr = attr->next) {
			if (!attr->name) {
				if (ppd->name
				    && ppd->vendor
				    && ppd->vendor_and_model
				    && ppd->language) {
					parse_ppd_vendor_and_model
						(ppd->vendor,
						 ppd->vendor_and_model,
						 &ppd->model,
						 &ppd->driver);

					ppd_ids = g_list_prepend (ppd_ids,
								  ppd);
				} else {
					destroy_ppd_id (ppd);
				}

				ppd = g_new0 (PPD, 1);
				continue;
			}

			if (!strcmp (attr->name, "ppd-name")) {
				/* FIXME */
				g_assert (ppd->name == NULL);
				ppd->name = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "ppd-make")) {
				/* FIXME */
				g_assert (ppd->vendor == NULL);
				ppd->vendor = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "ppd-make-and-model")) {
				/* FIXME */
				g_assert (ppd->vendor_and_model == NULL);
				ppd->vendor_and_model = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "ppd-natural-language")) {
				g_assert (ppd->language == NULL);
				ppd->language = g_strdup (attr->values[0].string.text);
			}
		}
		ippDelete (response);
	} else {
		/* FIXME */
	}

	return ppd_ids;
}


static int
compare_models_by_driver (gconstpointer a, gconstpointer b)
{
	const PPD *a1 = a;
	const PPD *b1 = b;
	return str_case_compare (a1->driver, b1->driver);
}

static guint
str_case_hash (gconstpointer key)
{
	char *str;
	guint ret;

	str = g_ascii_strup (key, -1);
	ret = g_str_hash (str);
	g_free (str);

	return ret;
}

static gboolean
str_case_equal (gconstpointer v1,
		gconstpointer v2)
{
	return str_case_compare (v1, v2) == 0;
}

static GHashTable *
categorize_ppd_ids (GList *ppd_ids)
{
	GHashTable *models_for_vendors;
	GList *l;

	models_for_vendors = g_hash_table_new_full (str_case_hash,
						    str_case_equal,
						    NULL,
						    (GDestroyNotify)g_hash_table_destroy);

	for (l = ppd_ids; l != NULL; l = l->next) {
		GHashTable *models;
		GList *drivers;
		PPD *ppd = l->data;

		models = g_hash_table_lookup (models_for_vendors,
					      ppd->vendor);
		if (!models) {
			models = g_hash_table_new_full
				(str_case_hash, str_case_equal,
				 NULL, (GDestroyNotify)destroy_ppd_id_list);
			g_hash_table_insert (models_for_vendors,
					     ppd->vendor,
					     models);
		}

		drivers = g_hash_table_lookup (models, ppd->model);
		drivers = g_list_insert_sorted (drivers, ppd, compare_models_by_driver);
		g_hash_table_steal (models, ppd->model);
		g_hash_table_insert (models, ppd->model, drivers);
	}

	return models_for_vendors;
}

static char *
get_selected_vendor (GladeXML *xml)
{
	GtkWidget *item;

	item = option_menu_get_selected_item (xml, "vendor_option_menu");
	if (!item) {
		return NULL;
	}

	return g_strdup (g_object_get_data (G_OBJECT (item), "name"));
}

static char *
get_selected_model (GladeXML *xml)
{
	GtkWidget *tree_view;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;

	tree_view = glade_xml_get_widget (xml, "model_treeview");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

	if (gtk_tree_selection_get_selected (selection,
					     &model,
					     &iter)) {
		char *ret;
		gtk_tree_model_get (model, &iter, 0, &ret, -1);
		return ret;
	} else {
		return NULL;
	}
}


static PPD *
get_selected_ppd (GladeXML *xml)
{
	GtkWidget *item;

	item = option_menu_get_selected_item (xml, "driver_option_menu");
	if (item) {
		PPD *ppd = g_object_get_data (G_OBJECT (item), "selected_ppd");
		return ppd;
	} else {
		return NULL;
	}
}

static void
populate_drivers_from_list (GladeXML *xml, GList *drivers)
{
	GtkWidget *option_menu;
	GtkWidget *menu;

	option_menu = glade_xml_get_widget (xml, "driver_option_menu");
	menu = gtk_menu_new ();

	for (; drivers != NULL; drivers = drivers->next) {
		PPD *ppd = drivers->data;
		GtkWidget *item;
		char *label;

		label = g_strdup_printf (_("%s (%s)"),
					 ppd->driver,
					 ppd->language);
		item = gtk_menu_item_new_with_label (label);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
		g_object_set_data (G_OBJECT (item), "selected_ppd", ppd);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), 0);
}

static void
populate_driver_option_menu (GladeXML *xml)
{
	char	   *vendor  = get_selected_vendor (xml);
	char	   *model   = get_selected_model (xml);
	GtkWidget  *drivers = glade_xml_get_widget (xml, "driver_option_menu");
	GHashTable *models  = NULL;

	g_warning ("populate driver");
	if (vendor && model) {
		GHashTable *vendors = g_object_get_data (
			G_OBJECT (xml), "vendors");
		models = g_hash_table_lookup (vendors, vendor);
	} else
		gtk_widget_set_sensitive (drivers, FALSE);

	gtk_widget_set_sensitive (drivers, models != NULL);
	if (models != NULL)
		populate_drivers_from_list (xml,
			g_hash_table_lookup (models, model));
	else
		gtk_option_menu_remove_menu (GTK_OPTION_MENU (drivers));

	g_free (vendor);
	g_free (model);
}

static void
populate_model_list (GladeXML *xml)
{
	char *vendor;
	GHashTable *vendors;
	GtkWidget *tree_view;
	GtkListStore *store;
	GList *models = NULL;
	GList *l;

	vendor = get_selected_vendor (xml);
	if (!vendor) {
		return;
	}

	tree_view = glade_xml_get_widget (xml, "model_treeview");
	store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tree_view)));
	gtk_list_store_clear (store);

	vendors = g_object_get_data (G_OBJECT (xml), "vendors");

	models = model_list_for_vendor (vendors, vendor);

	for (l = models; l != NULL; l = l->next) {
		GtkTreeIter iter;
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				    0, l->data,
				    -1);
	}

	g_list_free (models);
	g_free (vendor);
}

static void
vendor_option_menu_changed_cb (GtkWidget *option_menu,
			       gpointer user_data)
{
	GladeXML *xml = GLADE_XML (user_data);
	populate_model_list (xml);
}

static void
populate_vendor_option_menu (GladeXML *xml)
{
	GHashTable *ppds;
	GtkWidget *option_menu;
	GtkWidget *menu;
	GList *vendors = NULL;
	GList *l;

	ppds = g_object_get_data (G_OBJECT (xml), "vendors");
	option_menu = glade_xml_get_widget (xml, "vendor_option_menu");

	vendors = vendor_list (ppds);

	menu = gtk_menu_new ();

	for (l = vendors; l != NULL; l = l->next) {
		GtkWidget *item = gtk_menu_item_new_with_label (l->data);
		g_object_set_data_full (G_OBJECT (item),
					"name",
					g_strdup (l->data),
					g_free);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);

	g_list_free (vendors);
}

static void
details_setup (GladeXML *xml)
{
	GtkWidget *option_menu;
	GtkWidget *tree_view;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkListStore *store;
	GtkTreeSelection *selection;

	option_menu = glade_xml_get_widget (xml, "vendor_option_menu");
	g_signal_connect (option_menu, "changed",
			  G_CALLBACK (vendor_option_menu_changed_cb),
			  xml);

	tree_view = glade_xml_get_widget (xml, "model_treeview");
	store = gtk_list_store_new (1, G_TYPE_STRING);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Model"),
							   renderer,
							   "text",
							   0,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

	gtk_tree_view_set_model (GTK_TREE_VIEW (tree_view),
				 GTK_TREE_MODEL (store));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
	g_signal_connect_swapped (selection,
		"changed",
		G_CALLBACK (populate_driver_option_menu), xml);

	druid_watch_sensitivity_widget (xml, "model_treeview");
	druid_watch_sensitivity_widget (xml, "driver_option_menu");

	populate_vendor_option_menu (xml);
}

static void
details_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = TRUE;

	if (get_selected_ppd (xml) != NULL) {
		*next = TRUE;
	} else {
		*next = FALSE;
	}
}

static GtkWidget *
details_back (GladeXML *xml)
{
	if (toggle_button_is_active (xml, "local_radio")) {
		return glade_xml_get_widget (xml, "local_location_page");
	} else {
		return glade_xml_get_widget (xml, "network_location_page");
	}
}

static GtkWidget *
details_next (GladeXML *xml)
{
	PPD *ppd;

	ppd = get_selected_ppd (xml);
	if (ppd) {
		return glade_xml_get_widget (xml, "name_page");
	}

	return NULL;
}

static void
setup_ppds (GladeXML *xml)
{
	GList *ppds;
	GHashTable *vendors;

	ppds = get_ppds ();
	g_object_set_data_full (G_OBJECT (xml),
				"ppds",
				ppds,
				(GDestroyNotify)g_list_free);

	vendors = categorize_ppd_ids (ppds);
	g_object_set_data_full (G_OBJECT (xml),
				"vendors",
				vendors,
				(GDestroyNotify)g_hash_table_destroy);
}

static void
name_setup (GladeXML *xml)
{
	druid_watch_sensitivity_widget (xml, "name_entry");
}

static void
name_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = TRUE;
	*next = entry_has_text (xml, "name_entry");
}

static GtkWidget *
name_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "details_page");
}

static GtkWidget *
name_next (GladeXML *xml)
{
	char *c;
	char *text;
	gboolean valid_name;

	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, "name_entry"))));

	if (!text) {
		return NULL;
	}

	text = g_strstrip (text);

	valid_name = TRUE;
	c = text;
	if (*c) {
		gunichar uc = g_utf8_get_char (c);
		if (!g_unichar_isalpha (uc)) {
			valid_name = FALSE;
		}
	}

	while (valid_name && *c) {
		gunichar uc = g_utf8_get_char (c);
		/* FIXME: Check this */
		if (!g_unichar_isalpha (uc) && !g_unichar_isdigit (uc) && !g_unichar_ispunct (uc)) {
			valid_name = FALSE;
			break;
		}
		c = g_utf8_next_char (c);
	}

	if (valid_name) {
		g_object_set_data_full (G_OBJECT (xml), "selected_name",
					text, g_free);

		text = g_strdup (gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, "description_entry"))));
		if (text && text[0]) {
			g_object_set_data_full (G_OBJECT (xml), "selected_description",
						text, g_free);
		} else if (text) {
			g_free (text);
		}

		return glade_xml_get_widget (xml, "apply_page");
	} else {
		GtkWidget *dialog;

		g_free (text);

		dialog = gtk_message_dialog_new (GTK_WINDOW (glade_xml_get_widget (xml, "add_printer_window")),
						 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("The printer name must begin with a letter and may not contain spaces."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		return NULL;
	}
}

static ipp_t *
add_printer_request (const char *printer_name,
		     const char *device_uri,
		     PPD *ppd,
		     const char *human_name,
		     const char *printer_description,
		     GError **err)
{
	ipp_t *request;
	ipp_t *response;

	request = gnome_cups_request_new_for_printer (CUPS_ADD_PRINTER,
						      printer_name);
	ippAddString (request,
		      IPP_TAG_PRINTER, IPP_TAG_TEXT,
		      "printer-location", NULL, gnome_cups_strdup (device_uri));
	if (printer_description) {
		ippAddString (request,
			      IPP_TAG_PRINTER, IPP_TAG_TEXT,
			      "printer-info", NULL, gnome_cups_strdup (printer_description));
	}

	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_NAME, "ppd-name",
		      NULL, gnome_cups_strdup (ppd->name));
	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_URI, "device-uri",
		      NULL, gnome_cups_strdup (device_uri));
	ippAddBoolean (request, IPP_TAG_PRINTER, "printer-is-accepting-jobs", 1);
	ippAddInteger(request, IPP_TAG_PRINTER, IPP_TAG_ENUM, "printer-state",
		      IPP_PRINTER_IDLE);

	response = gnome_cups_request_execute (request, "/admin/", err);

	return response;
}

static gboolean
add_cups_printer (GladeXML *xml,
		  const char *device_uri, PPD *ppd,
		  const char *printer_name,
		  const char *printer_description)
{
	ipp_t *response;
	GError *err = NULL;

	response = add_printer_request (printer_name, device_uri, ppd,
					printer_name, printer_description,
					&err);
	ippDelete (response);

	if (err) {
		gnome_cups_error_dialog (glade_xml_get_widget (xml, "add_printer_window"),
					 _("Couldn't add printer"),
					 err);
		g_error_free (err);
		return FALSE;
	}

	return TRUE;
}

static void
apply_prepare (GladeXML *xml)
{
	GtkWidget *druid;
	GtkWidget *label;
	char *text;
	char *vendor;
	char *model;

	text = get_selected_location (xml);
	label = glade_xml_get_widget (xml, "location_label");
	gtk_label_set_text (GTK_LABEL (label), text);
	g_free (text);

	vendor = get_selected_vendor (xml);
	model = get_selected_model (xml);
	text = g_strdup_printf ("%s %s", vendor, model);
	label = glade_xml_get_widget (xml, "type_label");
	gtk_label_set_text (GTK_LABEL (label), text);
	g_free (text);
	g_free (vendor);
	g_free (model);

	text = g_object_get_data (G_OBJECT (xml), "selected_name");
	label = glade_xml_get_widget (xml, "name_label");
	gtk_label_set_text (GTK_LABEL (label), text);

	text = g_object_get_data (G_OBJECT (xml), "selected_description");
	label = glade_xml_get_widget (xml, "description_label");
	gtk_label_set_text (GTK_LABEL (label), (text && text[0]) ? text : "None");

	druid = glade_xml_get_widget (xml, "add_printer_druid");
	gnome_druid_set_show_finish (GNOME_DRUID (druid), TRUE);
}

static GtkWidget *
apply_finish (GladeXML *xml)
{
	GtkWidget *window;
	char *uri;
	char *printer_name;
	char *printer_description;
	PPD *ppd;

	uri = get_selected_uri (xml);
	printer_name = g_object_get_data (G_OBJECT (xml), "selected_name");
	printer_description = g_object_get_data (G_OBJECT (xml), "selected_name");
	ppd = get_selected_ppd (xml);

	g_return_val_if_fail (uri != NULL, NULL);
	g_return_val_if_fail (printer_name != NULL, NULL);
	g_return_val_if_fail (ppd != NULL, NULL);

	if (add_cups_printer (xml, uri, ppd, printer_name, printer_description)) {
		window = glade_xml_get_widget (xml, "add_printer_window");
		gtk_widget_destroy (window);
		g_object_unref (xml);
		gtk_main_quit ();
	}

	g_free (uri);

	return NULL;
}

DruidPageDescription pages[] = {
	{
		"local_or_remote_page",
		NULL,
		NULL,
		local_or_remote_sensitivity,
		NULL,
		local_or_remote_next,
	},
	{
		"local_location_page",
		local_location_setup,
		NULL,
		local_location_sensitivity,
		local_location_back,
		local_location_next,
	},
	{
		"network_location_page",
		network_location_setup,
		NULL,
		network_location_sensitivity,
		network_location_back,
		network_location_next,
	},
	{
		"details_page",
		details_setup,
		NULL,
		details_sensitivity,
		details_back,
		details_next
	},
	{
		"name_page",
		name_setup,
		NULL,
		name_sensitivity,
		name_back,
		name_next
	},
	{
		"apply_page",
		NULL,
		apply_prepare,
		NULL,
		NULL,
		apply_finish,
	},
	{ NULL }
};

static int
delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_main_quit ();
	return FALSE;
}

static void
cancel_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

int
main (int argc, char *argv[])
{
	GladeXML *xml;
	GtkWidget *widget;
	GnomeIconTheme *theme;
	char *filename;
	int base_size;
	GdkPixbuf *pixbuf;

	gnome_program_init ("gnome-cups-add",
			    VERSION,
			    LIBGNOMEUI_MODULE, argc, argv,
			    GNOME_PROGRAM_STANDARD_PROPERTIES,
			    GNOME_PARAM_HUMAN_READABLE_NAME, _("Add a Printer"),
			    NULL);
	glade_init ();
	gnome_cups_ui_init ();

	xml = glade_xml_new (GNOME_CUPS_MANAGER_DATADIR "/gnome-cups-add.glade",
			     "add_printer_window",
			     GETTEXT_PACKAGE);

	widget = glade_xml_get_widget (xml, "add_printer_window");
	g_signal_connect (widget, "delete_event",
			  G_CALLBACK (delete_event_cb), NULL);

	theme = gnome_icon_theme_new ();

	filename = gnome_icon_theme_lookup_icon (theme,
						 "gnome-dev-printer-new",
						 32,
						 NULL,
						 &base_size);
	g_object_unref (theme);

	if (filename != NULL) {
		pixbuf = gdk_pixbuf_new_from_file (filename, NULL);
		gtk_window_set_icon (GTK_WINDOW (widget), pixbuf);
		g_object_unref (pixbuf);
	} else
		g_warning ("unable to load icon 'gnome-dev-printer-new'");

	setup_ppds (xml);

	druid_pages_setup (xml, pages);

	gtk_widget_show (widget);

	widget = glade_xml_get_widget (xml, "add_printer_druid");
	g_signal_connect (widget, "cancel",
			  G_CALLBACK (cancel_cb), NULL);

	gtk_main ();

	return 0;
}
