/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2004 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailcombobox.h"

static void         gail_combo_box_class_init              (GailComboBoxClass *klass);
static void         gail_combo_box_object_init             (GailComboBox      *combo_box);
static void         gail_combo_box_real_initialize         (AtkObject      *obj,
                                                            gpointer       data);

static void         gail_combo_box_changed_gtk             (GtkWidget      *widget);

static G_CONST_RETURN gchar* gail_combo_box_get_name       (AtkObject      *obj);
static gint         gail_combo_box_get_n_children          (AtkObject      *obj);
static AtkObject*   gail_combo_box_ref_child               (AtkObject      *obj,
                                                            gint           i);
static void         gail_combo_box_finalize                (GObject        *object);
static void         atk_action_interface_init              (AtkActionIface *iface);

static gboolean     gail_combo_box_do_action               (AtkAction      *action,
                                                            gint           i);
static gboolean     idle_do_action                         (gpointer       data);
static gint         gail_combo_box_get_n_actions           (AtkAction      *action)
;
static G_CONST_RETURN gchar* gail_combo_box_get_description(AtkAction      *action,
                                                            gint           i);
static G_CONST_RETURN gchar* gail_combo_box_action_get_name(AtkAction      *action,
                                                            gint           i);
static gboolean              gail_combo_box_set_description(AtkAction      *action,
                                                            gint           i,
                                                            const gchar    *desc);

static void         atk_selection_interface_init           (AtkSelectionIface *iface);
static gboolean     gail_combo_box_add_selection           (AtkSelection   *selection,
                                                            gint           i);
static gboolean     gail_combo_box_clear_selection         (AtkSelection   *selection);
static AtkObject*   gail_combo_box_ref_selection           (AtkSelection   *selection,
                                                            gint           i);
static gint         gail_combo_box_get_selection_count     (AtkSelection   *selection);
static gboolean     gail_combo_box_is_child_selected       (AtkSelection   *selection,
                                                            gint           i);
static gboolean     gail_combo_box_remove_selection        (AtkSelection   *selection,
                                                            gint           i);

static gpointer parent_class = NULL;

GType
gail_combo_box_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo tinfo =
      {
        sizeof (GailComboBoxClass),
        (GBaseInitFunc) NULL, /* base init */
        (GBaseFinalizeFunc) NULL, /* base finalize */
        (GClassInitFunc) gail_combo_box_class_init, /* class init */
        (GClassFinalizeFunc) NULL, /* class finalize */
        NULL, /* class data */
        sizeof (GailComboBox), /* instance size */
        0, /* nb preallocs */
        (GInstanceInitFunc) gail_combo_box_object_init, /* instance init */
        NULL /* value table */
      };

      static const GInterfaceInfo atk_action_info =
      {
        (GInterfaceInitFunc) atk_action_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
      };
      static const GInterfaceInfo atk_selection_info =
      {
        (GInterfaceInitFunc) atk_selection_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
      };

      type = g_type_register_static (GAIL_TYPE_CONTAINER,
                                     "GailComboBox", &tinfo, 0);

      /*
       * It is not currently possible to implement AtkAction for GtkComboBox
       */
#if 0
      g_type_add_interface_static (type, ATK_TYPE_ACTION,
                                   &atk_action_info);
#endif
      g_type_add_interface_static (type, ATK_TYPE_SELECTION,
                                   &atk_selection_info);
    }

  return type;
}

static void
gail_combo_box_class_init (GailComboBoxClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  AtkObjectClass *class = ATK_OBJECT_CLASS (klass);

  gobject_class->finalize = gail_combo_box_finalize;

  parent_class = g_type_class_peek_parent (klass);

  class->get_name = gail_combo_box_get_name;
  class->get_n_children = gail_combo_box_get_n_children;
  class->ref_child = gail_combo_box_ref_child;
  class->initialize = gail_combo_box_real_initialize;
}

static void
gail_combo_box_object_init (GailComboBox      *combo_box)
{
  combo_box->press_description = NULL;
  combo_box->old_selection = -1;
  combo_box->name = NULL;
}

AtkObject* 
gail_combo_box_new (GtkWidget *widget)
{
  GObject *object;
  AtkObject *accessible;

  g_return_val_if_fail (GTK_IS_COMBO_BOX (widget), NULL);

  object = g_object_new (GAIL_TYPE_COMBO_BOX, NULL);

  accessible = ATK_OBJECT (object);
  atk_object_initialize (accessible, widget);

  return accessible;
}

static void
gail_combo_box_real_initialize (AtkObject *obj,
                                gpointer  data)
{
  GtkComboBox *combo_box;
  GailComboBox *gail_combo_box;

  ATK_OBJECT_CLASS (parent_class)->initialize (obj, data);

  combo_box = GTK_COMBO_BOX (data);

  gail_combo_box = GAIL_COMBO_BOX (obj);

  g_signal_connect (combo_box,
                    "changed",
                    G_CALLBACK (gail_combo_box_changed_gtk),
                    NULL);
  gail_combo_box->old_selection = gtk_combo_box_get_active (combo_box);

  obj->role = ATK_ROLE_COMBO_BOX;
}

static void
gail_combo_box_changed_gtk (GtkWidget *widget)
{
  GtkComboBox *combo_box;
  AtkObject *obj;
  GailComboBox *gail_combo_box;
  gint index;

  combo_box = GTK_COMBO_BOX (widget);

  index = gtk_combo_box_get_active (combo_box);
  obj = gtk_widget_get_accessible (widget);
  gail_combo_box = GAIL_COMBO_BOX (obj);
  if (gail_combo_box->old_selection != index)
    {
      gail_combo_box->old_selection == index;
      g_signal_emit_by_name (obj, "selection_changed");
    }
}

static G_CONST_RETURN gchar* 
gail_combo_box_get_name (AtkObject *obj)
{
  GtkWidget *widget;
  GtkComboBox *combo_box;
  GailComboBox *gail_combo_box;
  GtkTreeIter iter;
  GValue value = { 0, };
  G_CONST_RETURN gchar *name;

  g_return_val_if_fail (GAIL_IS_COMBO_BOX (obj), NULL);

  name = ATK_OBJECT_CLASS (parent_class)->get_name (obj);
  if (name)
    return name;

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return NULL;

  combo_box = GTK_COMBO_BOX (widget);
  gail_combo_box = GAIL_COMBO_BOX (obj);
  g_free (gail_combo_box->name);
  if (gtk_combo_box_get_active_iter (combo_box, &iter))
    {
      gtk_tree_model_get_value (gtk_combo_box_get_model (combo_box), &iter, 0, &value);
      if (G_VALUE_HOLDS_STRING (&value))
        {
          gail_combo_box->name =  (gchar *) g_value_get_string (&value);
        }
    }
  return gail_combo_box->name;
}

/*
 * The children of a GailComboBox are the list of items and the entry field
 * if it is editable.
 */
static gint
gail_combo_box_get_n_children (AtkObject* obj)
{
  gint n_children = 0;
  GtkWidget *widget;

  g_return_val_if_fail (GAIL_IS_COMBO_BOX (obj), 0);

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return 0;

  return n_children;
}

static AtkObject*
gail_combo_box_ref_child (AtkObject *obj,
                          gint      i)
{
  AtkObject *accessible;
  GtkWidget *widget;

  g_return_val_if_fail (GAIL_IS_COMBO_BOX (obj), NULL);

  widget = GTK_ACCESSIBLE (obj)->widget;

  if (widget == NULL)
    /*
     * State is defunct
     */
    return NULL;

  return NULL;
}

static void
atk_action_interface_init (AtkActionIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->do_action = gail_combo_box_do_action;
  iface->get_n_actions = gail_combo_box_get_n_actions;
  iface->get_description = gail_combo_box_get_description;
  iface->get_name = gail_combo_box_action_get_name;
  iface->set_description = gail_combo_box_set_description;
}

static gboolean
gail_combo_box_do_action (AtkAction *action,
                          gint      i)
{
  GailComboBox *combo_box;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (action)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return FALSE;

  if (!GTK_WIDGET_SENSITIVE (widget) || !GTK_WIDGET_VISIBLE (widget))
    return FALSE;

  combo_box = GAIL_COMBO_BOX (action);
  if (i == 0)
    {
      if (combo_box->action_idle_handler)
        return FALSE;

      combo_box->action_idle_handler = g_idle_add (idle_do_action, combo_box);
      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
idle_do_action (gpointer data)
{
  GtkComboBox *combo_box;
  GtkWidget *action_widget;
  GtkWidget *widget;
  GailComboBox *gail_combo_box;

  gail_combo_box = GAIL_COMBO_BOX (data);
  gail_combo_box->action_idle_handler = 0;
  widget = GTK_ACCESSIBLE (gail_combo_box)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return FALSE;

  if (!GTK_WIDGET_SENSITIVE (widget) || !GTK_WIDGET_VISIBLE (widget))
    return FALSE;

  combo_box = GTK_COMBO_BOX (widget);

  return FALSE;
}

static gint
gail_combo_box_get_n_actions (AtkAction *action)
{
  /*
   * The default behavior of a combo_box box is to have one action -
   */
  return 1;
}

static G_CONST_RETURN gchar*
gail_combo_box_get_description (AtkAction *action,
                           gint      i)
{
  if (i == 0)
    {
      GailComboBox *combo_box;

      combo_box = GAIL_COMBO_BOX (action);
      return combo_box->press_description;
    }
  else
    return NULL;
}

static G_CONST_RETURN gchar*
gail_combo_box_action_get_name (AtkAction *action,
                                gint      i)
{
  if (i == 0)
    return "press";
  else
    return NULL;
}

static void
atk_selection_interface_init (AtkSelectionIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->add_selection = gail_combo_box_add_selection;
  iface->clear_selection = gail_combo_box_clear_selection;
  iface->ref_selection = gail_combo_box_ref_selection;
  iface->get_selection_count = gail_combo_box_get_selection_count;
  iface->is_child_selected = gail_combo_box_is_child_selected;
  iface->remove_selection = gail_combo_box_remove_selection;
  /*
   * select_all_selection does not make sense for a combo_box
   * so no implementation is provided.
   */
}

static gboolean
gail_combo_box_add_selection (AtkSelection *selection,
                              gint         i)
{
  GtkComboBox *combo_box;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return FALSE;

  combo_box = GTK_COMBO_BOX (widget);

  gtk_combo_box_set_active (combo_box, i);
  return TRUE;
}

static gboolean 
gail_combo_box_clear_selection (AtkSelection *selection)
{
  GtkComboBox *combo_box;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return FALSE;

  combo_box = GTK_COMBO_BOX (widget);

  gtk_combo_box_set_active (combo_box, -1);
  return TRUE;
}

static AtkObject*
gail_combo_box_ref_selection (AtkSelection *selection,
                              gint         i)
{
  GtkComboBox *combo_box;
  GList * list;
  GtkWidget *item;
  AtkObject *obj;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return NULL;

  combo_box = GTK_COMBO_BOX (widget);

  /*
   * A combo_box box can have only one selection.
   */
  if (i != 0)
    return NULL;

  /*
   * FIXME: We should return the menu item corresponding to the active item.
   */
  return NULL;
}

static gint
gail_combo_box_get_selection_count (AtkSelection *selection)
{
  GtkComboBox *combo_box;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return 0;

  combo_box = GTK_COMBO_BOX (widget);

  return (gtk_combo_box_get_active (combo_box) == -1) ? 0 : 1;
}

static gboolean
gail_combo_box_is_child_selected (AtkSelection *selection,
                                  gint         i)
{
  GtkComboBox *combo_box;
  GtkWidget *item;
  gint j;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
    /*
     * State is defunct
     */
    return FALSE;

  combo_box = GTK_COMBO_BOX (widget);

  j = gtk_combo_box_get_active (combo_box);

  return (j == i);
}

static gboolean
gail_combo_box_remove_selection (AtkSelection *selection,
                                 gint         i)
{
  if (atk_selection_is_child_selected (selection, i))
    atk_selection_clear_selection (selection);

  return TRUE;
}

static gboolean
gail_combo_box_set_description (AtkAction   *action,
                                gint        i,
                                const gchar *desc)
{
  if (i == 0)
    {
      GailComboBox *combo_box;

      combo_box = GAIL_COMBO_BOX (action);
      g_free (combo_box->press_description);
      combo_box->press_description = g_strdup (desc);
      return TRUE;
    }
  else
    return FALSE;
}

static void
gail_combo_box_finalize (GObject *object)
{
  GailComboBox *combo_box = GAIL_COMBO_BOX (object);

  g_free (combo_box->press_description);
  g_free (combo_box->name);
  if (combo_box->action_idle_handler)
    {
      g_source_remove (combo_box->action_idle_handler);
      combo_box->action_idle_handler = 0;
    }
  G_OBJECT_CLASS (parent_class)->finalize (object);
}
