/* 
   Copyright (C) 2004 James Bowes <bowes@cs.dal.ca>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <gtk/gtk.h>
#include <gnome-keyring.h>

#include <string.h>

#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-new-item-dialog.h"

struct _GKMNewItemDialogPrivate
{
  GtkWidget *name_entry;
  GtkWidget *secret_entry;
  GtkWidget *note_entry;

  gboolean name_filled;
  gboolean secret_filled;
  gboolean note_filled;

  GSList *radio_list;
  GtkWidget **network_attributes;
  const gchar *keyring_name;
};

enum
{
  USER_ENTRY = 0,
  DOMAIN_ENTRY,
  SERVER_ENTRY,
  OBJECT_ENTRY,
  AUTHTYPE_ENTRY,
  PROTOCOL_ENTRY,
  PORT_ENTRY,
  NUM_ENTRIES
};

static void gkm_new_item_dialog_class_init (GKMNewItemDialog *class);
static void gkm_new_item_dialog_init       (GKMNewItemDialog *window);
static void gkm_new_item_dialog_finalize   (GObject *object);
static void gkm_new_item_dialog_destroy    (GtkObject *object);

static void gkm_new_item_dialog_set_network_attribute_sensitive (GKMNewItemDialog *dialog, gboolean state);
static void gkm_new_item_dialog_toggle_radio_button (GtkToggleButton *button, GKMNewItemDialog *dialog);

static void name_entry_changed_callback (GtkWidget *entry, GKMNewItemDialog *dialog);
static void secret_entry_changed_callback (GtkWidget *entry, GKMNewItemDialog *dialog);
static void note_entry_changed_callback (GtkTextBuffer *buffer, GKMNewItemDialog *dialog);

static GtkWidget *network_construct_table (GtkWidget **entries);
static GtkWidget *note_construct_table (GtkWidget **note_entry);

static GtkDialogClass *parent_class = NULL;

GType
gkm_new_item_dialog_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMNewItemDialogClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_new_item_dialog_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 

	sizeof (GKMNewItemDialog),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_new_item_dialog_init,
	0
      };

      type = g_type_register_static (GTK_TYPE_DIALOG, "GKMNewItemDialog", &info, 0);
    }

  return type;
}

static void
gkm_new_item_dialog_class_init (GKMNewItemDialog *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_new_item_dialog_finalize;

  object_class->destroy = gkm_new_item_dialog_destroy;
}

static void
gkm_new_item_dialog_init (GKMNewItemDialog *dialog)
{
  GtkWidget *label;
  GtkWidget *radio;
  GtkWidget *table;
  GtkWidget *hbox;

  dialog->priv = g_new0 (GKMNewItemDialogPrivate, 1);

  dialog->priv->network_attributes = g_new0 (GtkWidget *, NUM_ENTRIES);

  dialog->priv->name_filled = FALSE;
  dialog->priv->secret_filled = FALSE;
  dialog->priv->note_filled = FALSE;

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 12);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);
  gtk_table_set_col_spacings (GTK_TABLE (table), 12);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

  label = gtk_label_new_with_mnemonic (_("_Name:"));
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
 
  dialog->priv->name_entry = gtk_entry_new ();
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), dialog->priv->name_entry);
  gtk_table_attach_defaults (GTK_TABLE (table), dialog->priv->name_entry, 1, 2, 0, 1);

  label = gtk_label_new_with_mnemonic (_("_Secret:"));
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
 
  dialog->priv->secret_entry = gtk_entry_new ();
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), dialog->priv->secret_entry);
  gtk_table_attach_defaults (GTK_TABLE (table), dialog->priv->secret_entry, 1, 2, 1, 2);

  /* Generic Secret */
  radio = gtk_radio_button_new_with_mnemonic_from_widget (NULL, _("_Generic Secret"));
  g_signal_connect (G_OBJECT (radio), 
  		    "toggled", 
		    G_CALLBACK (gkm_new_item_dialog_toggle_radio_button), 
		    dialog);
  g_object_set_data (G_OBJECT (radio), "item-type", GINT_TO_POINTER (GNOME_KEYRING_ITEM_GENERIC_SECRET));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), radio, TRUE, TRUE, 0);

  /* Network Secret */
  radio = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON (radio), _("N_etwork Secret"));
  g_signal_connect (G_OBJECT (radio), 
  		    "toggled", 
		    G_CALLBACK (gkm_new_item_dialog_toggle_radio_button), 
		    dialog);
  g_object_set_data (G_OBJECT (radio), "item-type", GINT_TO_POINTER (GNOME_KEYRING_ITEM_NETWORK_PASSWORD));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), radio, TRUE, TRUE, 0);
 
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, 0);

  label = gtk_label_new ("    ");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
 
  table = network_construct_table (dialog->priv->network_attributes);
  gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);

  /* Note */
  radio = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON (radio), _("No_te"));
  g_object_set_data (G_OBJECT (radio), "item-type", GINT_TO_POINTER (GNOME_KEYRING_ITEM_NOTE));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), radio, TRUE, TRUE, 0);
  
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, 0);

  label = gtk_label_new ("    ");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  table = note_construct_table (&dialog->priv->note_entry);
  gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);

  dialog->priv->radio_list = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
  			  GTK_STOCK_CANCEL,
			  GTK_RESPONSE_CANCEL,
			  _("C_reate"),
			  GTK_RESPONSE_ACCEPT, NULL);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, FALSE);

  g_signal_connect (G_OBJECT (dialog->priv->name_entry), 
  		    "changed", 
		    G_CALLBACK (name_entry_changed_callback), 
		    dialog);
  g_signal_connect (G_OBJECT (dialog->priv->secret_entry), 
  		    "changed", 
		    G_CALLBACK (secret_entry_changed_callback), 
		    dialog);
  g_signal_connect (G_OBJECT (gtk_text_view_get_buffer (GTK_TEXT_VIEW (dialog->priv->note_entry))), 
  		    "changed", 
		    G_CALLBACK (note_entry_changed_callback), 
		    dialog);

  gkm_new_item_dialog_set_network_attribute_sensitive (dialog, FALSE);
  gtk_widget_set_sensitive (dialog->priv->note_entry, FALSE);
  
  gtk_widget_show_all (GTK_DIALOG (dialog)->vbox);
}

static void
gkm_new_item_dialog_finalize (GObject *object)
{
  GKMNewItemDialog *dialog;

  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (object));

  dialog = GKM_NEW_ITEM_DIALOG (object);

  g_free (dialog->priv->network_attributes);
  g_free ((void *) dialog->priv->keyring_name);
  g_free (dialog->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_new_item_dialog_destroy (GtkObject *object)
{
  GKMNewItemDialog *dialog;

  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (object));

  dialog = GKM_NEW_ITEM_DIALOG (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (GTK_OBJECT (dialog));
}

GtkWidget *
gkm_new_item_dialog_new (const gchar *keyring_name,
			 GtkWindow   *parent)
{
  GKMNewItemDialog *dialog;
  gchar *title;

  dialog = g_object_new (GKM_TYPE_NEW_ITEM_DIALOG, NULL);
 
  dialog->priv->keyring_name = g_strdup (keyring_name);
 
  if (parent != NULL)
    { 
      gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (parent));
      gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    }
  
  if (strcmp (keyring_name, "session") == 0)
    {
      title = g_strdup (_("New Item For Session Keyring"));
    }
  else
    {
      title = g_strdup_printf (_("New Item For Keyring %s"), keyring_name);
    }

  gtk_window_set_title (GTK_WINDOW (dialog), title);

  g_free (title);

  return GTK_WIDGET (dialog);
}

/**************************************************************
 * Build the widgets for network and note value input.
 */

static GtkWidget *
network_construct_table (GtkWidget **entries)
{
  static const struct 
    {
      char *name;
      int idx;
    } field[] =
    {
      { N_("User:"), USER_ENTRY },
      { N_("Domain:"),  DOMAIN_ENTRY },
      { N_("Server:"), SERVER_ENTRY },
      { N_("Object:"), OBJECT_ENTRY },
      { N_("Authentication Method:"), AUTHTYPE_ENTRY },
      { N_("Protocol:"), PROTOCOL_ENTRY },
      { N_("Port:"), PORT_ENTRY }
    };

  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *entry;
  guint32 i;

  table = gtk_table_new (7, 2, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);

  for (i = 0; i < G_N_ELEMENTS (field) - 1; i++)
    {
      label = gtk_label_new (field[i].name);
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5); 
      gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, i, i+1);
      
      entry = gtk_entry_new ();
      gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
      entries[field[i].idx] = entry; 
      gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, i, i+1);
    }
  
  label = gtk_label_new (field[i].name);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5); 
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, i, i+1);

  entry = gtk_spin_button_new_with_range (1.0, 65535.0, 1.0);
  entries[field[i].idx] = entry;
  gtk_spin_button_set_digits (GTK_SPIN_BUTTON (entry), 0);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
  gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, i, i+1);

  return table;
}

static GtkWidget *
note_construct_table (GtkWidget **note_entry)
{
  GtkWidget *hbox;
  GtkWidget *alignment;
  GtkWidget *label;
  GtkWidget *scrolled_window;

  hbox = gtk_hbox_new (FALSE, 12);
 
  alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
  
  label = gtk_label_new_with_mnemonic (_("Te_xt:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_container_add (GTK_CONTAINER (alignment), label);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), 
  				  GTK_POLICY_NEVER, 
				  GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_window, TRUE, TRUE, 0);

  *note_entry = gtk_text_view_new ();
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (*note_entry), GTK_WRAP_WORD_CHAR);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), *note_entry);
  gtk_container_add (GTK_CONTAINER (scrolled_window), *note_entry);

  return hbox;
}

/**************************************************************
 * Set the sensitivity of the accept button
 */
gchar *
get_note_secret_value (GtkTextBuffer *buffer)
{
  GtkTextIter start;
  GtkTextIter end;
      
  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
      
  return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
}

static void
gkm_new_item_dialog_set_accept_sensitive (GKMNewItemDialog *dialog)
{
  gboolean sensitive;

  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));

  if (gkm_new_item_dialog_get_item_type (dialog) == GNOME_KEYRING_ITEM_NOTE)
    {
      sensitive = dialog->priv->name_filled && dialog->priv->note_filled;
    }
  else
    {
      sensitive = dialog->priv->name_filled && dialog->priv->secret_filled;
    }

  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, sensitive);
}

static void 
name_entry_changed_callback (GtkWidget        *entry, 
			     GKMNewItemDialog *dialog)
{
  const gchar *text;

  g_return_if_fail (GTK_IS_ENTRY (entry));
  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));

  text = gtk_entry_get_text (GTK_ENTRY (entry));

  dialog->priv->name_filled = text != NULL && *text != 0;

  gkm_new_item_dialog_set_accept_sensitive (dialog);
}

static void 
secret_entry_changed_callback (GtkWidget        *entry, 
			       GKMNewItemDialog *dialog)
{
  const gchar *text;
  
  g_return_if_fail (GTK_IS_ENTRY (entry));
  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));

  text = gtk_entry_get_text (GTK_ENTRY (entry));

  dialog->priv->secret_filled = text != NULL && *text != 0;

  gkm_new_item_dialog_set_accept_sensitive (dialog);
}

static void 
note_entry_changed_callback (GtkTextBuffer    *buffer, 
			     GKMNewItemDialog *dialog)
{
  gchar *text;

  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));

  text = get_note_secret_value (buffer);

  dialog->priv->note_filled = text != NULL && *text != 0;

  g_free (text);

  gkm_new_item_dialog_set_accept_sensitive (dialog);
}

/**************************************************************
 * Toggle the sensitivity of widgets depending on the selected item type.
 */

static void
gkm_new_item_dialog_set_network_attribute_sensitive (GKMNewItemDialog *dialog,
						     gboolean	       state)
{
  guint32 i;

  for (i = 0; i < NUM_ENTRIES; i++)
    {
      gtk_widget_set_sensitive (GTK_WIDGET (dialog->priv->network_attributes[i]), state);
    }
}

static void 
gkm_new_item_dialog_toggle_radio_button (GtkToggleButton  *button G_GNUC_UNUSED,
					 GKMNewItemDialog *dialog)
{
  GnomeKeyringItemType type;

  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));

  type = gkm_new_item_dialog_get_item_type (dialog);

  switch (type)
    {
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        gtk_widget_set_sensitive (dialog->priv->secret_entry, TRUE);
	gtk_widget_set_sensitive (dialog->priv->note_entry, FALSE);
	gkm_new_item_dialog_set_network_attribute_sensitive (dialog, TRUE);
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        gtk_widget_set_sensitive (dialog->priv->secret_entry, FALSE);
	gtk_widget_set_sensitive (dialog->priv->note_entry, TRUE);
	gkm_new_item_dialog_set_network_attribute_sensitive (dialog, FALSE);
        break;
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        gtk_widget_set_sensitive (dialog->priv->secret_entry, TRUE);
	gtk_widget_set_sensitive (dialog->priv->note_entry, FALSE);
	gkm_new_item_dialog_set_network_attribute_sensitive (dialog, FALSE);
        break;
      default:
        g_assert_not_reached ();
    }

  gkm_new_item_dialog_set_accept_sensitive (dialog);
}

/**************************************************************
 * Return the selected type in the dialog.
 */

GnomeKeyringItemType 
gkm_new_item_dialog_get_item_type (GKMNewItemDialog *dialog)
{
  GSList *list;
  
  g_return_val_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog), -1);
 
  list = dialog->priv->radio_list;

  while (list != NULL)
    {
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (list->data)))
        {
	  return GPOINTER_TO_INT (g_object_get_data (G_OBJECT (list->data), "item-type"));
	}
    
      list = g_slist_next (list);
    }

  g_assert_not_reached ();
  return -1;
}

/**************************************************************
 * Return the item name.
 */

const gchar *
gkm_new_item_dialog_get_item_name (GKMNewItemDialog *dialog)
{
  const char *name;
  
  g_return_val_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog), NULL);

  name = gtk_entry_get_text (GTK_ENTRY (dialog->priv->name_entry)); 

  return name;
}

/**************************************************************
 * Return the secret.
 */
 
gchar *
gkm_new_item_dialog_get_item_secret (GKMNewItemDialog *dialog)
{
  gchar *secret;
  
  g_return_val_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog), NULL);
 
  if (gkm_new_item_dialog_get_item_type (dialog) == GNOME_KEYRING_ITEM_NOTE)
    {
      GtkTextBuffer *buffer;
      
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (dialog->priv->note_entry));
      
      secret =  get_note_secret_value (buffer);
    }
  else
    {
      secret = g_strdup (gtk_entry_get_text (GTK_ENTRY (dialog->priv->secret_entry)));
    }

  return secret;
}

/*************************************************************
 * Return an attribute list for the selected type.
 */
GnomeKeyringAttributeList *
gkm_new_item_dialog_get_item_attribute_list (GKMNewItemDialog *dialog)
{
  gchar * attributes[] =
    {
      "user",
      "domain",
      "server",
      "object",
      "authtype",
      "protocol",
      "port"
    };

  GnomeKeyringItemType       type;
  GnomeKeyringAttributeList *list;
  
  g_return_val_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog), NULL);

  list = gnome_keyring_attribute_list_new ();
  type = gkm_new_item_dialog_get_item_type (dialog);

  switch (type)
    {
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        {
          guint32 i;
	  guint32 port;
	  const gchar *value;
	
	  for (i = 0; i < NUM_ENTRIES - 1; i++)
	    {
	      value = gtk_entry_get_text (GTK_ENTRY (dialog->priv->network_attributes[i]));
	      gnome_keyring_attribute_list_append_string (list, attributes[i], value);
	    }

	  port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->network_attributes[i]));
	  gnome_keyring_attribute_list_append_uint32 (list, attributes[i], port);
	  
          break;
	}
      case GNOME_KEYRING_ITEM_NOTE:
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        break;
      default:
        g_assert_not_reached ();
    }

  return list;
}
