from desktop.GlassWindow import GlassWindow
from DisplayContainer import DisplayContainer
from WindowSnapper import WindowSnapper
from main import ICON, UNSET_COORD

import gtk


#
# Class for display windows.
#
class Window(GlassWindow, DisplayContainer):

    # mapping: str -> window type
    __WINDOW_FLAGS = {"below": GlassWindow.TYPE_KEEP_BELOW,
                      "above": GlassWindow.TYPE_KEEP_ABOVE,
                      "sticky": GlassWindow.TYPE_STICKY,
                      "managed": GlassWindow.TYPE_MANAGED}

    __window_snapper = WindowSnapper()


    def __init__(self, display):

        self.__display = display

        # the config gets notified about window moves; changes in the config
        # result in window moves - we have this counter to make sure we don't
        # make redundant moves, especially to avoid jerky movements since
        # GConf usually lags behind
        self.__ignore_moves = 0

        # window position for detecting moves
        self.__window_pos = (UNSET_COORD, UNSET_COORD)

        # window size for detecting resizing
        self.__window_size = (0, 0)

        # temporary data used for dragging windows
        self.__is_dragging = 0
        self.__drag_offset = (0, 0)

        self.__shape = None


        GlassWindow.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.set_size_request(-1, -1)
        self.set_default_size(10, 10)
        self.add(display)
        display.set_container(self)

        # set the icon
        self.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))

        # set up event handlers
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK)

        # sawfish needs this
        self.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DOCK)
        self.realize()
        self.set_property("skip-taskbar-hint", 1)
        self.set_property("skip-pager-hint", 1)

        self.set_window_type(self.TYPE_STICKY | self.TYPE_KEEP_BELOW)


    def __on_button(self, src, event, is_release):

        wx, wy = self.__window_pos
        if (not is_release and event.button == 2):
            x, y = self.get_pointer()
            self.__is_dragging = 1
            self.__drag_offset = (x, y)
            self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))
            dont_snap = (event.state & gtk.gdk.SHIFT_MASK)
            gtk.timeout_add(20, self.__on_motion, dont_snap)
            self.__display.set_sensitive(gtk.FALSE)
            w, h = self.__window_size
            self.__window_snapper.remove(wx, wy, w, h)
        elif (is_release and self.__is_dragging):
            self.__is_dragging = 0
            self.window.set_cursor(None)
            self.__display.set_sensitive(gtk.TRUE)
            w, h = self.size_request()
            self.__window_size = (w, h)
            self.__window_snapper.insert(wx, wy, w, h)

            # set window shape after the window has been placed
            if (self.__shape):
                self.set_shape(self.__shape)
                self.__shape = None
            

    def __on_motion(self, dont_snap = gtk.FALSE):

        if (self.__is_dragging):
            offx, offy = self.__drag_offset
            winx, winy = self.get_position()
            x, y = self.get_pointer()
            x += winx; y += winy

            new_x = x - offx
            new_y = y - offy
            if (self.__window_pos != (new_x, new_y)):
                if (not dont_snap):
                    w, h = self.size_request()
                    new_x, new_y = self.__window_snapper.snap(new_x, new_y,
                                                              w, h)
                self.move(new_x, new_y)
                self.__ignore_moves += 1
                self.__display.save_position(new_x, new_y)
                self.__window_pos = (new_x, new_y)
            
            return gtk.TRUE

        else:
            return gtk.FALSE


    def close(self):

        wx, wy = self.__window_pos
        w, h = self.__window_size
        self.__window_snapper.remove(wx, wy, w, h)
        self.destroy()


    def set_position(self, x, y):

        ox, oy = self.__window_pos
        w, h = self.__window_size
        self.__window_snapper.remove(ox, oy, w, h)
        if (x == y == UNSET_COORD and self.window):
            width, height = self.size_request()
            self.__is_dragging = 1
            self.__drag_offset = (width / 2, height / 2)
            self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))
            gtk.timeout_add(20, self.__on_motion)
            self.__display.set_sensitive(gtk.FALSE)
            return

        x = max(0, x)
        y = max(0, y)

        if (self.__ignore_moves == 0):
            if ((x, y) != self.__window_pos):
                self.move(x, y)
                self.__window_pos = (x, y)
        else:
            self.__ignore_moves -= 1

        w, h = self.size_request()
        self.__window_size = (w, h)
        self.__window_snapper.insert(x, y, w, h)


    def set_window_flags(self, value):

        flags = 0
        value = value.split(",")
        for p in value: flags |= self.__WINDOW_FLAGS[p.strip()]



    def set_shape(self, mask):

        if (self.__window_pos == (UNSET_COORD, UNSET_COORD)):
            self.__shape = mask
        else:
            self.shape_combine_mask(mask, 0, 0)
