/* gok-button.c
*
* Copyright 2002 Sun Microsystems, Inc.,
* Copyright 2001 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-button.h"
#include "gok-keyboard.h"
#include "gok-log.h"
#include "gok-scanner.h"
#include "main.h"
#include <stdio.h>
#include <gnome.h>

static void gok_button_class_init    (GokButtonClass *klass);
static void gok_button_init          (GokButton      *gok_button);
GokKey* gok_button_find_key (GtkWidget* pWidget);

/**
* gok_button_get_type
*
* returns: the gok button type
**/
GtkType
gok_button_get_type ()
{
  static GtkType GokButtonType = 0;
 if (!GokButtonType)
    {
      static const GtkTypeInfo GokButtonInfo =
      {
	"GokButton",
	sizeof (GokButton),
	sizeof (GokButtonClass),
	(GtkClassInitFunc) gok_button_class_init,
	(GtkObjectInitFunc) gok_button_init,
	NULL,
     NULL,
	(GtkClassInitFunc) NULL,
      };

      GokButtonType = gtk_type_unique (GTK_TYPE_TOGGLE_BUTTON, &GokButtonInfo);
    }

  return GokButtonType;
}

/**
* gok_button_class_init
* @class: Pointer to the class that will be initialized.
*
**/
static void
gok_button_class_init (GokButtonClass *class)
{
	GtkWidgetClass   *widget_class;
	
	widget_class   = (GtkWidgetClass*)   class;
}

/**
* gok_button_init
* @gok_button: Pointer to the button that will be initialized.
*
* Initializes a GOK button.
**/
static void
gok_button_init (GokButton* gok_button)
{
 GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) gok_button;
  gok_button->pLabel = NULL;
}

/**
* gok_button_new_with_label:
* @pText: Text string for the button's label.
*
* Creates a new GOK button with a label.
*
* Returns: A pointer to the new button, NULL if it could not be created.
*/
GtkWidget* gok_button_new_with_label (const gchar *pText)
{
	GtkWidget* pNewButton;

	/* create a new GOK button */
	pNewButton = GTK_WIDGET (gtk_type_new (gok_button_get_type ()));
	
	if (pNewButton == NULL)
	{
		gok_log_x ("Error: Can't create new GOK button '%s' in gok_button_new_with_label!\n", pText);
		return NULL;
	}

	/* add the label to it */
	((GokButton*)pNewButton)->pLabel = gtk_label_new (pText);
	gtk_widget_show (((GokButton*)pNewButton)->pLabel);
	gtk_container_add (GTK_CONTAINER (pNewButton), ((GokButton*)pNewButton)->pLabel);

	return pNewButton;
}

/**
* gok_button_enter_notify:
* @widget: Pointer to the widget that has just been entered.
* @event: Not sure?
*
* This handler is called whenever a widget on the keyboard is entered.
*
* Returns: TRUE if the given widget is associated with a GOK key, FALSE if
* the given button is not associated with a GOK key.
*/
gint gok_button_enter_notify   (GtkWidget *widget, GdkEventCrossing   *event)
{
	GokKey* pKey;

	pKey = gok_button_find_key (widget);
	if (pKey != NULL)
	{
		gok_scanner_on_key_enter (pKey);
	}

	return FALSE;
}

/**
* gok_button_leave_notify:
* @widget: Pointer to the widget that has just been left.
* @event: Not sure?
*
* This handler is called whenever a widget on the keyboard has been left.
*
* Returns: TRUE if the given widget is associated with a GOK key, FALSE if
* the given button is not associated with a GOK key.
*/
gint gok_button_leave_notify   (GtkWidget *widget, GdkEventCrossing   *event)
{
	GokKey* pKey;
	gint x, y;

	if (GOK_IS_BUTTON (widget))
	{
		gdk_drawable_get_size (GTK_BUTTON (widget)->event_window,
				       &x, &y);
		if (event->x <= 0 || event->x >= x ||
		    event->y <= 0 || event->y >= y)
		{
			pKey = gok_button_find_key (widget);
			if (pKey != NULL)
			{
				gok_scanner_on_key_leave (pKey);
			}	
		}
	}	
	return FALSE;
}

/**
* gok_button_state_changed:
* @widget: The button that has just changed state.
* @state: State requested (not necessarily the state we set it).
* @user_data: Any user data associated with the widget (ignored by us).
*
* This is called each time the button state is changed. We handle this call
* and make sure the button is set to the state we want.
*/
void gok_button_state_changed (GtkWidget *widget, GtkStateType state, gpointer user_data)
{
	GokKey* pKey;

	if (GOK_IS_BUTTON (widget)) 
	{
	        pKey = gok_button_find_key (widget);
		if (pKey)
			gtk_widget_set_state (widget, pKey->State);
	}

}

/**
* gok_button_find_key
* @pWidget: Pointer to the widget that you need the GOK key for.
*
* Finds the GOK key given a pointer to a widget.
*
* Returns: A pointer to the key that holds the widget, NULL if not found.
**/
GokKey* gok_button_find_key (GtkWidget* pWidget)
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;

	pKeyboard = gok_main_get_current_keyboard();
	g_assert (pKeyboard != NULL);

	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->pButton == pWidget)
		{
			return pKey;
		}
		pKey = pKey->pKeyNext;
	}

	return NULL;
}

