/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define DEBUG_MSG(x) g_print x
//#define DEBUG_MSG(x)

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);

#include "galeon-tab.h"
#include "galeon-macros.h"
#include "galeon-shell.h"
#include "galeon-embed-popup-bw.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include "galeon-embed-prefs.h"
#include "galeon-embed-autoscroller.h"
#include "galeon-embed-manual-scroller.h"
#include "gul-gestures.h"

#include <bonobo/bonobo-i18n.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkmisc.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtkselection.h>
#include <string.h>

struct GaleonTabPrivate
{
	GaleonEmbed *embed;
	GaleonWindow *window;
	gboolean is_active;
	TabLoadStatus load_status;
	char status_message[255];
	char *title;
	char *location;
	int load_percent;
	gboolean visibility;
	int cur_requests;
	int total_requests;
	int width;
	int height;
};

static GHashTable *gestures_actions = NULL;

static void
galeon_tab_class_init (GaleonTabClass *klass);
static void
galeon_tab_init (GaleonTab *tab);
static void
galeon_tab_finalize (GObject *object);

enum
{
	PROP_0,
	PROP_GALEON_SHELL
};

static void
galeon_tab_link_message_cb (GaleonEmbed *embed, 
			    const char *message,
			    GaleonTab *tab);
static void
galeon_tab_js_status_cb (GaleonEmbed *embed, 
			 const char *status,
			 GaleonTab *tab);
static void
galeon_tab_location_cb (GaleonEmbed *embed, GaleonTab *tab);
static void
galeon_tab_title_cb (GaleonEmbed *embed, GaleonTab *tab);
static void
galeon_tab_progress_cb (GaleonEmbed *embed, const char *uri,
			gint curprogress, gint maxprogress,
			GaleonTab *tab);
static void
galeon_tab_net_state_cb (GaleonEmbed *embed, const char *uri,
			 EmbedState state, GaleonTab *tab);
static void
galeon_tab_new_window_cb (GaleonEmbed *embed, GaleonEmbed **new_embed,
			  EmbedChromeMask chromemask, GaleonTab *tab);
static void
galeon_tab_visibility_cb (GaleonEmbed *embed, gboolean visibility,
			  GaleonTab *tab);
static void
galeon_tab_destroy_brsr_cb (GaleonEmbed *embed, GaleonTab *tab);
static gint
galeon_tab_open_uri_cb (GaleonEmbed *embed, const char *uri,
			GaleonTab *tab);
static void
galeon_tab_size_to_cb (GaleonEmbed *embed, gint width, gint height,
		       GaleonTab *tab);
static gint
galeon_tab_dom_mouse_click_cb (GaleonEmbed *embed,
			       GaleonEmbedEvent *event,
			       GaleonTab *tab);
static gint
galeon_tab_dom_mouse_down_cb (GaleonEmbed *embed,
			      GaleonEmbedEvent *event,
			      GaleonTab *tab);
static void
galeon_tab_security_change_cb (GaleonEmbed *embed, EmbedSecurityLevel level,
			       GaleonTab *tab);
static void
galeon_tab_zoom_changed_cb (GaleonEmbed *embed, gint zoom, 
			    GaleonTab *tab);

static void
galeon_tab_gesture_performed_cb (GulGestures *g, const gchar *sequence, GaleonTab *tab);

static void
galeon_tab_gesture_cancelled_cb (GulGestures *ges, GaleonTab *tab);

static GObjectClass *parent_class = NULL;

/* Class functions */

GType 
galeon_tab_get_type (void)
{
        static GType galeon_tab_type = 0;

        if (galeon_tab_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (GaleonTabClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) galeon_tab_class_init,
                        NULL,
                        NULL, /* class_data */
                        sizeof (GaleonTab),
                        0, /* n_preallocs */
                        (GInstanceInitFunc) galeon_tab_init
                };

                galeon_tab_type = g_type_register_static (G_TYPE_OBJECT,
							  "GaleonTab",
							  &our_info, 0);
        }

        return galeon_tab_type;
}

static void
galeon_tab_class_init (GaleonTabClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        parent_class = g_type_class_peek_parent (klass);

        object_class->finalize = galeon_tab_finalize;
}

static void
galeon_tab_embed_destroy_cb (GtkWidget *widget, GaleonTab *tab)
{
#ifdef DEBUG_MARCO
	g_print ("GtkMozEmbed destroy signal on GaleonTab\n");
#endif
	g_object_unref (tab);
}

static void
galeon_tab_init (GaleonTab *tab)
{
	GObject *embed, *embed_widget;
	GaleonEmbedShell *shell;
	
        tab->priv = g_new0 (GaleonTabPrivate, 1);

	shell = galeon_shell_get_embed_shell (galeon_shell);
	
	tab->priv->embed = galeon_embed_new (G_OBJECT(shell));

	tab->priv->window = NULL;
	tab->priv->is_active = FALSE;
	*tab->priv->status_message = '\0';
	tab->priv->load_status = TAB_LOAD_NONE;
	tab->priv->load_percent = 0;
	tab->priv->title = NULL;
	tab->priv->location = NULL;
	tab->priv->total_requests = 0;
	tab->priv->cur_requests = 0;
	tab->priv->width = -1;
	tab->priv->height = -1;

	
	embed = G_OBJECT (tab->priv->embed);
	embed_widget = G_OBJECT (tab->priv->embed);

	/* set a pointer in the embed's widget back to the tab */
	g_object_set_data (embed_widget, "GaleonTab", tab);

	g_signal_connect (embed_widget, "destroy",
			  GTK_SIGNAL_FUNC (galeon_tab_embed_destroy_cb), 
			  tab);
	g_signal_connect (embed, "ge_link_message",
			  GTK_SIGNAL_FUNC (galeon_tab_link_message_cb), 
			  tab);
	g_signal_connect (embed, "ge_js_status",
			  GTK_SIGNAL_FUNC (galeon_tab_js_status_cb), 
			  tab);
	g_signal_connect (embed, "ge_location",
			  GTK_SIGNAL_FUNC (galeon_tab_location_cb), 
			  tab);
	g_signal_connect (embed, "ge_title",
			  GTK_SIGNAL_FUNC (galeon_tab_title_cb), 
			  tab);
	g_signal_connect (embed, "ge_zoom_change",
			  GTK_SIGNAL_FUNC (galeon_tab_zoom_changed_cb),
			  tab);
	g_signal_connect (embed, "ge_progress",
			  GTK_SIGNAL_FUNC (galeon_tab_progress_cb), 
			  tab);
	g_signal_connect (embed, "ge_net_state",
			  GTK_SIGNAL_FUNC (galeon_tab_net_state_cb), 
			  tab);
	g_signal_connect (embed, "ge_new_window",
			  GTK_SIGNAL_FUNC (galeon_tab_new_window_cb), 
			  tab);
	g_signal_connect (embed, "ge_visibility",
			  GTK_SIGNAL_FUNC (galeon_tab_visibility_cb), 
			  tab);
	g_signal_connect (embed, "ge_destroy_brsr",
			  GTK_SIGNAL_FUNC (galeon_tab_destroy_brsr_cb), 
			  tab);
	g_signal_connect (embed, "ge_open_uri",
			  GTK_SIGNAL_FUNC (galeon_tab_open_uri_cb), 
			  tab);
	g_signal_connect (embed, "ge_size_to",
			  GTK_SIGNAL_FUNC (galeon_tab_size_to_cb), 
			  tab);
	g_signal_connect (embed, "ge_dom_mouse_click",
			  GTK_SIGNAL_FUNC (galeon_tab_dom_mouse_click_cb), 
			  tab);
	g_signal_connect (embed, "ge_dom_mouse_down",
			  GTK_SIGNAL_FUNC (galeon_tab_dom_mouse_down_cb), 
			  tab);
	g_signal_connect (embed, "ge_security_change",
			  GTK_SIGNAL_FUNC (galeon_tab_security_change_cb), 
			  tab);
}

/* Destructor */

static void
galeon_tab_finalize (GObject *object)
{
        GaleonTab *tab;

	g_return_if_fail (IS_GALEON_TAB (object));

	tab = GALEON_TAB (object);

        g_return_if_fail (tab->priv != NULL);

	g_idle_remove_by_data (tab->priv->embed);
	
        g_free (tab->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);

#ifdef DEBUG_MARCO
	g_print ("GaleonTab finalized %p\n", tab);
#endif
}

/* Public functions */

GaleonTab *
galeon_tab_new ()
{
	GaleonTab *tab;

	tab = GALEON_TAB (g_object_new (GALEON_TAB_TYPE, NULL));
	g_return_val_if_fail (tab->priv != NULL, NULL);
	return tab;
}

static void	      
galeon_tab_set_load_status (GaleonTab *tab,
			    TabLoadStatus status)
{
	tab->priv->load_status &= TAB_LOAD_ACTIVATED;
	tab->priv->load_status |= status;

	if (tab->priv->load_status & TAB_LOAD_COMPLETED)
	{
		Session *s;
	    	s = galeon_shell_get_session (galeon_shell);
        	session_save (s, SESSION_CRASHED);
	}
}

GaleonEmbed *
galeon_tab_get_embed (GaleonTab *tab)
{
	g_return_val_if_fail (IS_GALEON_TAB (G_OBJECT (tab)), NULL);

	return tab->priv->embed;
}

void
galeon_tab_set_window (GaleonTab *tab, GaleonWindow *window)
{
	g_return_if_fail (IS_GALEON_TAB (G_OBJECT (tab)));
	if (window) g_return_if_fail (IS_GALEON_WINDOW (G_OBJECT (window)));

	tab->priv->window = window;
}

GaleonWindow *
galeon_tab_get_window (GaleonTab *tab)
{
	g_return_val_if_fail (IS_GALEON_TAB (G_OBJECT (tab)), NULL);

	return tab->priv->window;
}

static void
galeon_tab_update_color (GaleonTab *tab)
{
	TabLoadStatus status = galeon_tab_get_load_status (tab);
	GulNotebookPageLoadStatus page_status = 0;
	GtkWidget *nb;

	nb = galeon_window_get_notebook (tab->priv->window);
	
	if (status & TAB_LOAD_ACTIVATED)
	{
		page_status = GUL_NOTEBOOK_TAB_LOAD_ACTIVATED;
	}
	else if (status & TAB_LOAD_STARTED)
        {
       		page_status = GUL_NOTEBOOK_TAB_LOAD_STARTED;
	}
        else if (status & TAB_LOAD_COMPLETED)
        {
		page_status = GUL_NOTEBOOK_TAB_LOAD_COMPLETED;
        }
	else
	{
		g_assert_not_reached ();
	}

	gul_notebook_set_page_status (GUL_NOTEBOOK (nb),
				      GTK_WIDGET (tab->priv->embed),
				      page_status);
}

void
galeon_tab_set_is_active (GaleonTab *tab, gboolean is_active)
{
	g_return_if_fail (IS_GALEON_TAB (G_OBJECT (tab)));

	tab->priv->is_active = is_active;

	galeon_tab_set_load_status (tab, TAB_LOAD_ACTIVATED);
	galeon_tab_update_color (tab);
}

gboolean
galeon_tab_get_is_active (GaleonTab *tab)
{
	g_return_val_if_fail (IS_GALEON_TAB (G_OBJECT (tab)), FALSE);

	return tab->priv->is_active;
}

gboolean
galeon_tab_get_visibility (GaleonTab *tab)
{
	return tab->priv->visibility;
}

void
galeon_tab_get_size (GaleonTab *tab, int *width, int *height)
{
	*width = tab->priv->width;
	*height = tab->priv->height;
}

static void 
galeon_tab_set_visibility (GaleonTab *tab,
                           gboolean visible)
{	
	g_return_if_fail (tab->priv->window != NULL);

	/* FIXME show/hide the tab widget */
	
	tab->priv->visibility = visible;
}

/* Private callbacks for embed signals */

static void
galeon_tab_link_message_cb (GaleonEmbed *embed, 
			    const char *message,
			    GaleonTab *tab)
{
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);
	if (!tab->priv->is_active) return;

	g_strlcpy (tab->priv->status_message, 
		   message, 255);
	
	galeon_window_update_control (tab->priv->window,
				      StatusbarMessageControl);
}

static void
galeon_tab_js_status_cb (GaleonEmbed *embed,
			 const char *status,
			 GaleonTab *tab)
{
	gboolean allow_statusbar_rewrite;
	
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);
	
	allow_statusbar_rewrite = eel_gconf_get_boolean 
		(CONF_FILTERING_DEFAULT_STATUSBAR);
	
	if (!tab->priv->is_active || !allow_statusbar_rewrite)
		return;

	g_strlcpy (tab->priv->status_message, 
		   status, 255);
		
	galeon_window_update_control (tab->priv->window,
				      StatusbarMessageControl);
}

static void
galeon_tab_location_cb (GaleonEmbed *embed, GaleonTab *tab)
{		
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);

	if (tab->priv->location) g_free (tab->priv->location);
	galeon_embed_get_location (embed, TRUE, FALSE, 
				   &tab->priv->location);

	if (tab->priv->is_active)
	{
		galeon_window_update_control (tab->priv->window, LocationControl);
		galeon_window_update_control (tab->priv->window, NavControl);
	}

	galeon_window_site_visited (tab->priv->window, tab->priv->location, tab->priv->title);
}

static void
galeon_tab_zoom_changed_cb (GaleonEmbed *embed, gint zoom, GaleonTab *tab)
{
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);
	
	if (tab->priv->is_active)
	{
		galeon_window_update_control (tab->priv->window, ZoomControl);
	}
}

static void
galeon_tab_set_title (GaleonTab *tab, const char *title)
{
	GtkWidget *nb;
	
	nb = galeon_window_get_notebook (tab->priv->window);
	gul_notebook_set_page_title (GUL_NOTEBOOK (nb),
				     GTK_WIDGET (tab->priv->embed),
				     title);
}

static void
galeon_tab_title_cb (GaleonEmbed *embed, GaleonTab *tab)
{
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);

	if (tab->priv->title) g_free (tab->priv->title);
	galeon_embed_get_title (embed, &tab->priv->title);
	
	if (*(tab->priv->title) == '\0')
	{
		g_free (tab->priv->title);
		tab->priv->title = g_strdup (_("Untitled"));
	}
	
	galeon_window_site_visited (tab->priv->window, tab->priv->location, tab->priv->title);

	galeon_tab_set_title (tab, tab->priv->title);
	
	if (tab->priv->is_active)
	{
		galeon_window_update_control (tab->priv->window, 
					      TitleControl);
	}
}

static int
build_load_percent (int bytes_loaded, int max_bytes_loaded)
{
	if (max_bytes_loaded > 0)
	{
		return (bytes_loaded * 100) / max_bytes_loaded;
	}
	else
	{
		return -1;
	}
}

static char *
get_host_name_from_uri (const char *uri)
{
	GnomeVFSURI *vfs_uri = NULL;
	const char *host = NULL;
	char *result;
	
	if (uri)
	{
		vfs_uri = gnome_vfs_uri_new (uri);
	}
	
	if (vfs_uri)
	{
		host = gnome_vfs_uri_get_host_name (vfs_uri);
	}
	
	if (!host)
	{
		host = _("site");
	}

	result = g_strdup (host);
	
	if (vfs_uri) gnome_vfs_uri_unref (vfs_uri);
	
	return result;
}

static void
build_progress_message (char *message,
			const char *uri,
			int bytes_loaded, 
			int max_bytes_loaded, 
			int load_percent)
{
	char *host;

	host = get_host_name_from_uri (uri);
	
	if (max_bytes_loaded == 0 || 
	    bytes_loaded > max_bytes_loaded ) 
	{
		g_snprintf (message, 255, 
			    _("Transferring data from %s (%d kB loaded)"), 
                            host, bytes_loaded / 1024);	
	}
	else
	{
		g_snprintf (message, 255, 
                    	    _("Transferring data from %s (%d%% complete, %d kB of %d kB loaded)"),
                    	    host, load_percent, bytes_loaded / 1024, max_bytes_loaded / 1024);
	}

	g_free (host);
}

static void
galeon_tab_progress_cb (GaleonEmbed *embed, const char *uri,
			gint curprogress, gint maxprogress,
			GaleonTab *tab)
{
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);
	if (!tab->priv->is_active) return;

	build_progress_message (tab->priv->status_message,
				uri,
				curprogress,
				maxprogress,
				tab->priv->load_percent);
	
	galeon_window_update_control (tab->priv->window,
				      StatusbarMessageControl);
}

static void
build_net_state_message (char *message,
			 const char *uri,
			 EmbedState flags)
{
	const char *msg = NULL;	
	char *host;

	host = get_host_name_from_uri (uri);
	
	/* IS_REQUEST and IS_NETWORK can be both set */
	
	if (flags & EMBED_STATE_IS_REQUEST)
        {
                if (flags & EMBED_STATE_REDIRECTING)
                {
                	msg = _("Redirecting to %s...");
                }
                else if (flags & EMBED_STATE_TRANSFERRING)
                {
                    	msg = _("Transferring data from %s...");
                }
                else if (flags & EMBED_STATE_NEGOTIATING)
                {
                   	msg = _("Waiting for authorization from %s...");
                }
        }
	
	if (flags & EMBED_STATE_IS_NETWORK)
        {
                if (flags & EMBED_STATE_START)
                {
                        msg = _("Loading %s...");
                }
                else if (flags & EMBED_STATE_STOP)
                {
                      	msg = _("Done.");
                }
        }

	if (msg)
	{
		g_snprintf (message, 255, msg, host);
	}

	g_free (host);
}

static void
build_progress_from_requests (GaleonTab *tab, EmbedState state)
{
	int load_percent;
	
	if (state & EMBED_STATE_IS_REQUEST)
        {
                if (state & EMBED_STATE_START)
                {
			tab->priv->total_requests ++;
		}
		else if (state & EMBED_STATE_STOP)
		{
			tab->priv->cur_requests ++;
		}
		
		load_percent = build_load_percent (tab->priv->cur_requests,
						   tab->priv->total_requests);
		if (load_percent > tab->priv->load_percent)
		{
			tab->priv->load_percent = load_percent;
		}
	}
}

static void
ensure_location (GaleonTab *tab, const char *uri)
{
	if (tab->priv->location == NULL)
	{
		tab->priv->location = g_strdup (uri);
		galeon_window_update_control (tab->priv->window,
					      LocationControl);
	}
}

static void
galeon_tab_net_state_cb (GaleonEmbed *embed, const char *uri,
			 EmbedState state, GaleonTab *tab)
{	
	build_net_state_message (tab->priv->status_message, uri, state);

	galeon_window_update_control (tab->priv->window,
				      StatusbarMessageControl);
	
	if (state & EMBED_STATE_IS_NETWORK)
	{
		if (state & EMBED_STATE_START)
		{
			tab->priv->total_requests = 0;
			tab->priv->cur_requests = 0;
			tab->priv->load_percent = 0;
		
			ensure_location (tab, uri);
			
			galeon_tab_set_load_status (tab, TAB_LOAD_STARTED);
			galeon_window_update_control (tab->priv->window,
						      SpinnerControl);
			galeon_tab_update_color (tab);
		}
		else if (state & EMBED_STATE_STOP)
		{			
			tab->priv->load_percent = 0;
			
			galeon_tab_set_load_status (tab, TAB_LOAD_COMPLETED);
			
			galeon_window_update_control (tab->priv->window,
						      NavControl);
			galeon_window_update_control (tab->priv->window,
						      SpinnerControl);
			galeon_tab_update_color (tab);
		}		
	}	
	
	build_progress_from_requests (tab, state);
	
	galeon_window_update_control (tab->priv->window,
				      StatusbarProgressControl);
}

static void
galeon_tab_new_window_cb (GaleonEmbed *embed, GaleonEmbed **new_embed,
			  EmbedChromeMask chromemask, GaleonTab *tab)
{
	GaleonTab *new_tab;
	GaleonWindow *window;
	gboolean open_in_tab;

	open_in_tab = (chromemask & EMBED_CHROME_OPENASCHROME) ?
		      FALSE :
		      eel_gconf_get_boolean (CONF_TABS_TABBED_POPUPS);

	if (open_in_tab)
	{
		window = galeon_tab_get_window (tab);
	}
	else
	{
		window = galeon_window_new ();
		galeon_window_set_chrome (window, chromemask);
	}
	
	new_tab = galeon_tab_new ();
        galeon_window_add_tab (window, new_tab, FALSE);
	
	*new_embed = galeon_tab_get_embed (new_tab);
}

static gboolean
let_me_resize_hack (gpointer data)
{
	gtk_widget_set_size_request (GTK_WIDGET(data), 
				     -1, -1);
	return FALSE;
}

static void
galeon_tab_visibility_cb (GaleonEmbed *embed, gboolean visibility,
			  GaleonTab *tab)
{
	GaleonWindow *window;
	
	if (visibility)
	{
		gtk_widget_show (GTK_WIDGET(embed));
	}
	else
	{
		gtk_widget_hide (GTK_WIDGET(embed));
	}
        
	galeon_tab_set_visibility (tab, visibility);

	window = galeon_tab_get_window (tab);
	g_return_if_fail (window != NULL);
	
	galeon_window_update_control (window, WindowVisibilityControl);
}

static void
galeon_tab_destroy_brsr_cb (GaleonEmbed *embed, GaleonTab *tab)
{
	GaleonWindow *window;

	window = galeon_tab_get_window (tab);

	galeon_window_remove_tab (window, tab);
}

static gint
galeon_tab_open_uri_cb (GaleonEmbed *embed, const char *uri,
			GaleonTab *tab)
{
	return FALSE;
}

static void
galeon_tab_size_to_cb (GaleonEmbed *embed, gint width, gint height,
		       GaleonTab *tab)
{
	GList *tabs;
	GaleonWindow *window;
	GtkWidget *widget;
	EmbedChromeMask chromemask;

	tab->priv->width = width;
	tab->priv->height = height;
	
	window = galeon_tab_get_window (tab);
	tabs = (GList *) galeon_window_get_tabs (window);
	widget = GTK_WIDGET (embed);
	chromemask = galeon_window_get_chrome (window);
	
	/* Do not resize window with multiple tabs.
	 * Do not resize window already showed because
	 * it's not possible to calculate a sensible window
	 * size based on the embed size */
	if (g_list_length (tabs) == 1 && !tab->priv->visibility)
	{
		gtk_widget_set_size_request 
			(widget, width, height);

		/* HACK reset widget requisition after the container
		 * has been resized. It appears to be the only way
		 * to have the window sized according to embed 
		 * size correctly.
		 * We dont do it for XUL dialogs because in that case
		 * a "forced" requisition appear correct.
		 */
		if (!(chromemask & EMBED_CHROME_OPENASCHROME))
		{
			g_idle_add (let_me_resize_hack, embed);
		}
	}
}

static gint
galeon_tab_dom_mouse_click_cb (GaleonEmbed *embed,
			       GaleonEmbedEvent *event,
			       GaleonTab *tab)
{
	return FALSE;
}

static gint
galeon_tab_dom_mouse_down_cb  (GaleonEmbed *embed,
			       GaleonEmbedEvent *event,
			       GaleonTab *tab)
{
	GaleonEmbedPopup *popup;
	GaleonWindow *window;
	BonoboWindow *bwin;
	int button;
	EmbedEventContext context;
        
	galeon_embed_event_get_context (event, &context); 
	
	g_assert (IS_GALEON_EMBED_EVENT(event));
	
	galeon_embed_event_get_mouse_button (event, &button);
	
	window = galeon_tab_get_window (tab);
	g_return_val_if_fail (window != NULL, FALSE);

	bwin = BONOBO_WINDOW(window);
	g_assert (bwin != NULL);

	if (button == 2)
	{
		gboolean gestures = eel_gconf_get_boolean (CONF_MOUSE_RIGHT_BUTTON_GESTURES);
		
		if (!gestures) /* bring up the menu immediately */
		{
			popup = GALEON_EMBED_POPUP (galeon_window_get_popup_factory (window));
			galeon_embed_popup_set_event (popup, event);
			galeon_embed_popup_show (popup, embed);
		}
		else /* start the gesture */
		{ 
			GulGestures *g = gul_gestures_new ();
			guint x, y;
			g_object_ref(event);
			g_object_set_data_full(G_OBJECT(g), "embed_event", event, g_object_unref);
			
			galeon_embed_event_get_mouse_coords (event, &x, &y);
			gul_gestures_set_autocancel (g, TRUE);
			gul_gestures_start (g, GTK_WIDGET (bwin), 3, x, y);
			
			g_signal_connect (g, "gesture-performed", 
					  G_CALLBACK (galeon_tab_gesture_performed_cb), tab);

			g_signal_connect (g, "cancelled", 
					  G_CALLBACK (galeon_tab_gesture_cancelled_cb), tab);
				
			g_object_unref (g);
		}
	}
	else if (button == 1
		 && (context & EMBED_CONTEXT_LINK))
	{
		GValue *value;
	
		galeon_embed_event_get_property (event, "link", &value);
		galeon_shell_new_tab (galeon_shell, window, tab, 
				      g_value_get_string (value), 0);
	}
	else if (button == 1
		&& !(context & EMBED_CONTEXT_LINK
		     || context & EMBED_CONTEXT_EMAIL_LINK
		     || context & EMBED_CONTEXT_INPUT))
	{
		gint action = eel_gconf_get_integer (CONF_MOUSE_MIDDLE_BUTTON_ACTION);
		switch (action) 
		{
		case 0: /* bookmarks context menu */
			g_warning ("Bookmarks context menu is not implemented yet\n");
			break;
		case 1: /* paste url */
			gtk_selection_convert (GTK_WIDGET (window), 
					       GDK_SELECTION_PRIMARY,
					       GDK_SELECTION_TYPE_STRING,
					       GDK_CURRENT_TIME);
			break;
		case 2: /* go back */
			galeon_embed_go_back (embed);
			break;
		case 3: /* gestures */
		{
			GulGestures *g = gul_gestures_new ();
			guint x, y;
			g_object_set_data_full(G_OBJECT(g), "embed_event", event, g_object_unref);
			g_object_ref(event);
			galeon_embed_event_get_mouse_coords (event, &x, &y);
			gul_gestures_start (g, GTK_WIDGET (bwin), 3, x, y);
			
			g_signal_connect (g, "gesture-performed", 
					  G_CALLBACK (galeon_tab_gesture_performed_cb), tab);
			
			g_object_unref (g);
		}
		break;
		case 4: /* auto scroll */
		{
			GaleonEmbedAutoscroller *as = galeon_embed_autoscroller_new ();
			guint x, y;
			galeon_embed_event_get_mouse_coords (event, &x, &y);
			galeon_embed_autoscroller_set_embed (as, embed);
			galeon_embed_autoscroller_start_scroll (as, GTK_WIDGET (bwin), x, y);
			g_object_unref (as);
		}
		break;
		case 5: /* manual scroll */
		{
			GaleonEmbedManualScroller *ms = galeon_embed_manual_scroller_new ();
			guint x, y;
			galeon_embed_event_get_mouse_coords (event, &x, &y);
			galeon_embed_manual_scroller_set_embed (ms, embed);
			galeon_embed_manual_scroller_start_scroll (ms, GTK_WIDGET (bwin), x, y);
			g_object_unref (ms);
		}
		break;
		default:
			g_warning ("Unexpected pref value for " CONF_MOUSE_MIDDLE_BUTTON_ACTION "\n");
			break;
		}
	}
	
	return FALSE;
}

static void
galeon_tab_security_change_cb (GaleonEmbed *embed, EmbedSecurityLevel level,
			       GaleonTab *tab)
{
	CHECK_EMBED_TAB_WINDOW (embed, tab, tab->priv->window);
	if (!tab->priv->is_active) return;
	
	galeon_window_update_control (tab->priv->window,
				      StatusbarSecurityControl);
}

TabLoadStatus 
galeon_tab_get_load_status (GaleonTab *tab)
{
	return tab->priv->load_status;	
}

int
galeon_tab_get_load_percent (GaleonTab *tab)
{
	return tab->priv->load_percent;
}

const char *
galeon_tab_get_status_message (GaleonTab *tab)
{
	if (tab->priv->status_message)
	{
		return tab->priv->status_message;
	}
	else
	{
		return " ";
	}
}

const char *
galeon_tab_get_title (GaleonTab *tab)
{
	if (tab->priv->title && 
	    g_utf8_strlen(tab->priv->title, -1))
	{
		return tab->priv->title;
	}
	else
	{
		return _("Untitled");
	}
}

const char *
galeon_tab_get_location (GaleonTab *tab)
{
	return tab->priv->location;
}

void 
galeon_tab_set_location (GaleonTab *tab,
			 char *location)
{
	if (tab->priv->location) g_free (tab->priv->location);
	tab->priv->location = location;
}

void 
galeon_tab_update_control (GaleonTab *tab,
			   TabControlID id)
{
	switch (id)
	{
		case TAB_CONTROL_TITLE:
			galeon_tab_set_title (tab, tab->priv->title);
			break;
	}
}

static void
galeon_tab_update_gestures (void)
{
	/* this should read a pref, parse it, setup a listener, etc */
	/* for now, just hardcode some */

	int i;
	static const struct 
	{
		gchar *sequence;
		gchar *action;
	} default_gestures[] = {
                /* menu */
                { "5"           , "menu"          },

		/* down */
                { "258"         , "new_tab"       },
		
                /* up */
                { "852"         , "new_window"    },
		
                /* up-down */
                { "85258"       , "reload"        },
                { "8525"        , "reload"        },
                { "5258"        , "reload"        },

                /* up-down-up */
                { "8525852"     , "reload_bypass" },
                { "852585"      , "reload_bypass" },
                { "85252"       , "reload_bypass" },
                { "85852"       , "reload_bypass" },
                { "525852"      , "reload_bypass" },
                { "52585"       , "reload_bypass" },

                /* up-left-down */
                { "9632147"     , "homepage"      },
                { "963214"      , "homepage"      },
                { "632147"      , "homepage"      },
                { "962147"      , "homepage"      },
                { "963147"      , "homepage"      },

                /* down-up */
                { "25852"       , "clone_window"  },
                { "2585"        , "clone_window"  },
                { "5852"        , "clone_window"  },

                /* down-up-down */
                { "2585258"     , "clone_tab"     },
                { "258525"      , "clone_tab"     },
                { "25858"       , "clone_tab"     },
                { "25258"       , "clone_tab"     },
                { "585258"      , "clone_tab"     },
                { "58525"       , "clone_tab"     },

                /* up-left-up */
                { "96541"       , "up"            },
                { "9651"        , "up"            },
                { "9541"        , "up"            },

                /* right-left-right */
                { "4565456"     , "close"         },
                { "456545"      , "close"         },
                { "45656"       , "close"         },
                { "45456"       , "close"         },
                { "565456"      , "close"         },
                { "56545"       , "close"         },

                /* down-right */
                { "14789"       , "close"         },
                { "1489"        , "close"         },

                /* left */
                { "654"         , "back"          },

                /* right */
                { "456"         , "forward"       },

                /* down-left */
                { "36987"       , "fullscreen"    },
                { "3687"        , "fullscreen"    },

                /* up-right */
                { "74123"       , "next_tab"      },
                { "7423"        , "next_tab"      },

                /* up-left */
                { "96321"       , "prev_tab"      },
                { "9621"        , "prev_tab"      },

                /* s */
                { "321456987"   , "view_source"   },
                { "21456987"    , "view_source"   },
                { "32145698"    , "view_source"   },
                { "3256987"     , "view_source"   },
                { "3214587"     , "view_source"   },
                { "32145987"    , "view_source"   },
                { "32156987"    , "view_source"   },

                /* left-up */
                { "98741"       , "stop"          },
                { "9841"        , "stop"          },

                /* left-down */
                { "32147"       , "prev_link"     },
                { "3247"        , "prev_link"     },
		
                /* right-down */
                { "12369"       , "next_link"     },
                { "1269"        , "next_link"     },

                /* up-right-up */
                { "74563"       , "contents_link" },
                { "7453"        , "contents_link" },
                { "7563"        , "contents_link" },

		{ NULL		, NULL 		  }

	};
	
	if (gestures_actions)
	{
		g_hash_table_destroy (gestures_actions);
	}

	gestures_actions = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

	for (i = 0; default_gestures[i].sequence; ++i)
	{
		g_hash_table_insert (gestures_actions, 
				     g_strdup (default_gestures[i].sequence), 
				     g_strdup (default_gestures[i].action));
	}
}

static void
galeon_tab_gesture_performed_cb (GulGestures *g, const gchar *sequence, GaleonTab *tab)
{
	const gchar *action;
	
	DEBUG_MSG (("gesture: %s\n", sequence));
	
	if (!gestures_actions)
	{
		galeon_tab_update_gestures ();
	}
	
	action = g_hash_table_lookup (gestures_actions, sequence);
	
	if (!action) 
	{
		return;
	}
	
	DEBUG_MSG (("action: %s\n", action));
	
	if (!strcmp (action, "menu"))
	{
		GaleonEmbedEvent *event = g_object_get_data(G_OBJECT(g), "embed_event");
		GaleonWindow *window;
		GaleonEmbed *embed;
		GaleonEmbedPopup *popup;
		
		window = galeon_tab_get_window (tab);
		embed = galeon_tab_get_embed (tab);
      
		popup = GALEON_EMBED_POPUP (galeon_window_get_popup_factory (window));
		galeon_embed_popup_set_embed(popup, embed);
		galeon_embed_popup_set_event (popup, event);
		galeon_embed_popup_show (popup, embed);

	}
	else if (!strcmp (action, "new_tab"))
	{
		GaleonWindow *window = galeon_tab_get_window (tab);
		
		galeon_shell_new_tab (galeon_shell, window, tab, NULL, 
				      GALEON_NEW_TAB_HOMEPAGE |
				      GALEON_NEW_TAB_IN_EXISTING_WINDOW |
				      GALEON_NEW_TAB_JUMP);
	}
	else if (!strcmp (action, "new_window"))
	{
		GaleonWindow *window = galeon_tab_get_window (tab);
		
		galeon_shell_new_tab (galeon_shell, window, tab, NULL,
				      GALEON_NEW_TAB_HOMEPAGE |
				      GALEON_NEW_TAB_IN_NEW_WINDOW |
				      GALEON_NEW_TAB_JUMP);
	}
	else if (!strcmp (action, "reload"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "reload_bypass"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "homepage"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "clone_window"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "clone_tab"))
	{
		GaleonWindow *window = galeon_tab_get_window (tab);

		galeon_shell_new_tab (galeon_shell, window, tab, NULL,
				      GALEON_NEW_TAB_IN_EXISTING_WINDOW |
				      GALEON_NEW_TAB_JUMP |
				      GALEON_NEW_TAB_IS_A_COPY);
	}
	else if (!strcmp (action, "up"))
	{
		galeon_embed_go_up (galeon_tab_get_embed (tab));
	}
	else if (!strcmp (action, "close"))
	{
		galeon_window_remove_tab (galeon_tab_get_window (tab), tab);
	}
	else if (!strcmp (action, "back"))
	{
		galeon_embed_go_back (galeon_tab_get_embed (tab));
	}
	else if (!strcmp (action, "forward"))
	{
		galeon_embed_go_forward (galeon_tab_get_embed (tab));
	}
	else if (!strcmp (action, "fullscreen"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "new_tab"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "prev_tab"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "view_source"))
	{
		GaleonWindow *window = galeon_tab_get_window (tab);

		galeon_shell_new_tab (galeon_shell, window, tab, NULL,
				      GALEON_NEW_TAB_IN_EXISTING_WINDOW |
				      GALEON_NEW_TAB_JUMP |
				      GALEON_NEW_TAB_VIEW_SOURCE);
	}
	else if (!strcmp (action, "stop"))
	{
		galeon_embed_stop_load (galeon_tab_get_embed (tab));
	}
	else if (!strcmp (action, "prev_link"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "next_link"))
	{
		NOT_IMPLEMENTED;
	}
	else if (!strcmp (action, "contents_link"))
	{
		NOT_IMPLEMENTED;
	}
	else
	{
		/* unrecognized */
	}
}

static void
galeon_tab_gesture_cancelled_cb (GulGestures *ges, GaleonTab *tab)
{
	GaleonEmbedEvent *event = g_object_get_data (G_OBJECT (ges), "embed_event");
	GaleonWindow *window;
	GaleonEmbed *embed;
	GaleonEmbedPopup *popup;
	
	window = galeon_tab_get_window (tab);
	embed = galeon_tab_get_embed (tab);
	
	popup = GALEON_EMBED_POPUP (galeon_window_get_popup_factory (window));
	galeon_embed_popup_set_embed(popup, embed);
	galeon_embed_popup_set_event (popup, event);
	galeon_embed_popup_show (popup, embed);
}	
