import { GlIcon } from '@gitlab/ui';
import { shallowMount } from '@vue/test-utils';
import { extendedWrapper } from 'helpers/vue_test_utils_helper';
import CommitComponent from '~/vue_shared/components/commit.vue';
import UserAvatarLink from '~/vue_shared/components/user_avatar/user_avatar_link.vue';

describe('Commit component', () => {
  let props;
  let wrapper;

  const findIcon = (name) => {
    const icons = wrapper.findAllComponents(GlIcon).filter((c) => c.attributes('name') === name);
    return icons.length ? icons.at(0) : icons;
  };

  const findUserAvatar = () => wrapper.findComponent(UserAvatarLink);
  const findRefName = () => wrapper.findByTestId('ref-name');

  const createComponent = (propsData) => {
    wrapper = extendedWrapper(
      shallowMount(CommitComponent, {
        propsData,
      }),
    );
  };

  it('should render a fork icon if it does not represent a tag', () => {
    createComponent({
      tag: false,
      commitRef: {
        name: 'main',
        ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
      },
      commitUrl:
        'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
      shortSha: 'b7836edd',
      title: 'Commit message',
      author: {
        avatar_url: 'https://gitlab.com/uploads/-/system/user/avatar/300478/avatar.png',
        web_url: 'https://gitlab.com/jschatz1',
        path: '/jschatz1',
        username: 'jschatz1',
      },
    });

    expect(wrapper.find('.icon-container').findComponent(GlIcon).exists()).toBe(true);
  });

  describe('Given all the props', () => {
    beforeEach(() => {
      props = {
        tag: true,
        commitRef: {
          name: 'main',
          ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
        commitUrl:
          'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
        shortSha: 'b7836edd',
        title: 'Commit message',
        author: {
          avatar_url: 'https://gitlab.com/uploads/-/system/user/avatar/300478/avatar.png',
          web_url: 'https://gitlab.com/jschatz1',
          path: '/jschatz1',
          username: 'jschatz1',
        },
      };
      createComponent(props);
    });

    it('should render a tag icon if it represents a tag', () => {
      expect(findIcon('tag').exists()).toBe(true);
    });

    it('should render a link to the ref url', () => {
      expect(wrapper.find('.ref-name').attributes('href')).toBe(props.commitRef.ref_url);
    });

    it('should render the ref name', () => {
      expect(wrapper.find('.ref-name').text()).toContain(props.commitRef.name);
    });

    it('should render the commit short sha with a link to the commit url', () => {
      expect(wrapper.find('.commit-sha').attributes('href')).toEqual(props.commitUrl);

      expect(wrapper.find('.commit-sha').text()).toContain(props.shortSha);
    });

    it('should render icon for commit', () => {
      expect(findIcon('commit').exists()).toBe(true);
    });

    describe('Given commit title and author props', () => {
      it('should render a link to the author profile', () => {
        const userAvatar = findUserAvatar();

        expect(userAvatar.props('linkHref')).toBe(props.author.path);
      });

      it('Should render the author avatar with title and alt attributes', () => {
        const userAvatar = findUserAvatar();

        expect(userAvatar.exists()).toBe(true);

        expect(userAvatar.props('imgAlt')).toBe(`${props.author.username}'s avatar`);
      });
    });

    it('should render the commit title', () => {
      expect(wrapper.find('.commit-row-message').attributes('href')).toEqual(props.commitUrl);

      expect(wrapper.find('.commit-row-message').text()).toContain(props.title);
    });
  });

  describe('When commit title is not provided', () => {
    it('should render default message', () => {
      props = {
        tag: false,
        commitRef: {
          name: 'main',
          ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
        commitUrl:
          'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
        shortSha: 'b7836edd',
        title: null,
        author: {},
      };

      createComponent(props);

      expect(wrapper.find('.commit-title span').text()).toContain(
        "Can't find HEAD commit for this branch",
      );
    });
  });

  describe('When commit ref is provided, but merge ref is not', () => {
    it('should render the commit ref', () => {
      props = {
        tag: false,
        commitRef: {
          name: 'main',
          ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
        commitUrl:
          'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
        shortSha: 'b7836edd',
        title: null,
        author: {},
      };

      createComponent(props);
      const refEl = wrapper.find('.ref-name');

      expect(refEl.text()).toContain('main');

      expect(refEl.attributes('href')).toBe(props.commitRef.ref_url);

      expect(findIcon('branch').exists()).toBe(true);
    });
  });

  describe('When both commit and merge ref are provided', () => {
    it('should render the merge ref', () => {
      props = {
        tag: false,
        commitRef: {
          name: 'main',
          ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
        commitUrl:
          'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
        mergeRequestRef: {
          iid: 1234,
          path: 'https://example.com/path/to/mr',
          title: 'Test MR',
        },
        shortSha: 'b7836edd',
        title: null,
        author: {},
      };

      createComponent(props);
      const refEl = wrapper.find('.ref-name');

      expect(refEl.text()).toContain('1234');

      expect(refEl.attributes('href')).toBe(props.mergeRequestRef.path);

      expect(findIcon('git-merge').exists()).toBe(true);
    });
  });

  describe('When showRefInfo === false', () => {
    it('should not render any ref info', () => {
      props = {
        tag: false,
        commitRef: {
          name: 'main',
          ref_url: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
        commitUrl:
          'https://gitlab.com/gitlab-org/gitlab-foss/commit/b7836eddf62d663c665769e1b0960197fd215067',
        mergeRequestRef: {
          iid: 1234,
          path: '/path/to/mr',
          title: 'Test MR',
        },
        shortSha: 'b7836edd',
        title: null,
        author: {},
        showRefInfo: false,
      };

      createComponent(props);

      expect(wrapper.find('.ref-name').exists()).toBe(false);
    });
  });

  describe('When commitRef has a path property instead of ref_url property', () => {
    it('should render path as href attribute', () => {
      props = {
        commitRef: {
          name: 'main',
          path: 'http://localhost/namespace2/gitlabhq/tree/main',
        },
      };

      createComponent(props);

      expect(findRefName().exists()).toBe(true);
      expect(findRefName().attributes('href')).toBe(props.commitRef.path);
    });
  });
});
