## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----built--------------------------------------------------------------------
message(paste0('Datetime: ',Sys.Date(),':',Sys.time()))

## ----setup, message=FALSE-----------------------------------------------------
library(repfun)
library(dplyr)
library(DT)
library(kableExtra)

## ----envir--------------------------------------------------------------------
#====================================
# Set up the reporting environment.
#====================================
tmpdr <- tempdir()
setup <- function(tlfid){
  repfun::rs_setup(
    D_DATADATE=Sys.Date(),
    D_DSPLYNUM=tlfid,
    D_FOOT1='1.) Baseline is defined as the last non-missing value prior to the first dose.',
    D_FOOT2='2.) Treatment duration is calculated as: (last dose date - first dose date) + 28 days.',
    D_FOOT3='3.) Age is collected in years.',
    D_KEEPPOPVARS=c('STUDYID','USUBJID','SAFFL'),
    D_OUTFILE=paste0(tmpdr,"/t_ru_list_",tlfid,".rtf"),
    D_PGMPTH="~/vignettes/Dem-Tbls.Rmd",
    D_STUDYID='ABCXYZPDQ',
    D_POP="SAFFL",
    D_POPDATA=repfun::adsl %>% dplyr::filter(SAFFL =='Y') %>%
      dplyr::mutate(TRT01AN=ifelse(TRT01A=='Placebo',1,ifelse(TRT01A=='Xanomeline Low Dose',2,3))) %>%
      repfun::ru_labels(varlabels=list('TRT01AN'='Actual Treatment for Period 01 (n)')),
    D_POPLBL="Safety",
    D_SUBJID=c("STUDYID","USUBJID"),
    D_TITLE1=paste0('Table ',tlfid,': Summary of Demographic and Baseline Characteristics'),
    R_DDDATA=paste0(tmpdr,'/t_ru_list_',tlfid,'.rds'),
    R_ADAMDATA="../data")
}
setup(15)
tbl15dd <- paste0(tmpdr,"/t_ru_list_15.rds")

## ----makelists----------------------------------------------------------------
fmtlist <- list('SEXS'=list('START'=list('M','F'),
                            'LABEL'=c('Male','Female')),
                'SEX1'=list('START'=list('Male','Female','Unknown'),
                            'LABEL'=c(1,2,3)),
                'AGE1'=list('START'=list('18-64','>64'),
                            'LABEL'=c(1,2)),
                'RAC1'=list('START'=list('AMERICAN INDIAN OR ALASKA NATIVE','BLACK OR AFRICAN AMERICAN','WHITE'),
                            'LABEL'=c(1,2,3)),
                'ETH1'=list('START'=list('HISPANIC OR LATINO','NOT HISPANIC OR LATINO'),
                            'LABEL'=c(1,2)))

## ----codedecode---------------------------------------------------------------
G_POPDATA2 <- repfun::ru_fillcodedcode(
              repfun:::rfenv$G_POPDATA,
              codedecodevarpairs=c("SEX", "SEXC", "AGEGR1", "AGEGR1N", "RACE", "RACEN", "ETHNIC", "ETHNICN"),
              varcodelistpairs=c("SEX", "SEXS", "AGEGR1", "AGE1", "RACE", "RAC1", "ETHNIC", "ETH1"),
              codelistnames=fmtlist,
              completetypes=FALSE) %>% 
  repfun::ru_labels(varlabels=list('SEXC'='Gender','AGE'='Age (years)'))

G_POPDATA <- repfun::ru_fillcodedcode(
             G_POPDATA2,
             codedecodevarpairs=c("SEXC", "SEXCN"),
             varcodelistpairs=c("SEXC", "SEX1"),
             codelistnames=fmtlist,
             completetypes=FALSE) 

## ----sumstats-----------------------------------------------------------------
dflst <- list()
df <- data.frame()

i <- 1
for(v in c('AGEGR1','SEXC','AGE','RACE','ETHNIC','TRTDURD')){
  lbl <- attr(G_POPDATA[[v]],'label')
  if (paste0(v,'N') %in% names(G_POPDATA)){
    #==========================================#
    # Generate counts/percents for categories. #
    #==========================================#
    dflst[[v]] <- repfun::ru_freq(
        G_POPDATA, dsetindenom=G_POPDATA, countdistinctvars=c("STUDYID", "USUBJID"),
        groupbyvarsnumer=c("STUDYID", "TRT01AN", paste0(v,'N')),
        anyeventvars = NULL, anyeventvalues = NULL, groupminmaxvar=NULL,
        totalforvar=c("TRT01AN"), totalid = 99, totaldecode = 'Total',
        groupbyvarsdenom=c("STUDYID", "TRT01AN"), resultstyle="NUMERPCT",
        codedecodevarpairs=c("TRT01AN", "TRT01A", paste0(v,'N'), v),
        resultpctdps=0) %>%
      dplyr::mutate(tt_avid=i, tt_avnm=lbl) %>%
      dplyr::rename(tt_svid=as.name(paste0(v,'N')), tt_svnm=as.name(v)) %>%
      dplyr::arrange(tt_avid,tt_svid) %>%
      dplyr::select(tt_avid,tt_avnm,tt_svid,tt_svnm,TRT01AN,TRT01A,tt_result,DENOMCNT) %>%
      dplyr::mutate(tt_avnm=paste0(tt_avnm,', n (%)'))
    df <- repfun::ru_setdata(df,dflst[[v]]) %>% dplyr::select(-DENOMCNT) %>% dplyr::select(-Row.names)
  } else {
    lbl <- attr(G_POPDATA[[v]],'label')
    #============================================#
    # Generate summary stats for non-categories. #
    #============================================#
    dflst[[v]] <- repfun::ru_sumstats(
        G_POPDATA, v, groupbyvars=c("STUDYID", "TRT01AN"),
        codedecodevarpairs=c("TRT01AN", "TRT01A"),
        totalforvar="TRT01AN", totalid=99, totaldecode="Total",
        statsinrowsyn = "Y",
        statslist=c("n", "mean", "median", "sd", "min", "max")) %>%
      dplyr::mutate(tt_avid=tt_avid + i - 1, tt_avnm=lbl) %>%
      dplyr::arrange(tt_avid,tt_svid) %>%
      dplyr::select(tt_avid,tt_avnm,tt_svid,tt_svnm,TRT01AN,TRT01A,tt_result)
    df <- repfun::ru_setdata(df,dflst[[v]]) %>% dplyr::select(-Row.names)
  }

  #===================================================#
  # Make alignment consistent across all data frames. #
  #===================================================#
  df <- df %>%
    dplyr::mutate(tt_result=paste0(strrep(' ',
                                   3-nchar(gsub(' ','',sub('(\\(|\\.).*$','',tt_result)))),
                                   trimws(tt_result)))
  i <- i+1
}

## ----addbign------------------------------------------------------------------
rptdf <- repfun::ru_addbignvar(df,
                       G_POPDATA,
                       groupbyvars=c("TRT01AN", "TRT01A"),
                       totalforvar=c("TRT01AN"),
                       totalid = 99,
                       varcodelistpairs=c(""),
                       codedecodevarpairs=c("TRT01AN", "TRT01A"),
                       splitchar='\\line') %>% dplyr::select(-tt_bnnm) %>%
         dplyr::arrange(tt_avid,tt_svid)

## ----maket15------------------------------------------------------------------
repfun::ru_list(dsetin=rptdf,
        columns=c('tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        widths=c(5,5.1,1.65,1.9,1.9,1.65),
        skipvars=c('tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        ordervars=c('tt_avid','tt_svid'),
        noprintvars=c('tt_avid','tt_svid'),
        denormyn='Y',
        varsToDenorm=c('tt_result'),
        groupByVars=c('tt_avid','tt_avnm','tt_svid','tt_svnm'),
        acrossVar="TRT01AN",
        acrossVarLabel="TRT01A",
        acrossColVarPrefix='tt_',
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM),
        labels=list('tt_avnm'='Variable','tt_svnm'='Category/Statistic'))

## ----dddata15, results='asis'-------------------------------------------------
t15 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(tt_svnm != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t15,function(x){attr(x,"label")})
names(t15) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t15, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket16------------------------------------------------------------------
setup(16)
repfun::ru_list(dsetin=rptdf,
        columns=c('tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        widths=c(8,1.65,1.9,1.9,1.65),
        skipvars=c('tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        ordervars=c('tt_avid','tt_svid'),
        sharecolvars=c('tt_avnm','tt_svnm'),
        noprintvars=c('tt_avid','tt_svid'),
        denormyn='Y',
        varsToDenorm=c('tt_result'),
        groupByVars=c('tt_avid','tt_avnm','tt_svid','tt_svnm'),
        acrossVar="TRT01AN",
        acrossVarLabel="TRT01A",
        acrossColVarPrefix='tt_',
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM),
        labels=list('tt_avnm'='Variable','tt_svnm'='Category/Statistic'))

## ----dddata16, results='asis'-------------------------------------------------
t16 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t16,function(x){attr(x,"label")})
names(t16) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t16, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket17------------------------------------------------------------------
setup(17)

df <- readRDS(file=tbl15dd) %>% dplyr::select(-PAGEVAR) %>% dplyr::filter(!(tt_svnm == ''))
dfpg <- rbind(df %>% dplyr::mutate(COUNTRY='CANADA'),df %>% dplyr::mutate(COUNTRY='USA')) %>% dplyr::select(COUNTRY, everything())

dfpg %>% dplyr::mutate(rnum=row_number()) %>% dplyr::group_by(COUNTRY,tt_avnm) %>% dplyr::filter (row_number()==1) %>% dplyr::ungroup() %>%
  dplyr::distinct(COUNTRY,tt_avnm,rnum) -> dford

dfpg %>% left_join(dford,by=c('COUNTRY','tt_avnm')) %>% repfun::ru_labels(varlabels=list('COUNTRY'='Country')) -> dfpg

repfun::ru_list(dsetin=dfpg,
        columns=c('COUNTRY','tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        sharecolvars=c('COUNTRY','tt_avnm','tt_svnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        widths=c(7,1.65,1.9,1.9,1.65),
        skipvars=c('COUNTRY','tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        lpp=27,
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata17, results='asis'-------------------------------------------------
t17 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t17,function(x){attr(x,"label")})
names(t17) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t17, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket18------------------------------------------------------------------
setup(18)

dfpg %>% repfun::ru_labels(list('COUNTRY'='',
                        'tt_avnm'=paste0("\\u160* ",attr(dfpg$tt_avnm,'label')),
                        'tt_02'=gsub('Xanomeline ','',attr(dfpg$tt_02,'label')),
                        'tt_03'=gsub('Xanomeline ','',attr(dfpg$tt_03,'label'))))-> dfpg

repfun::ru_list(dsetin=dfpg,
        columns=c('COUNTRY','tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        sharecolvars=c('COUNTRY','tt_avnm','tt_svnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        widths=c(7,1.65,1.9,1.9,1.65),
        skipvars=c('COUNTRY','tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        lpp=27,
        toprow='',
        spanlbls=paste0("Country|Xanomeline| "),
        spanwidths=c(8.65,3.8,1.65),
        spanjust=c('l','c','c'),
        spanbbord=c('','single',''),
        spantbord=c('single','single','single'),
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata18, results='asis'-------------------------------------------------
t18 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t18,function(x){attr(x,"label")})
names(t18) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t18, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket19------------------------------------------------------------------
setup(19)

repfun::ru_list(dsetin=dfpg,
        columns=c('COUNTRY','tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        sharecolvars=c('COUNTRY','tt_avnm','tt_svnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        widths=c(7,1.65,1.9,1.9,1.65),
        skipvars=c('COUNTRY','tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        lpp=27,
        spanlbls=paste0("Country| |Xanomeline| "),
        toprow='',
        spanwidths=c(7,1.65,3.8,1.65),
        spanjust=c('l','c','c','c'),
        spanbbord=c('','','single',''),
        spantbord=c('','','',''),
        span2lbls=paste0(" |All Active Treatments"),
        span2widths=c(7,7.1),
        span2just=c('c','c'),
        span2bbord=c('','single'),
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata19, results='asis'-------------------------------------------------
t19 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t19,function(x){attr(x,"label")})
names(t19) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t19, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket20------------------------------------------------------------------
setup(20)

df <- readRDS(file=tbl15dd) %>% dplyr::select(-PAGEVAR) %>% dplyr::filter(!(tt_svnm == ''))
dfpg <- rbind(df %>% dplyr::mutate(COUNTRY='CANADA'),df %>% dplyr::mutate(COUNTRY='USA')) %>% dplyr::select(COUNTRY, everything())

dfpg %>% dplyr::mutate(rnum=row_number()) %>% dplyr::group_by(COUNTRY,tt_avnm) %>% dplyr::filter (row_number()==1) %>% dplyr::ungroup() %>%
  dplyr::distinct(COUNTRY,tt_avnm,rnum) -> dford

dfpg <- dfpg %>% left_join(dford,by=c('COUNTRY','tt_avnm')) %>% repfun::ru_labels(varlabels=list('COUNTRY'='Country')) %>%
  repfun::ru_labels(list('COUNTRY'='Country'))

mytbls <- list('Stratum: Normal'=dfpg[0,], 'Stratum: Abnormal'=dfpg)
repfun::ru_list(dsetin=mytbls,
        columns=c('COUNTRY','tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        sharecolvars=c('COUNTRY','tt_avnm','tt_svnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        widths=c(7,1.65,1.9,1.9,1.65),
        skipvars=c('COUNTRY','tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        lpp = 28,
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata20, results='asis'-------------------------------------------------
t20 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t20,function(x){attr(x,"label")})
names(t20) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t20, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket21------------------------------------------------------------------
setup(21)

df <- readRDS(file=tbl15dd) %>% dplyr::select(-PAGEVAR) %>% dplyr::filter(!(tt_svnm == ''))
dfpg <- rbind(df %>% dplyr::mutate(MYPAGEVAR=1),df %>% dplyr::mutate(MYPAGEVAR=2))
dfpg %>% dplyr::mutate(rnum=row_number()) %>% dplyr::group_by(tt_avnm) %>% dplyr::filter (row_number()==1) %>% dplyr::ungroup() %>% dplyr::distinct(tt_avnm,rnum) -> dford
dfpg <- dfpg %>% left_join(dford,by='tt_avnm') %>% repfun::ru_labels(varlabels=list('MYPAGEVAR'='My Page Variable'))

repfun::ru_list(dsetin=dfpg,
        columns=c('tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        widths=c(5,5,1.65,1.9,1.9,1.65),
        nowidowvar='tt_avnm',
        skipvars=c('tt_avnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        pagevars=c('MYPAGEVAR'),
        computebeforepagelines="paste0('This is page value: ',MYPAGEVAR)",
        lpp=28,
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata21, results='asis'-------------------------------------------------
t21 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", tt_svnm) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t21,function(x){attr(x,"label")})
names(t21) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t21, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket22------------------------------------------------------------------
setup(22)

repfun::ru_list(dsetin=dfpg,
        columns=c('tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        widths=c(8,1.65,1.9,1.9,1.65),
        nowidowvar='tt_avnm',
        skipvars=c('tt_avnm'),
        sharecolvars=c('tt_avnm','tt_svnm'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        pagevars=c('MYPAGEVAR'),
        computebeforepagelines="paste0('This is page value: ',MYPAGEVAR)",
        lpp=26,
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata22, results='asis'-------------------------------------------------
t22 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", STACKVAR) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t22,function(x){attr(x,"label")})
names(t22) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t22, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----maket23------------------------------------------------------------------
setup(23)

df <- readRDS(file=tbl15dd) %>% 
  dplyr::select(-PAGEVAR) %>% dplyr::filter(!(tt_svnm == ''))
dfpg <- rbind(df %>% dplyr::mutate(MYPAGEVAR1=1,MYPAGEVAR2=1),
              df %>% dplyr::mutate(MYPAGEVAR1=1,MYPAGEVAR2=2),
              df %>% dplyr::mutate(MYPAGEVAR1=2,MYPAGEVAR2=3),
              df %>% dplyr::mutate(MYPAGEVAR1=2,MYPAGEVAR2=4))
dford <- dfpg %>% dplyr::mutate(rnum=row_number()) %>% 
  dplyr::group_by(MYPAGEVAR1,MYPAGEVAR2,tt_avnm) %>% 
  dplyr::filter (row_number()==1) %>% dplyr::ungroup() %>% 
  dplyr::distinct(MYPAGEVAR1,MYPAGEVAR2,tt_avnm,rnum)
dfpg <- dfpg %>% left_join(dford,by=c('tt_avnm', 'MYPAGEVAR1', 'MYPAGEVAR2')) 

lblist <- list('tt_avnm'='New Variable Label', 'tt_svnm'='New Category Label')
repfun::ru_list(dsetin=dfpg,
        columns=c('tt_avnm','tt_svnm','tt_01','tt_02','tt_03','tt_99'),
        nowidowvar='tt_avnm',
        widths=c(6,4,2,2,2,2),
        skipvars=c('tt_avnm'),
        centrevars=c('tt_01','tt_02','tt_03','tt_99'),
        ordervars=c('rnum'),
        noprintvars=c('rnum'),
        byvars=c('MYPAGEVAR2'),
        pagevars=c('MYPAGEVAR1'),
        computebeforepagelines="paste0('This is page value 1: ',MYPAGEVAR1,' and this is page value 2: ',MYPAGEVAR2)",
        lpp=28,
        labels=lblist,
        dddatasetlabel=paste0('DD Dataframe for DEMO Table ',repfun:::rfenv$G_DSPLYNUM))

## ----dddata23, results='asis'-------------------------------------------------
t23 <- readRDS(repfun:::rfenv$G_DDDATA) %>% dplyr::filter(gsub("\\s+", "", tt_svnm) != '') %>% dplyr::filter(row_number()<=10)
lbls <- sapply(t23,function(x){attr(x,"label")})
names(t23) <- paste(names(lbls),lbls,sep=": ")
datatable(
  t23, extensions = 'FixedColumns',
  options = list(
  dom = 't',
  pageLength=1000,
  scrollY = 300,
  scrollX = TRUE,
  scrollCollapse = TRUE
))

## ----cleanup------------------------------------------------------------------
unlink(tmpdr, recursive = TRUE)

