"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.visualReportSchema = exports.triggerSchema = exports.scheduleSchema = exports.reportSchema = exports.reportParamsSchema = exports.reportDefinitionSchema = exports.opensearchDashboardsUserSchema = exports.intervalSchema = exports.deliverySchema = exports.dataReportSchema = exports.cronSchema = exports.channelSchema = void 0;
var _configSchema = require("@osd/config-schema");
var _validationHelper = require("../utils/validationHelper");
var _cronValidator = require("cron-validator");
var _constants = require("../routes/utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const dataReportSchema = exports.dataReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  saved_search_id: _configSchema.schema.string(),
  //ISO duration format. 'PT10M' means 10 min
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  }),
  timeFrom: _configSchema.schema.maybe(_configSchema.schema.string({
    validate(value) {
      if (isNaN(Date.parse(value))) {
        return `invalid timeFrom: ${value}`;
      }
    }
  })),
  timeTo: _configSchema.schema.maybe(_configSchema.schema.string({
    validate(value) {
      if (isNaN(Date.parse(value))) {
        return `invalid timeTo: ${value}`;
      }
    }
  })),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.csv), _configSchema.schema.literal(_constants.FORMAT.xlsx)]),
  limit: _configSchema.schema.number({
    defaultValue: _constants.DEFAULT_MAX_SIZE,
    min: 0
  }),
  excel: _configSchema.schema.boolean({
    defaultValue: true
  })
});
const visualReportSchema = exports.visualReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  window_width: _configSchema.schema.number({
    defaultValue: 1600,
    min: 0
  }),
  window_height: _configSchema.schema.number({
    defaultValue: 800,
    min: 0
  }),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.pdf), _configSchema.schema.literal(_constants.FORMAT.png)]),
  header: _configSchema.schema.maybe(_configSchema.schema.string()),
  footer: _configSchema.schema.maybe(_configSchema.schema.string()),
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  }),
  timeFrom: _configSchema.schema.maybe(_configSchema.schema.string({
    validate(value) {
      if (isNaN(Date.parse(value))) {
        return `invalid timeFrom: ${value}`;
      }
    }
  })),
  timeTo: _configSchema.schema.maybe(_configSchema.schema.string({
    validate(value) {
      if (isNaN(Date.parse(value))) {
        return `invalid timeTo: ${value}`;
      }
    }
  }))
});
const intervalSchema = exports.intervalSchema = _configSchema.schema.object({
  interval: _configSchema.schema.object({
    period: _configSchema.schema.number({
      min: 0
    }),
    // Refer to job scheduler SPI https://github.com/opensearch-project/job-scheduler/blob/main/spi/src/main/java/org/opensearch/jobscheduler/spi/schedule/IntervalSchedule.java
    unit: _configSchema.schema.oneOf([_configSchema.schema.literal('MINUTES'), _configSchema.schema.literal('HOURS'), _configSchema.schema.literal('DAYS'),
    // Job scheduler in reporting OpenSearch plugin always saves as following format
    _configSchema.schema.literal('Minutes'), _configSchema.schema.literal('Hours'), _configSchema.schema.literal('Days')]),
    // timestamp
    start_time: _configSchema.schema.number()
  })
});
const cronSchema = exports.cronSchema = _configSchema.schema.object({
  cron: _configSchema.schema.object({
    expression: _configSchema.schema.string({
      validate(value) {
        if (!(0, _cronValidator.isValidCron)(value)) {
          return `invalid cron expression: ${value}`;
        }
      }
    }),
    //TODO: add more validation once we add full support of timezone
    timezone: _configSchema.schema.string()
  })
});
const scheduleSchema = exports.scheduleSchema = _configSchema.schema.object({
  schedule_type: _configSchema.schema.oneOf([
  /*
  TODO: Future Date option will be added in the future.
  Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.SCHEDULE_TYPE.recurring), _configSchema.schema.literal(_constants.SCHEDULE_TYPE.cron)]),
  schedule: _configSchema.schema.conditional(_configSchema.schema.siblingRef('schedule_type'), _constants.SCHEDULE_TYPE.recurring, intervalSchema, cronSchema),
  enabled_time: _configSchema.schema.number(),
  enabled: _configSchema.schema.boolean()
});
const opensearchDashboardsUserSchema = exports.opensearchDashboardsUserSchema = _configSchema.schema.object({
  opensearch_dashboards_recipients: _configSchema.schema.arrayOf(_configSchema.schema.string())
});
const channelSchema = exports.channelSchema = _configSchema.schema.object({
  recipients: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexEmailAddress.test(value)) {
        return `invalid email address ${value}`;
      }
    }
  }), {
    minSize: 1
  }),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.maybe(_configSchema.schema.string()),
  configIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
});
const triggerSchema = exports.triggerSchema = _configSchema.schema.object({
  trigger_type: _configSchema.schema.oneOf([
  /*
    TODO: Alerting will be added in the future.
    Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.TRIGGER_TYPE.schedule), _configSchema.schema.literal(_constants.TRIGGER_TYPE.onDemand)]),
  trigger_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('trigger_type'), _constants.TRIGGER_TYPE.onDemand, _configSchema.schema.never(), scheduleSchema)
});
const deliverySchema = exports.deliverySchema = _configSchema.schema.object({
  configIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.string()
});
const reportParamsSchema = exports.reportParamsSchema = _configSchema.schema.object({
  report_name: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexReportName.test(value)) {
        return `invald report name ${value}.\nMust be non-empty, allow a-z, A-Z, 0-9, (), [], ',' - and _ and spaces`;
      }
    }
  }),
  report_source: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_TYPE.dashboard), _configSchema.schema.literal(_constants.REPORT_TYPE.visualization), _configSchema.schema.literal(_constants.REPORT_TYPE.savedSearch), _configSchema.schema.literal(_constants.REPORT_TYPE.notebook)]),
  description: _configSchema.schema.string(),
  core_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('report_source'), _constants.REPORT_TYPE.savedSearch, dataReportSchema, visualReportSchema)
});
const reportDefinitionSchema = exports.reportDefinitionSchema = _configSchema.schema.object({
  report_params: reportParamsSchema,
  delivery: deliverySchema,
  trigger: triggerSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  status: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.active), _configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.disabled)]))
});
const reportSchema = exports.reportSchema = _configSchema.schema.object({
  query_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  time_from: _configSchema.schema.number(),
  time_to: _configSchema.schema.number(),
  report_definition: reportDefinitionSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  state: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_STATE.created), _configSchema.schema.literal(_constants.REPORT_STATE.error), _configSchema.schema.literal(_constants.REPORT_STATE.pending), _configSchema.schema.literal(_constants.REPORT_STATE.shared)]))
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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