﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/Tag.h>
#include <aws/iot/model/ThingTypeProperties.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 * <p>The input for the CreateThingType operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iot-2015-05-28/CreateThingTypeRequest">AWS
 * API Reference</a></p>
 */
class CreateThingTypeRequest : public IoTRequest {
 public:
  AWS_IOT_API CreateThingTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateThingType"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the thing type.</p>
   */
  inline const Aws::String& GetThingTypeName() const { return m_thingTypeName; }
  inline bool ThingTypeNameHasBeenSet() const { return m_thingTypeNameHasBeenSet; }
  template <typename ThingTypeNameT = Aws::String>
  void SetThingTypeName(ThingTypeNameT&& value) {
    m_thingTypeNameHasBeenSet = true;
    m_thingTypeName = std::forward<ThingTypeNameT>(value);
  }
  template <typename ThingTypeNameT = Aws::String>
  CreateThingTypeRequest& WithThingTypeName(ThingTypeNameT&& value) {
    SetThingTypeName(std::forward<ThingTypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ThingTypeProperties for the thing type to create. It contains information
   * about the new thing type including a description, and a list of searchable thing
   * attribute names.</p>
   */
  inline const ThingTypeProperties& GetThingTypeProperties() const { return m_thingTypeProperties; }
  inline bool ThingTypePropertiesHasBeenSet() const { return m_thingTypePropertiesHasBeenSet; }
  template <typename ThingTypePropertiesT = ThingTypeProperties>
  void SetThingTypeProperties(ThingTypePropertiesT&& value) {
    m_thingTypePropertiesHasBeenSet = true;
    m_thingTypeProperties = std::forward<ThingTypePropertiesT>(value);
  }
  template <typename ThingTypePropertiesT = ThingTypeProperties>
  CreateThingTypeRequest& WithThingTypeProperties(ThingTypePropertiesT&& value) {
    SetThingTypeProperties(std::forward<ThingTypePropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata which can be used to manage the thing type.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateThingTypeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateThingTypeRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_thingTypeName;

  ThingTypeProperties m_thingTypeProperties;

  Aws::Vector<Tag> m_tags;
  bool m_thingTypeNameHasBeenSet = false;
  bool m_thingTypePropertiesHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
