/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.model.listing;

import generic.algorithms.CRC64;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressFactory;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.lang.ProgramArchitecture;
import ghidra.program.model.lang.Register;
import ghidra.program.model.lang.UnknownRegister;
import ghidra.program.model.listing.AutoParameterType;
import ghidra.program.model.pcode.Varnode;
import ghidra.program.util.LanguageTranslator;
import ghidra.util.exception.InvalidInputException;
import java.util.ArrayList;
import java.util.List;

public class VariableStorage
implements Comparable<VariableStorage> {
    private static final String BAD = "<BAD>";
    private static final String UNASSIGNED = "<UNASSIGNED>";
    private static final String VOID = "<VOID>";
    public static final VariableStorage BAD_STORAGE = new VariableStorage();
    public static final VariableStorage UNASSIGNED_STORAGE = new VariableStorage();
    public static final VariableStorage VOID_STORAGE = new VariableStorage();
    protected final Varnode[] varnodes;
    protected final ProgramArchitecture programArch;
    private List<Register> registers;
    private int size;
    private long hashcode;
    private String serialization;
    private static final int PRECEDENCE_MAPPED = 1;
    private static final int PRECEDENCE_UNMAPPED = 2;
    private static final int PRECEDENCE_BAD = 3;

    protected VariableStorage() {
        this.programArch = null;
        this.varnodes = null;
    }

    public VariableStorage(ProgramArchitecture programArch, Varnode ... varnodes) throws InvalidInputException {
        this.programArch = programArch;
        this.varnodes = (Varnode[])varnodes.clone();
        this.checkVarnodes();
    }

    public VariableStorage(ProgramArchitecture programArch, Register ... registers) throws InvalidInputException {
        this(programArch, VariableStorage.getVarnodeList(registers));
    }

    public VariableStorage(ProgramArchitecture programArch, int stackOffset, int size) throws InvalidInputException {
        this(programArch, new Varnode(programArch.getAddressFactory().getStackSpace().getAddress(stackOffset), size));
    }

    private static Varnode[] getVarnodeList(Register[] registers) {
        Varnode[] varnodes = new Varnode[registers.length];
        for (int i = 0; i < registers.length; ++i) {
            varnodes[i] = new Varnode(registers[i].getAddress(), registers[i].getMinimumByteSize());
        }
        return varnodes;
    }

    public VariableStorage(ProgramArchitecture programArch, List<Varnode> varnodes) throws InvalidInputException {
        this.programArch = programArch;
        this.varnodes = varnodes.toArray(new Varnode[varnodes.size()]);
        this.checkVarnodes();
    }

    public VariableStorage(ProgramArchitecture programArch, Address address, int size) throws InvalidInputException {
        this(programArch, new Varnode(address, size));
    }

    public static VariableStorage deserialize(ProgramArchitecture programArch, String serialization) throws InvalidInputException {
        if (serialization == null || UNASSIGNED.equals(serialization)) {
            return UNASSIGNED_STORAGE;
        }
        if (VOID.equals(serialization)) {
            return VOID_STORAGE;
        }
        if (BAD.equals(serialization)) {
            return BAD_STORAGE;
        }
        List<Varnode> varnodes = VariableStorage.getVarnodes(programArch.getAddressFactory(), serialization);
        if (varnodes == null) {
            return BAD_STORAGE;
        }
        return new VariableStorage(programArch, varnodes);
    }

    public ProgramArchitecture getProgramArchitecture() {
        return this.programArch;
    }

    public int size() {
        return this.size;
    }

    private void checkVarnodes() throws InvalidInputException {
        int i;
        if (this.varnodes.length == 0) {
            throw new IllegalArgumentException("A minimum of one varnode must be specified");
        }
        AddressFactory addrFactory = this.programArch.getAddressFactory();
        this.size = 0;
        for (i = 0; i < this.varnodes.length; ++i) {
            Varnode varnode = this.varnodes[i];
            if (varnode == null) {
                throw new InvalidInputException("Null varnode not permitted");
            }
            if (varnode.getSize() <= 0) {
                throw new InvalidInputException("Unsupported varnode size: " + varnode.getSize());
            }
            boolean isRegister = false;
            Address storageAddr = varnode.getAddress();
            if (storageAddr.isHashAddress() || storageAddr.isUniqueAddress() || storageAddr.isConstantAddress()) {
                if (this.varnodes.length != 1) {
                    throw new InvalidInputException("Hash, Unique and Constant storage may only use a single varnode");
                }
            } else {
                AddressSpace varnodeSpace;
                AddressSpace space = addrFactory.getAddressSpace(varnode.getSpace());
                if (space != (varnodeSpace = varnode.getAddress().getAddressSpace())) {
                    throw new InvalidInputException("Invalid varnode address for specified program: " + varnode.getAddress().toString(true));
                }
            }
            if (!storageAddr.isStackAddress()) {
                Register reg = this.programArch.getLanguage().getRegister(storageAddr, varnode.getSize());
                if (reg != null && !(reg instanceof UnknownRegister)) {
                    isRegister = true;
                    if (this.registers == null) {
                        this.registers = new ArrayList<Register>();
                    }
                    this.registers.add(reg);
                }
            } else {
                long stackOffset = storageAddr.getOffset();
                if (stackOffset < 0L && -stackOffset < (long)varnode.getSize()) {
                    throw new InvalidInputException("Stack varnode violates stack frame constraints (stack offset=" + stackOffset + ", size=" + varnode.getSize());
                }
            }
            if (i < this.varnodes.length - 1 && !isRegister) {
                throw new InvalidInputException("Compound storage must use registers except for last varnode");
            }
            this.size += varnode.getSize();
        }
        for (i = 0; i < this.varnodes.length; ++i) {
            for (int j = i + 1; j < this.varnodes.length; ++j) {
                if (!this.varnodes[i].intersects(this.varnodes[j])) continue;
                throw new InvalidInputException("One or more conflicting varnodes");
            }
        }
    }

    public VariableStorage clone(ProgramArchitecture newProgramArch) throws InvalidInputException {
        if (this.programArch == null || newProgramArch == this.programArch) {
            if (this.getClass().equals(VariableStorage.class)) {
                return this;
            }
            if (this.isVoidStorage()) {
                return VOID_STORAGE;
            }
            if (this.isUnassignedStorage()) {
                return UNASSIGNED_STORAGE;
            }
            if (this.isBadStorage()) {
                return BAD_STORAGE;
            }
            return new VariableStorage(newProgramArch, this.varnodes);
        }
        if (!newProgramArch.getLanguage().equals(this.programArch.getLanguage())) {
            throw new IllegalArgumentException("Variable storage incompatible with program language: " + newProgramArch.getLanguage().toString());
        }
        AddressFactory newAddressFactory = newProgramArch.getAddressFactory();
        Varnode[] v = this.getVarnodes();
        Varnode[] newVarnodes = new Varnode[v.length];
        for (int i = 0; i < v.length; ++i) {
            AddressSpace newSpace = newAddressFactory.getAddressSpace(v[i].getSpace());
            AddressSpace curSpace = v[i].getAddress().getAddressSpace();
            if (newSpace == null) {
                throw new InvalidInputException("Variable storage incompatible with program, address space not found: " + curSpace.getName());
            }
            newVarnodes[i] = new Varnode(newSpace.getAddress(v[i].getOffset()), v[i].getSize());
        }
        return new VariableStorage(newProgramArch, newVarnodes);
    }

    public String toString() {
        if (this.isBadStorage()) {
            return BAD;
        }
        if (this.isUnassignedStorage()) {
            return UNASSIGNED;
        }
        if (this.isVoidStorage()) {
            return VOID;
        }
        StringBuilder builder = new StringBuilder();
        Varnode varnode = this.varnodes[0];
        this.addVarnodeInfo(builder, varnode);
        for (int i = 1; i < this.varnodes.length; ++i) {
            builder.append(",");
            this.addVarnodeInfo(builder, this.varnodes[i]);
        }
        return builder.toString();
    }

    private void addVarnodeInfo(StringBuilder builder, Varnode varnode) {
        Address address = varnode.getAddress();
        builder.append(this.getAddressString(address, varnode.getSize()));
        builder.append(":");
        builder.append(varnode.getSize());
    }

    private String getAddressString(Address address, int sz) {
        Register register;
        if ((address.isRegisterAddress() || address.isMemoryAddress()) && (register = this.programArch.getLanguage().getRegister(address, sz)) != null) {
            return register.toString();
        }
        return address.toString();
    }

    public int getVarnodeCount() {
        if (this.varnodes == null) {
            return 0;
        }
        return this.varnodes.length;
    }

    public Varnode[] getVarnodes() {
        if (this.varnodes == null) {
            return new Varnode[0];
        }
        return (Varnode[])this.varnodes.clone();
    }

    public boolean isAutoStorage() {
        return false;
    }

    public AutoParameterType getAutoParameterType() {
        return null;
    }

    public boolean isForcedIndirect() {
        return false;
    }

    public boolean isBadStorage() {
        return this == BAD_STORAGE;
    }

    public boolean isUnassignedStorage() {
        return this == UNASSIGNED_STORAGE;
    }

    public boolean isValid() {
        return !this.isUnassignedStorage() && !this.isBadStorage();
    }

    public boolean isVoidStorage() {
        return this == VOID_STORAGE;
    }

    public Varnode getFirstVarnode() {
        return this.varnodes == null || this.varnodes.length == 0 ? null : this.varnodes[0];
    }

    public Varnode getLastVarnode() {
        return this.varnodes == null || this.varnodes.length == 0 ? null : this.varnodes[this.varnodes.length - 1];
    }

    public boolean isStackStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.getFirstVarnode().getAddress();
        return storageAddr.isStackAddress();
    }

    public boolean hasStackStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.getLastVarnode().getAddress();
        return storageAddr.isStackAddress();
    }

    public boolean isRegisterStorage() {
        return this.varnodes != null && this.varnodes.length == 1 && this.registers != null;
    }

    public Register getRegister() {
        return this.registers != null ? this.registers.get(0) : null;
    }

    public List<Register> getRegisters() {
        return this.registers;
    }

    public int getStackOffset() {
        Address storageAddr;
        if (this.varnodes != null && this.varnodes.length != 0 && (storageAddr = this.getLastVarnode().getAddress()).isStackAddress()) {
            return (int)storageAddr.getOffset();
        }
        throw new UnsupportedOperationException("Storage does not have a stack varnode");
    }

    public Address getMinAddress() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return null;
        }
        return this.varnodes[0].getAddress();
    }

    public boolean isMemoryStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.varnodes[0].getAddress();
        return storageAddr.isMemoryAddress() && this.registers == null;
    }

    public boolean isConstantStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.varnodes[0].getAddress();
        return storageAddr.isConstantAddress();
    }

    public boolean isHashStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.varnodes[0].getAddress();
        return storageAddr.isHashAddress();
    }

    public boolean isUniqueStorage() {
        if (this.varnodes == null || this.varnodes.length == 0) {
            return false;
        }
        Address storageAddr = this.varnodes[0].getAddress();
        return storageAddr.isUniqueAddress();
    }

    public boolean isCompoundStorage() {
        return this.varnodes != null && this.varnodes.length > 1;
    }

    public long getLongHash() {
        if (this.hashcode == 0L) {
            CRC64 crc = new CRC64();
            byte[] bytes = this.getSerializationString().getBytes();
            crc.update(bytes, 0, bytes.length);
            this.hashcode = crc.finish();
        }
        return this.hashcode;
    }

    public int hashCode() {
        return (int)this.getLongHash();
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof VariableStorage)) {
            return false;
        }
        VariableStorage otherVS = (VariableStorage)obj;
        if (this.isAutoStorage() != otherVS.isAutoStorage()) {
            return false;
        }
        if (this.isForcedIndirect() != otherVS.isForcedIndirect()) {
            return false;
        }
        if (this.isBadStorage() != otherVS.isBadStorage()) {
            return false;
        }
        if (this.isUnassignedStorage() != otherVS.isUnassignedStorage()) {
            return false;
        }
        if (this.isVoidStorage() != otherVS.isVoidStorage()) {
            return false;
        }
        return this.compareTo(otherVS) == 0;
    }

    public boolean intersects(VariableStorage variableStorage) {
        Varnode[] otherVarnodes = variableStorage.varnodes;
        if (this.varnodes == null || otherVarnodes == null) {
            return false;
        }
        for (Varnode varnode : this.varnodes) {
            for (Varnode otherVarnode : otherVarnodes) {
                if (!varnode.intersects(otherVarnode)) continue;
                return true;
            }
        }
        return false;
    }

    public boolean intersects(AddressSetView set) {
        if (this.varnodes == null || set == null || set.isEmpty()) {
            return false;
        }
        for (Varnode varnode : this.varnodes) {
            if (!varnode.intersects(set)) continue;
            return true;
        }
        return false;
    }

    public boolean intersects(Register reg) {
        if (this.varnodes == null || reg == null) {
            return false;
        }
        Varnode regVarnode = new Varnode(reg.getAddress(), reg.getMinimumByteSize());
        for (Varnode varnode : this.varnodes) {
            if (!varnode.intersects(regVarnode)) continue;
            return true;
        }
        return false;
    }

    public boolean contains(Address address) {
        if (this.varnodes == null) {
            return false;
        }
        for (Varnode varnode : this.varnodes) {
            if (!varnode.contains(address)) continue;
            return true;
        }
        return false;
    }

    private static int getPrecedence(VariableStorage storage) {
        if (storage.isUnassignedStorage()) {
            return 2;
        }
        if (storage.varnodes != null && storage.varnodes.length != 0) {
            return 1;
        }
        return 3;
    }

    @Override
    public int compareTo(VariableStorage otherStorage) {
        int otherPrecedence;
        int myPrecedence = VariableStorage.getPrecedence(this);
        int diff = myPrecedence - (otherPrecedence = VariableStorage.getPrecedence(otherStorage));
        if (diff != 0 || myPrecedence != 1) {
            return diff;
        }
        int compareIndexCnt = Math.min(this.varnodes.length, otherStorage.varnodes.length);
        for (int i = 0; i < compareIndexCnt; ++i) {
            Address otherStorageAddr;
            Address myStorageAddr = this.varnodes[i].getAddress();
            diff = myStorageAddr.compareTo(otherStorageAddr = otherStorage.varnodes[i].getAddress());
            if (diff != 0) {
                return diff;
            }
            diff = this.varnodes[i].getSize() - otherStorage.varnodes[i].getSize();
            if (diff == 0) continue;
            return diff;
        }
        return this.varnodes.length - otherStorage.varnodes.length;
    }

    public String getSerializationString() {
        if (this.serialization != null) {
            return this.serialization;
        }
        this.serialization = this.isBadStorage() ? BAD : (this.isUnassignedStorage() ? UNASSIGNED : (this.isVoidStorage() ? VOID : VariableStorage.getSerializationString(this.varnodes)));
        return this.serialization;
    }

    public static String getSerializationString(Varnode ... varnodes) {
        if (varnodes == null || varnodes.length == 0) {
            throw new IllegalArgumentException("varnodes may not be null or empty");
        }
        StringBuilder strBuilder = new StringBuilder();
        for (Varnode v : varnodes) {
            if (strBuilder.length() != 0) {
                strBuilder.append(",");
            }
            strBuilder.append(v.getAddress().toString(true));
            strBuilder.append(":");
            strBuilder.append(Integer.toString(v.getSize()));
        }
        return strBuilder.toString();
    }

    public static List<Varnode> getVarnodes(AddressFactory addrFactory, String serialization) throws InvalidInputException {
        if (BAD.equals(serialization)) {
            return null;
        }
        ArrayList<Varnode> list = new ArrayList<Varnode>();
        String[] varnodeStrings = serialization.split(",");
        try {
            for (String piece : varnodeStrings) {
                int index = piece.lastIndexOf(58);
                if (index <= 0) {
                    list = null;
                    break;
                }
                String addrStr = piece.substring(0, index);
                String sizeStr = piece.substring(index + 1);
                Address addr = addrFactory.getAddress(addrStr);
                if (addr == null) {
                    list = null;
                    break;
                }
                if (addr == Address.NO_ADDRESS) {
                    return null;
                }
                int size = Integer.parseInt(sizeStr);
                list.add(new Varnode(addr, size));
            }
        }
        catch (NumberFormatException e) {
            list = null;
        }
        if (list == null) {
            throw new InvalidInputException("Invalid varnode serialization: '" + serialization + "'");
        }
        return list;
    }

    public static String translateSerialization(LanguageTranslator translator, String serialization) throws InvalidInputException {
        String[] varnodeStrings;
        if (serialization == null || UNASSIGNED.equals(serialization)) {
            return null;
        }
        if (VOID.equals(serialization)) {
            return VOID;
        }
        if (BAD.equals(serialization)) {
            return BAD;
        }
        StringBuilder strBuilder = new StringBuilder();
        for (String piece : varnodeStrings = serialization.split(",")) {
            int index = piece.lastIndexOf(58);
            if (index <= 0) {
                strBuilder = null;
                break;
            }
            if (strBuilder.length() != 0) {
                strBuilder.append(",");
            }
            String addrStr = piece.substring(0, index);
            String sizeStr = piece.substring(index + 1);
            index = addrStr.indexOf(58);
            if (index > 0) {
                String spaceName = addrStr.substring(0, index);
                String offsetStr = addrStr.substring(index + 1);
                AddressSpace space = translator.getNewAddressSpace(spaceName);
                if (space != null) {
                    if (space.isRegisterSpace()) {
                        long offset = Long.parseUnsignedLong(offsetStr, 16);
                        int size = Integer.parseInt(sizeStr);
                        Address oldRegAddr = translator.getOldLanguage().getAddressFactory().getRegisterSpace().getAddress(offset);
                        String newOffsetStr = VariableStorage.translateRegisterVarnodeOffset(oldRegAddr, size, translator);
                        if (newOffsetStr != null) {
                            offsetStr = newOffsetStr;
                        }
                    }
                    strBuilder.append(space.getName());
                    strBuilder.append(':');
                    strBuilder.append(offsetStr);
                    strBuilder.append(':');
                    strBuilder.append(sizeStr);
                    continue;
                }
            }
            strBuilder.append(piece);
        }
        if (strBuilder == null) {
            throw new InvalidInputException("Invalid varnode serialization: '" + serialization + "'");
        }
        return strBuilder.toString();
    }

    private static String translateRegisterVarnodeOffset(Address oldRegAddr, int varnodeSize, LanguageTranslator translator) {
        Register newReg;
        long offset = oldRegAddr.getOffset();
        Register oldReg = translator.getOldRegister(oldRegAddr, varnodeSize);
        if (oldReg == null) {
            oldReg = translator.getOldRegisterContaining(oldRegAddr);
        }
        if (oldReg != null && !(oldReg instanceof UnknownRegister) && (newReg = translator.getNewRegister(oldReg)) != null) {
            int regSizeDiff;
            int origByteShift = (int)offset - oldReg.getOffset();
            offset = newReg.getOffset() + origByteShift;
            if (newReg.isBigEndian() ? (offset += (long)(regSizeDiff = newReg.getMinimumByteSize() - oldReg.getMinimumByteSize())) < (long)newReg.getOffset() : origByteShift + varnodeSize > newReg.getMinimumByteSize()) {
                return null;
            }
            return Long.toHexString(offset);
        }
        return null;
    }
}

