﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/RequestCharged.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace S3Crt {
namespace Model {
class DeleteObjectResult {
 public:
  AWS_S3CRT_API DeleteObjectResult() = default;
  AWS_S3CRT_API DeleteObjectResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_S3CRT_API DeleteObjectResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>Indicates whether the specified object version that was permanently deleted
   * was (true) or was not (false) a delete marker before deletion. In a simple
   * DELETE, this header indicates whether (true) or not (false) the current version
   * of the object is a delete marker. To learn more about delete markers, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/DeleteMarker.html">Working
   * with delete markers</a>.</p>  <p>This functionality is not supported for
   * directory buckets.</p>
   */
  inline bool GetDeleteMarker() const { return m_deleteMarker; }
  inline void SetDeleteMarker(bool value) {
    m_deleteMarkerHasBeenSet = true;
    m_deleteMarker = value;
  }
  inline DeleteObjectResult& WithDeleteMarker(bool value) {
    SetDeleteMarker(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the version ID of the delete marker created as a result of the DELETE
   * operation.</p>  <p>This functionality is not supported for directory
   * buckets.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  DeleteObjectResult& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline RequestCharged GetRequestCharged() const { return m_requestCharged; }
  inline void SetRequestCharged(RequestCharged value) {
    m_requestChargedHasBeenSet = true;
    m_requestCharged = value;
  }
  inline DeleteObjectResult& WithRequestCharged(RequestCharged value) {
    SetRequestCharged(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  DeleteObjectResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_deleteMarker{false};
  bool m_deleteMarkerHasBeenSet = false;

  Aws::String m_versionId;
  bool m_versionIdHasBeenSet = false;

  RequestCharged m_requestCharged{RequestCharged::NOT_SET};
  bool m_requestChargedHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
