﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workdocs/WorkDocsRequest.h>
#include <aws/workdocs/WorkDocs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace WorkDocs {
namespace Model {

/**
 */
class DescribeActivitiesRequest : public WorkDocsRequest {
 public:
  AWS_WORKDOCS_API DescribeActivitiesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeActivities"; }

  AWS_WORKDOCS_API Aws::String SerializePayload() const override;

  AWS_WORKDOCS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_WORKDOCS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Amazon WorkDocs authentication token. Not required when using Amazon Web
   * Services administrator credentials to access the API.</p>
   */
  inline const Aws::String& GetAuthenticationToken() const { return m_authenticationToken; }
  inline bool AuthenticationTokenHasBeenSet() const { return m_authenticationTokenHasBeenSet; }
  template <typename AuthenticationTokenT = Aws::String>
  void SetAuthenticationToken(AuthenticationTokenT&& value) {
    m_authenticationTokenHasBeenSet = true;
    m_authenticationToken = std::forward<AuthenticationTokenT>(value);
  }
  template <typename AuthenticationTokenT = Aws::String>
  DescribeActivitiesRequest& WithAuthenticationToken(AuthenticationTokenT&& value) {
    SetAuthenticationToken(std::forward<AuthenticationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that determines the starting time of the activities. The
   * response includes the activities performed after the specified timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  DescribeActivitiesRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that determines the end time of the activities. The response
   * includes the activities performed before the specified timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  DescribeActivitiesRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the organization. This is a mandatory parameter when using
   * administrative API (SigV4) requests.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  DescribeActivitiesRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies which activity types to include in the response. If this field is
   * left empty, all activity types are returned.</p>
   */
  inline const Aws::String& GetActivityTypes() const { return m_activityTypes; }
  inline bool ActivityTypesHasBeenSet() const { return m_activityTypesHasBeenSet; }
  template <typename ActivityTypesT = Aws::String>
  void SetActivityTypes(ActivityTypesT&& value) {
    m_activityTypesHasBeenSet = true;
    m_activityTypes = std::forward<ActivityTypesT>(value);
  }
  template <typename ActivityTypesT = Aws::String>
  DescribeActivitiesRequest& WithActivityTypes(ActivityTypesT&& value) {
    SetActivityTypes(std::forward<ActivityTypesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document or folder ID for which to describe activity types.</p>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  DescribeActivitiesRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the user who performed the action. The response includes activities
   * pertaining to this user. This is an optional parameter and is only applicable
   * for administrative API (SigV4) requests.</p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  DescribeActivitiesRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Includes indirect activities. An indirect activity results from a direct
   * activity performed on a parent resource. For example, sharing a parent folder
   * (the direct activity) shares all of the subfolders and documents within the
   * parent folder (the indirect activity).</p>
   */
  inline bool GetIncludeIndirectActivities() const { return m_includeIndirectActivities; }
  inline bool IncludeIndirectActivitiesHasBeenSet() const { return m_includeIndirectActivitiesHasBeenSet; }
  inline void SetIncludeIndirectActivities(bool value) {
    m_includeIndirectActivitiesHasBeenSet = true;
    m_includeIndirectActivities = value;
  }
  inline DescribeActivitiesRequest& WithIncludeIndirectActivities(bool value) {
    SetIncludeIndirectActivities(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline DescribeActivitiesRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeActivitiesRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_authenticationToken;
  bool m_authenticationTokenHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::String m_organizationId;
  bool m_organizationIdHasBeenSet = false;

  Aws::String m_activityTypes;
  bool m_activityTypesHasBeenSet = false;

  Aws::String m_resourceId;
  bool m_resourceIdHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  bool m_includeIndirectActivities{false};
  bool m_includeIndirectActivitiesHasBeenSet = false;

  int m_limit{0};
  bool m_limitHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkDocs
}  // namespace Aws
