﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/states/SFNRequest.h>
#include <aws/states/SFN_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SFN {
namespace Model {

/**
 */
class SendTaskFailureRequest : public SFNRequest {
 public:
  AWS_SFN_API SendTaskFailureRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SendTaskFailure"; }

  AWS_SFN_API Aws::String SerializePayload() const override;

  AWS_SFN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The token that represents this task. Task tokens are generated by Step
   * Functions when tasks are assigned to a worker, or in the <a
   * href="https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html">context
   * object</a> when a workflow enters a task state. See
   * <a>GetActivityTaskOutput$taskToken</a>.</p>
   */
  inline const Aws::String& GetTaskToken() const { return m_taskToken; }
  inline bool TaskTokenHasBeenSet() const { return m_taskTokenHasBeenSet; }
  template <typename TaskTokenT = Aws::String>
  void SetTaskToken(TaskTokenT&& value) {
    m_taskTokenHasBeenSet = true;
    m_taskToken = std::forward<TaskTokenT>(value);
  }
  template <typename TaskTokenT = Aws::String>
  SendTaskFailureRequest& WithTaskToken(TaskTokenT&& value) {
    SetTaskToken(std::forward<TaskTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The error code of the failure.</p>
   */
  inline const Aws::String& GetError() const { return m_error; }
  inline bool ErrorHasBeenSet() const { return m_errorHasBeenSet; }
  template <typename ErrorT = Aws::String>
  void SetError(ErrorT&& value) {
    m_errorHasBeenSet = true;
    m_error = std::forward<ErrorT>(value);
  }
  template <typename ErrorT = Aws::String>
  SendTaskFailureRequest& WithError(ErrorT&& value) {
    SetError(std::forward<ErrorT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A more detailed explanation of the cause of the failure.</p>
   */
  inline const Aws::String& GetCause() const { return m_cause; }
  inline bool CauseHasBeenSet() const { return m_causeHasBeenSet; }
  template <typename CauseT = Aws::String>
  void SetCause(CauseT&& value) {
    m_causeHasBeenSet = true;
    m_cause = std::forward<CauseT>(value);
  }
  template <typename CauseT = Aws::String>
  SendTaskFailureRequest& WithCause(CauseT&& value) {
    SetCause(std::forward<CauseT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_taskToken;
  bool m_taskTokenHasBeenSet = false;

  Aws::String m_error;
  bool m_errorHasBeenSet = false;

  Aws::String m_cause;
  bool m_causeHasBeenSet = false;
};

}  // namespace Model
}  // namespace SFN
}  // namespace Aws
