﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/security-ir/SecurityIRRequest.h>
#include <aws/security-ir/SecurityIR_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SecurityIR {
namespace Model {

/**
 */
class BatchGetMemberAccountDetailsRequest : public SecurityIRRequest {
 public:
  AWS_SECURITYIR_API BatchGetMemberAccountDetailsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchGetMemberAccountDetails"; }

  AWS_SECURITYIR_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Required element used in combination with BatchGetMemberAccountDetails to
   * identify the membership ID to query. </p>
   */
  inline const Aws::String& GetMembershipId() const { return m_membershipId; }
  inline bool MembershipIdHasBeenSet() const { return m_membershipIdHasBeenSet; }
  template <typename MembershipIdT = Aws::String>
  void SetMembershipId(MembershipIdT&& value) {
    m_membershipIdHasBeenSet = true;
    m_membershipId = std::forward<MembershipIdT>(value);
  }
  template <typename MembershipIdT = Aws::String>
  BatchGetMemberAccountDetailsRequest& WithMembershipId(MembershipIdT&& value) {
    SetMembershipId(std::forward<MembershipIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional element to query the membership relationship status to a provided
   * list of account IDs.</p>  <p> AWS account ID's may appear less than 12
   * characters and need to be zero-prepended. An example would be
   * <code>123123123</code> which is nine digits, and with zero-prepend would be
   * <code>000123123123</code>. Not zero-prepending to 12 digits could result in
   * errors. </p>
   */
  inline const Aws::Vector<Aws::String>& GetAccountIds() const { return m_accountIds; }
  inline bool AccountIdsHasBeenSet() const { return m_accountIdsHasBeenSet; }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  void SetAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds = std::forward<AccountIdsT>(value);
  }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  BatchGetMemberAccountDetailsRequest& WithAccountIds(AccountIdsT&& value) {
    SetAccountIds(std::forward<AccountIdsT>(value));
    return *this;
  }
  template <typename AccountIdsT = Aws::String>
  BatchGetMemberAccountDetailsRequest& AddAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds.emplace_back(std::forward<AccountIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_membershipId;
  bool m_membershipIdHasBeenSet = false;

  Aws::Vector<Aws::String> m_accountIds;
  bool m_accountIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityIR
}  // namespace Aws
