﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/ModelApprovalStatus.h>
#include <aws/sagemaker/model/ModelPackageSortBy.h>
#include <aws/sagemaker/model/ModelPackageType.h>
#include <aws/sagemaker/model/SortOrder.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListModelPackagesRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListModelPackagesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListModelPackages"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A filter that returns only model packages created after the specified time
   * (timestamp).</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeAfter() const { return m_creationTimeAfter; }
  inline bool CreationTimeAfterHasBeenSet() const { return m_creationTimeAfterHasBeenSet; }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  void SetCreationTimeAfter(CreationTimeAfterT&& value) {
    m_creationTimeAfterHasBeenSet = true;
    m_creationTimeAfter = std::forward<CreationTimeAfterT>(value);
  }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  ListModelPackagesRequest& WithCreationTimeAfter(CreationTimeAfterT&& value) {
    SetCreationTimeAfter(std::forward<CreationTimeAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only model packages created before the specified time
   * (timestamp).</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeBefore() const { return m_creationTimeBefore; }
  inline bool CreationTimeBeforeHasBeenSet() const { return m_creationTimeBeforeHasBeenSet; }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  void SetCreationTimeBefore(CreationTimeBeforeT&& value) {
    m_creationTimeBeforeHasBeenSet = true;
    m_creationTimeBefore = std::forward<CreationTimeBeforeT>(value);
  }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  ListModelPackagesRequest& WithCreationTimeBefore(CreationTimeBeforeT&& value) {
    SetCreationTimeBefore(std::forward<CreationTimeBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of model packages to return in the response.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListModelPackagesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string in the model package name. This filter returns only model packages
   * whose name contains the specified string.</p>
   */
  inline const Aws::String& GetNameContains() const { return m_nameContains; }
  inline bool NameContainsHasBeenSet() const { return m_nameContainsHasBeenSet; }
  template <typename NameContainsT = Aws::String>
  void SetNameContains(NameContainsT&& value) {
    m_nameContainsHasBeenSet = true;
    m_nameContains = std::forward<NameContainsT>(value);
  }
  template <typename NameContainsT = Aws::String>
  ListModelPackagesRequest& WithNameContains(NameContainsT&& value) {
    SetNameContains(std::forward<NameContainsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only the model packages with the specified approval
   * status.</p>
   */
  inline ModelApprovalStatus GetModelApprovalStatus() const { return m_modelApprovalStatus; }
  inline bool ModelApprovalStatusHasBeenSet() const { return m_modelApprovalStatusHasBeenSet; }
  inline void SetModelApprovalStatus(ModelApprovalStatus value) {
    m_modelApprovalStatusHasBeenSet = true;
    m_modelApprovalStatus = value;
  }
  inline ListModelPackagesRequest& WithModelApprovalStatus(ModelApprovalStatus value) {
    SetModelApprovalStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only model versions that belong to the specified model
   * group.</p>
   */
  inline const Aws::String& GetModelPackageGroupName() const { return m_modelPackageGroupName; }
  inline bool ModelPackageGroupNameHasBeenSet() const { return m_modelPackageGroupNameHasBeenSet; }
  template <typename ModelPackageGroupNameT = Aws::String>
  void SetModelPackageGroupName(ModelPackageGroupNameT&& value) {
    m_modelPackageGroupNameHasBeenSet = true;
    m_modelPackageGroupName = std::forward<ModelPackageGroupNameT>(value);
  }
  template <typename ModelPackageGroupNameT = Aws::String>
  ListModelPackagesRequest& WithModelPackageGroupName(ModelPackageGroupNameT&& value) {
    SetModelPackageGroupName(std::forward<ModelPackageGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only the model packages of the specified type. This can
   * be one of the following values.</p> <ul> <li> <p> <code>UNVERSIONED</code> -
   * List only unversioined models. This is the default value if no
   * <code>ModelPackageType</code> is specified.</p> </li> <li> <p>
   * <code>VERSIONED</code> - List only versioned models.</p> </li> <li> <p>
   * <code>BOTH</code> - List both versioned and unversioned models.</p> </li> </ul>
   */
  inline ModelPackageType GetModelPackageType() const { return m_modelPackageType; }
  inline bool ModelPackageTypeHasBeenSet() const { return m_modelPackageTypeHasBeenSet; }
  inline void SetModelPackageType(ModelPackageType value) {
    m_modelPackageTypeHasBeenSet = true;
    m_modelPackageType = value;
  }
  inline ListModelPackagesRequest& WithModelPackageType(ModelPackageType value) {
    SetModelPackageType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the response to a previous <code>ListModelPackages</code> request was
   * truncated, the response includes a <code>NextToken</code>. To retrieve the next
   * set of model packages, use the token in the next request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListModelPackagesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter by which to sort the results. The default is
   * <code>CreationTime</code>.</p>
   */
  inline ModelPackageSortBy GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(ModelPackageSortBy value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListModelPackagesRequest& WithSortBy(ModelPackageSortBy value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort order for the results. The default is <code>Ascending</code>.</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListModelPackagesRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_creationTimeAfter{};
  bool m_creationTimeAfterHasBeenSet = false;

  Aws::Utils::DateTime m_creationTimeBefore{};
  bool m_creationTimeBeforeHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nameContains;
  bool m_nameContainsHasBeenSet = false;

  ModelApprovalStatus m_modelApprovalStatus{ModelApprovalStatus::NOT_SET};
  bool m_modelApprovalStatusHasBeenSet = false;

  Aws::String m_modelPackageGroupName;
  bool m_modelPackageGroupNameHasBeenSet = false;

  ModelPackageType m_modelPackageType{ModelPackageType::NOT_SET};
  bool m_modelPackageTypeHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  ModelPackageSortBy m_sortBy{ModelPackageSortBy::NOT_SET};
  bool m_sortByHasBeenSet = false;

  SortOrder m_sortOrder{SortOrder::NOT_SET};
  bool m_sortOrderHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
