﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/MetadataProperties.h>
#include <aws/sagemaker/model/Tag.h>
#include <aws/sagemaker/model/TrialComponentArtifact.h>
#include <aws/sagemaker/model/TrialComponentParameterValue.h>
#include <aws/sagemaker/model/TrialComponentStatus.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateTrialComponentRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateTrialComponentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTrialComponent"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the component. The name must be unique in your Amazon Web
   * Services account and is not case-sensitive.</p>
   */
  inline const Aws::String& GetTrialComponentName() const { return m_trialComponentName; }
  inline bool TrialComponentNameHasBeenSet() const { return m_trialComponentNameHasBeenSet; }
  template <typename TrialComponentNameT = Aws::String>
  void SetTrialComponentName(TrialComponentNameT&& value) {
    m_trialComponentNameHasBeenSet = true;
    m_trialComponentName = std::forward<TrialComponentNameT>(value);
  }
  template <typename TrialComponentNameT = Aws::String>
  CreateTrialComponentRequest& WithTrialComponentName(TrialComponentNameT&& value) {
    SetTrialComponentName(std::forward<TrialComponentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the component as displayed. The name doesn't need to be unique.
   * If <code>DisplayName</code> isn't specified, <code>TrialComponentName</code> is
   * displayed.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreateTrialComponentRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the component. States include:</p> <ul> <li> <p>InProgress</p>
   * </li> <li> <p>Completed</p> </li> <li> <p>Failed</p> </li> </ul>
   */
  inline const TrialComponentStatus& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = TrialComponentStatus>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = TrialComponentStatus>
  CreateTrialComponentRequest& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component started.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  CreateTrialComponentRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component ended.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  CreateTrialComponentRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The hyperparameters for the component.</p>
   */
  inline const Aws::Map<Aws::String, TrialComponentParameterValue>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Map<Aws::String, TrialComponentParameterValue>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Map<Aws::String, TrialComponentParameterValue>>
  CreateTrialComponentRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersKeyT = Aws::String, typename ParametersValueT = TrialComponentParameterValue>
  CreateTrialComponentRequest& AddParameters(ParametersKeyT&& key, ParametersValueT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace(std::forward<ParametersKeyT>(key), std::forward<ParametersValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The input artifacts for the component. Examples of input artifacts are
   * datasets, algorithms, hyperparameters, source code, and instance types.</p>
   */
  inline const Aws::Map<Aws::String, TrialComponentArtifact>& GetInputArtifacts() const { return m_inputArtifacts; }
  inline bool InputArtifactsHasBeenSet() const { return m_inputArtifactsHasBeenSet; }
  template <typename InputArtifactsT = Aws::Map<Aws::String, TrialComponentArtifact>>
  void SetInputArtifacts(InputArtifactsT&& value) {
    m_inputArtifactsHasBeenSet = true;
    m_inputArtifacts = std::forward<InputArtifactsT>(value);
  }
  template <typename InputArtifactsT = Aws::Map<Aws::String, TrialComponentArtifact>>
  CreateTrialComponentRequest& WithInputArtifacts(InputArtifactsT&& value) {
    SetInputArtifacts(std::forward<InputArtifactsT>(value));
    return *this;
  }
  template <typename InputArtifactsKeyT = Aws::String, typename InputArtifactsValueT = TrialComponentArtifact>
  CreateTrialComponentRequest& AddInputArtifacts(InputArtifactsKeyT&& key, InputArtifactsValueT&& value) {
    m_inputArtifactsHasBeenSet = true;
    m_inputArtifacts.emplace(std::forward<InputArtifactsKeyT>(key), std::forward<InputArtifactsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The output artifacts for the component. Examples of output artifacts are
   * metrics, snapshots, logs, and images.</p>
   */
  inline const Aws::Map<Aws::String, TrialComponentArtifact>& GetOutputArtifacts() const { return m_outputArtifacts; }
  inline bool OutputArtifactsHasBeenSet() const { return m_outputArtifactsHasBeenSet; }
  template <typename OutputArtifactsT = Aws::Map<Aws::String, TrialComponentArtifact>>
  void SetOutputArtifacts(OutputArtifactsT&& value) {
    m_outputArtifactsHasBeenSet = true;
    m_outputArtifacts = std::forward<OutputArtifactsT>(value);
  }
  template <typename OutputArtifactsT = Aws::Map<Aws::String, TrialComponentArtifact>>
  CreateTrialComponentRequest& WithOutputArtifacts(OutputArtifactsT&& value) {
    SetOutputArtifacts(std::forward<OutputArtifactsT>(value));
    return *this;
  }
  template <typename OutputArtifactsKeyT = Aws::String, typename OutputArtifactsValueT = TrialComponentArtifact>
  CreateTrialComponentRequest& AddOutputArtifacts(OutputArtifactsKeyT&& key, OutputArtifactsValueT&& value) {
    m_outputArtifactsHasBeenSet = true;
    m_outputArtifacts.emplace(std::forward<OutputArtifactsKeyT>(key), std::forward<OutputArtifactsValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const MetadataProperties& GetMetadataProperties() const { return m_metadataProperties; }
  inline bool MetadataPropertiesHasBeenSet() const { return m_metadataPropertiesHasBeenSet; }
  template <typename MetadataPropertiesT = MetadataProperties>
  void SetMetadataProperties(MetadataPropertiesT&& value) {
    m_metadataPropertiesHasBeenSet = true;
    m_metadataProperties = std::forward<MetadataPropertiesT>(value);
  }
  template <typename MetadataPropertiesT = MetadataProperties>
  CreateTrialComponentRequest& WithMetadataProperties(MetadataPropertiesT&& value) {
    SetMetadataProperties(std::forward<MetadataPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to associate with the component. You can use <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Search.html">Search</a>
   * API to search on the tags.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateTrialComponentRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateTrialComponentRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trialComponentName;
  bool m_trialComponentNameHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  TrialComponentStatus m_status;
  bool m_statusHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::Map<Aws::String, TrialComponentParameterValue> m_parameters;
  bool m_parametersHasBeenSet = false;

  Aws::Map<Aws::String, TrialComponentArtifact> m_inputArtifacts;
  bool m_inputArtifactsHasBeenSet = false;

  Aws::Map<Aws::String, TrialComponentArtifact> m_outputArtifacts;
  bool m_outputArtifactsHasBeenSet = false;

  MetadataProperties m_metadataProperties;
  bool m_metadataPropertiesHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
