﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/route53-recovery-cluster/Route53RecoveryClusterRequest.h>
#include <aws/route53-recovery-cluster/Route53RecoveryCluster_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Route53RecoveryCluster {
namespace Model {

/**
 */
class ListRoutingControlsRequest : public Route53RecoveryClusterRequest {
 public:
  AWS_ROUTE53RECOVERYCLUSTER_API ListRoutingControlsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListRoutingControls"; }

  AWS_ROUTE53RECOVERYCLUSTER_API Aws::String SerializePayload() const override;

  AWS_ROUTE53RECOVERYCLUSTER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the control panel of the routing controls
   * to list.</p>
   */
  inline const Aws::String& GetControlPanelArn() const { return m_controlPanelArn; }
  inline bool ControlPanelArnHasBeenSet() const { return m_controlPanelArnHasBeenSet; }
  template <typename ControlPanelArnT = Aws::String>
  void SetControlPanelArn(ControlPanelArnT&& value) {
    m_controlPanelArnHasBeenSet = true;
    m_controlPanelArn = std::forward<ControlPanelArnT>(value);
  }
  template <typename ControlPanelArnT = Aws::String>
  ListRoutingControlsRequest& WithControlPanelArn(ControlPanelArnT&& value) {
    SetControlPanelArn(std::forward<ControlPanelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of results. You receive this token from a previous
   * call.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListRoutingControlsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of routing controls objects that you want to return with this
   * call. The default value is 500.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListRoutingControlsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_controlPanelArn;
  bool m_controlPanelArnHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Route53RecoveryCluster
}  // namespace Aws
