﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resourcegroupstaggingapi/ResourceGroupsTaggingAPIRequest.h>
#include <aws/resourcegroupstaggingapi/ResourceGroupsTaggingAPI_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ResourceGroupsTaggingAPI {
namespace Model {

/**
 */
class TagResourcesRequest : public ResourceGroupsTaggingAPIRequest {
 public:
  AWS_RESOURCEGROUPSTAGGINGAPI_API TagResourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "TagResources"; }

  AWS_RESOURCEGROUPSTAGGINGAPI_API Aws::String SerializePayload() const override;

  AWS_RESOURCEGROUPSTAGGINGAPI_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the list of ARNs of the resources that you want to apply tags
   * to.</p> <p>An ARN (Amazon Resource Name) uniquely identifies a resource. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a> in the
   * <i>Amazon Web Services General Reference</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceARNList() const { return m_resourceARNList; }
  inline bool ResourceARNListHasBeenSet() const { return m_resourceARNListHasBeenSet; }
  template <typename ResourceARNListT = Aws::Vector<Aws::String>>
  void SetResourceARNList(ResourceARNListT&& value) {
    m_resourceARNListHasBeenSet = true;
    m_resourceARNList = std::forward<ResourceARNListT>(value);
  }
  template <typename ResourceARNListT = Aws::Vector<Aws::String>>
  TagResourcesRequest& WithResourceARNList(ResourceARNListT&& value) {
    SetResourceARNList(std::forward<ResourceARNListT>(value));
    return *this;
  }
  template <typename ResourceARNListT = Aws::String>
  TagResourcesRequest& AddResourceARNList(ResourceARNListT&& value) {
    m_resourceARNListHasBeenSet = true;
    m_resourceARNList.emplace_back(std::forward<ResourceARNListT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of tags that you want to add to the specified resources. A
   * tag consists of a key and a value that you define.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  TagResourcesRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  TagResourcesRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_resourceARNList;
  bool m_resourceARNListHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResourceGroupsTaggingAPI
}  // namespace Aws
